package LinkFuture.Core.GenericRepository.Repository;

import LinkFuture.Core.GenericRepository.Entity.*;
import LinkFuture.Core.MemoryManager.StaticMemoryCache.StaticMemoryCacheHelper;
import LinkFuture.Core.OperationManager.Operation;
import LinkFuture.Core.Utility;
import LinkFuture.Core.WebClient.HttpMethod;
import LinkFuture.Init.Extensions.DateExtension;
import LinkFuture.Init.Extensions.StringExtension;
import com.fasterxml.jackson.annotation.JsonFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

/**
 * Created by Cyokin
 * on 10/15/2015.
 */
public abstract class BaseRepository<T,ID extends Serializable> implements IGenericRepository<T,ID> {
    protected static final Logger logger = LoggerFactory.getLogger(GenericRepository.class);
    public static final String JSON_QUERY = "$JsonQuery";

    protected Class<? extends T> entityType;
    protected Class<? extends ID> keyType;

    protected String requestPath;

    public BaseRepository(String requestPath){
        this.requestPath = requestPath;
        Type t = getClass().getGenericSuperclass();
        ParameterizedType pt = (ParameterizedType) t;
        entityType = (Class) pt.getActualTypeArguments()[0];
        keyType = (Class) pt.getActualTypeArguments()[1];
    }

    protected abstract <T> Object genericDBResourceLoad(String contentName,LinkFuture.Core.WebClient.HttpMethod method,T requestObject) throws Exception;
    protected abstract <T> Object genericDBResourceLoad(LinkFuture.Core.WebClient.HttpMethod method,T requestObject) throws Exception;


    protected <R> PageableList<R> toList(Object genericDBResult, Class<R> entityType) throws IOException {
        if(genericDBResult == null)
        {
            return null;
        }
        ResponseData response = Utility.jacksonFromJson(genericDBResult.toString(), ResponseData.class, entityType);
        return response==null
                || response.data==null
                || response.data.length ==0
                ?null:
                new PageableList(response.data[0], (response.page==null || response.page.length==0)?null:response.page[0]);
    }

    public synchronized static HashMap<String,FieldInfo> findEntityFieldInfo(Class<?> type) throws Exception {
        String classKey = "$BaseRepository$findEntityFieldInfo".concat(type.getName());
        return StaticMemoryCacheHelper.AddNeverExpiredMemoryCache(classKey, new Operation<HashMap<String,FieldInfo>>(type) {
            @Override
            public HashMap<String,FieldInfo> call() throws Exception {
                HashMap<String, FieldInfo> fieldList = new HashMap<>();
                for (Field field : type.getFields()) {
                    //ignore static field
                    if(java.lang.reflect.Modifier.isStatic(field.getModifiers()))
                    {
                        continue;
                    }
                    FieldAttribute fieldAttribute = field.getAnnotation(FieldAttribute.class);
                    FieldInfo fieldMeta = new FieldInfo();
                    fieldMeta.field = field;
                    fieldMeta.jsonFormatAttribute =field.getAnnotation(JsonFormat.class);
                    if (fieldAttribute != null) {
                        String fieldName = fieldAttribute.name().length() == 0 ? field.getName() : fieldAttribute.name();
                        fieldMeta.fieldAttribute = fieldAttribute;
                        fieldList.put(fieldName, fieldMeta);
                    } else {
                        fieldList.put(field.getName().toLowerCase(), fieldMeta);
                    }
                }
                return fieldList;
            }
        });
    }
    public synchronized static HashMap<String,FieldInfo> findEntityKeyFieldInfo(Class<?> type) throws Exception {
        String classKey = "$BaseRepository$findEntityKeyFieldInfo".concat(type.getName());
        return StaticMemoryCacheHelper.AddNeverExpiredMemoryCache(classKey, new Operation<HashMap<String,FieldInfo>>(type) {
            @Override
            public HashMap<String,FieldInfo> call() throws Exception {
                HashMap<String, FieldInfo> fieldList = new HashMap<>();
                HashMap<String,FieldInfo> fields = findEntityFieldInfo(type);
                for (String fieldName:fields.keySet())
                {
                    FieldInfo field = fields.get(fieldName);
                    if(field.fieldAttribute!=null && field.fieldAttribute.isKey())
                    {
                        fieldList.put(fieldName, field);
                    }
                }
                return fieldList;
            }
        });
    }

    @Override
    public <S extends T> ID insert(S obj) throws Exception {
        logger.trace("Insert {}", this.requestPath);
        Object output = genericDBResourceLoad(HttpMethod.Put, obj);
        if(output == null)
        {
            throw new RuntimeException("Insert failed");
        }
        //return (ID)Utility.jacksonFromJson(output.toString(), ResponseIdentity.class,entityType).identity;
        //jacksonFromJson UUID not working, so have to get object and cast it;
        Object identity = Utility.jacksonFromJson(output.toString(), ResponseIdentity.class).identity;
        return LinkFuture.Init.Utility.cast(identity, keyType);
    }
    @Override
    public <S extends T> List<ID> insert(List<S> obj) throws Exception {
        logger.trace("Insert {}", this.requestPath);
        Object output = genericDBResourceLoad(HttpMethod.Put, obj);
        if(output == null)
        {
            throw new RuntimeException("Insert failed");
        }
        ResponseIdentityList identityList = Utility.jacksonFromJson(output.toString(), ResponseIdentityList.class);
        List<ID> response = new ArrayList<>();
        for (Object ob : identityList.identity)
        {
            response.add(LinkFuture.Init.Utility.cast(ob, keyType));
        }
        return response;
    }

    @Override
    public <S extends T> Integer save(S obj) throws Exception {
        GenericSet.Builder builder = GenericSet.builder();
        GenericWhere.Builder where = GenericWhere.builder();
        HashMap<String,FieldInfo> fields = findEntityFieldInfo(entityType);
        for (String name : fields.keySet()) {
            FieldInfo field = fields.get(name);
            Object myValue = field.field.get(obj);
            if(myValue!=null)
            {
                if(field.fieldAttribute.isKey())
                {
                    where.and(name,myValue);
                }
                else
                {
                    builder.set(name,myValue);
                }
            }
        }
        builder.where(where.build());
        return save(builder.build());
    }
    @Override
    public Integer save(GenericSet q) throws IOException, URISyntaxException {
        logger.trace("Save {}", this.requestPath);
        try {
            return (int)genericDBResourceLoad(HttpMethod.Post, q);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
    @Override
    public Integer delete(ID id) throws Exception {
        GenericWhere.Builder builder = GenericWhere.builder();
        HashMap<String,FieldInfo> fields = findEntityKeyFieldInfo(entityType);
        if(fields.size()==1)
        {
            builder.and(fields.keySet().iterator().next(), id);
        }
        else
        {
            throw new NoSuchMethodError("this api " + requestPath + " have multiple key, please try delete(S obj) instead.");
        }
        try {
            return delete(builder.build());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public <S extends T> Integer delete(S obj) throws Exception {
        HashMap<String, FieldInfo> fields = findEntityKeyFieldInfo(entityType);
        GenericWhere.Builder builder = GenericWhere.builder();
        for (String name : fields.keySet()) {
            FieldInfo field = fields.get(name);
            field.field.setAccessible(true);
            builder.and(name, field.field.get(obj));
        }
        try {
            return delete(builder.build());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
    @Override
    public Integer delete(GenericWhere q) throws Exception{
        logger.trace("Delete {}", this.requestPath);
        return (int)genericDBResourceLoad(HttpMethod.Delete, q);
    }
    @Override
    public PageableList<T> findAll(Pageable pager) throws Exception {
        return find(null, pager);
    }
    @Override
    public <S extends T> PageableList<T> find(S obj) throws Exception {
        return find(obj,null);
    }
    @Override
    public T find(ID id) throws Exception {
        HashMap<String,FieldInfo> fields = findEntityKeyFieldInfo(entityType);
        if(fields.size()==1)
        {
            PageableList<T> results = find(
                    GenericQuery.builder().where(GenericWhere.builder().and(
                                    fields.keySet().iterator().next(), id
                            ).build()
                    ).build()
            );
            return results!=null && results.data!=null && results.data.length >0 ? results.data[0]:null;
        }
        else
        {
            throw new IllegalArgumentException("We can't find unique primary key from this entity type, please use PageableList<T> find(GenericQuery q) instead.");
        }
    }
    @Override
    public <S extends T> PageableList<T> find(S obj, Pageable pager) throws Exception {
        GenericQuery.Builder query = GenericQuery.builder();
        if(obj!=null)
        {
            GenericWhere.Builder where = GenericWhere.builder();
            HashMap<String, FieldInfo> fields = findEntityFieldInfo(entityType);
            for (String name : fields.keySet()) {
                FieldInfo field = fields.get(name);
                Object myValue = field.field.get(obj);
                if(myValue!=null)
                {
                    if(field.jsonFormatAttribute!=null && !StringExtension.IsNullOrEmpty(field.jsonFormatAttribute.pattern()) && myValue instanceof Date)
                    {
                        where.and(name, DateExtension.Format((Date) myValue, field.jsonFormatAttribute.pattern()));
                    }
                    else if(isCustomClass(myValue))
                    {
                        where.and(name, Utility.jacksonToJson(myValue));
                    }
                    else
                    {
                        where.and(name, myValue);
                    }

                }
            }
            GenericWhere whereBuilder = where.build();
            if(whereBuilder.size()==0)
            {
                throw new IllegalArgumentException("Missing where condition.");
            }
            query.where(whereBuilder);
        }
        if(pager!=null)
        {
            query.limit(pager.getPageSize());
            query.offset(pager.getOffset());
        }
        return find(query.build());
    }
    @Override
    public PageableList<T> find(GenericQuery q) throws Exception{
        logger.trace("Find {}", this.requestPath);
        Object output = genericDBResourceLoad(HttpMethod.Get, q);
        return toList(output, (Class<T>) entityType);
    }

    public static boolean isCustomClass(Object check){
        return !check.getClass().isEnum()  && !check.getClass().getName().startsWith("java");
    }
}
