package LinkFuture.Core.DBHelper;

import LinkFuture.Core.DBHelper.Model.ColumnInfo;
import LinkFuture.Core.DBHelper.Model.CommandTypeInfo;
import LinkFuture.Core.DBHelper.Model.DBTypeInfo;
import LinkFuture.Core.DBHelper.Model.TableInfo;
import LinkFuture.Init.Config;
import LinkFuture.Init.Extensions.StringExtension;
import LinkFuture.Init.ObjectExtend.NameValuePair;
import org.json.JSONArray;
import org.json.JSONObject;

import javax.naming.NamingException;
import java.io.IOException;
import java.sql.SQLException;
import java.sql.Types;
import java.util.*;

/**
 * Created by Cyokin
 * on 7/1/2015.
 */
@SuppressWarnings("StringBufferReplaceableByString")
public class GenericDBHelper extends DBHelper{
    List<String> whereList = new ArrayList<>();
    List<String> insertList = new ArrayList<>();
    List<String> columnList = new ArrayList<>();
    List<String> whereKeyList = new ArrayList<>();
    List<String> updateList = new ArrayList<>();

    public GenericDBHelper(String connectionString) throws IOException, SQLException, ClassNotFoundException, NamingException {
        super(connectionString);
    }

    //region Build Condition
    private void buildCondition(TableInfo table,JSONObject jsonQuery)
    {
        whereList.clear();
        columnList.clear();
        insertList.clear();
        whereKeyList.clear();
        updateList.clear();
        for (ColumnInfo column:table.columnList)
        {
            Iterator<?> keys = jsonQuery.keys();
            while (keys.hasNext()){
                String key = (String)keys.next();
                if(!key.startsWith("$"))
                {
                    String inputColumnName = key;
                    int subQueryIndex = key.indexOf(".");
                    String subQuery=null;
                    if(column.sqlType==Types.STRUCT && subQueryIndex>0)
                    {
                        inputColumnName = key.substring(0,subQueryIndex);
                        subQuery = key.substring(subQueryIndex+1);
                    }
                    if(column.columnName.equalsIgnoreCase(inputColumnName))
                    {
                        columnList.add(column.columnName);
                        insertList.add(buildColumnValue(column));
                        if(StringExtension.IsNullOrEmpty(subQuery))
                        {
                            updateList.add(String.format("%s=%s", column.columnName, buildColumnValue(column)));
                        }
                        else
                        {
                            updateList.add(String.format("%s.%s= $%s|%s", column.columnName,subQuery,column.columnName,Types.VARCHAR ));
                        }
                        super.addParameter(column.columnName, jsonQuery.get(key));
                    }
                }
            }
        }
    }
    private void buildCondition(TableInfo table,NameValuePair...params) throws IllegalAccessException {
        whereList.clear();
        columnList.clear();
        insertList.clear();
        whereKeyList.clear();
        updateList.clear();

        for (ColumnInfo column:table.columnList)
        {
            for (NameValuePair param :params)
            {
                //contain query
                String inputColumnName = param.id;
                int subQueryIndex = param.id.indexOf(".");
                String subQuery=null;
                if(column.sqlType==Types.STRUCT && subQueryIndex>0)
                {
                    inputColumnName = param.id.substring(0,subQueryIndex);
                    subQuery = param.id.substring(subQueryIndex+1);
                }
                if(column.columnName.equalsIgnoreCase(inputColumnName))
                {
                    columnList.add(column.columnName);

                    whereList.add(buildWhere(column, subQuery));
                    insertList.add(buildColumnValue(column));
                    if(column.isKey)
                    {
                        whereKeyList.add(buildWhere(column, subQuery));
                    }
                    else
                    {
                        if(StringExtension.IsNullOrEmpty(subQuery))
                        {
                            updateList.add(String.format("%s=%s", column.columnName, buildColumnValue(column)));
                        }
                        else
                        {
                            updateList.add(String.format("%s.%s= $%s|%s", column.columnName,subQuery,column.columnName,Types.VARCHAR ));
                        }
                    }
                    super.addParameter(column.columnName, param.value);
                    break;
                }
            }
        }
    }
    private String buildWhere(ColumnInfo column,String subQuery) throws IllegalAccessException {
        if(column.sqlType==Types.ARRAY)
        {
            return String.format("$%s|%s::%s = ANY (%s)", column.columnName,column.getArrayElementType() ,column.getArrayElementTypeName(),column.columnName);
        }
        if(column.isSqlJsonType())
        {
            return String.format("%s @> $%s|%s::%s", column.columnName,column.columnName,column.sqlType,column.sqlTypeName);
        }
        if(column.sqlType==Types.STRUCT)
        {
            if(StringExtension.IsNullOrEmpty(subQuery))
            {
                return String.format("%s = $%s|%s::%s", column.columnName, column.columnName,column.sqlType,column.sqlTypeName);
            }
            return String.format("(%s).%s = $%s|%s", column.columnName,subQuery, column.columnName,Types.VARCHAR);
        }
        return String.format("%s = $%s|%s", column.columnName, column.columnName,column.sqlType);
    }
    private String buildColumnValue(ColumnInfo column)  {
        if(column.sqlType== Types.STRUCT)
        {
            return String.format("$%s|%s::%s", column.columnName, column.sqlType,column.sqlTypeName);
        }
        if(column.sqlType== Types.ARRAY)
        {
            //it will return _varchar when array type, must remove it
            return String.format("$%s|%s::%s[]", column.columnName, column.sqlType,column.getArrayElementTypeName());
            //return String.format("ARRAY[$%s|%s::%s[]", column.columnName, column.sqlType,column.getArrayElementTypeName());
        }
        return String.format("$%s|%s", column.columnName, column.sqlType);
    }
    //endregion

    //region UPDATE
    public int update(String tableName,JSONObject jsonQuery) throws Exception {
        TableInfo table = init(tableName);
        buildCondition(table, jsonQuery);
        if(!jsonQuery.has("$where"))
        {
            throw new IllegalArgumentException("Please specific parameter or none of input parameter is valid for " + tableName);
        }
        String where =  buildWhere(super.getTableColumnList(tableName),jsonQuery.getJSONObject("$where"));
        if(updateList.size() >0 && !StringExtension.IsNullOrEmpty(where))
        {

            StringBuilder sb = new StringBuilder();
            sb.append("UPDATE ");
            sb.append(tableName);
            sb.append(" SET ");
            sb.append(StringExtension.Join(updateList, ","));
            sb.append(" WHERE ");
            sb.append(where);
            return super.executeSQL(sb.toString());
        }
        else
        {
            throw new IllegalArgumentException("Please specific parameter or none of input parameter is valid for " + tableName);
        }
    }

    public int update(String tableName,Collection<NameValuePair> params) throws Exception {
        return update(tableName, params.toArray(new NameValuePair[params.size()]));
    }
    public int update(String tableName,NameValuePair...params) throws Exception {
        TableInfo table = init(tableName);
        buildCondition(table,params);
        if(updateList.size() >0 && whereKeyList.size() >0)
        {
            StringBuilder sb = new StringBuilder();
            sb.append("UPDATE ");
            sb.append(tableName);
            sb.append(" SET ");
            sb.append(StringExtension.Join(updateList, ","));
            sb.append(" WHERE ");
            sb.append(StringExtension.Join(whereKeyList, " and "));
            return super.executeSQL(sb.toString());
        }
        else
        {
            throw new IllegalArgumentException("Please specific parameter or none of input parameter is valid for " + tableName);
        }
    }
    //endregion

    //region DELETE
    public int delete(String tableName,Collection<NameValuePair> params) throws Exception {
        return delete(tableName, params.toArray(new NameValuePair[params.size()]));
    }
    public int delete(String tableName,NameValuePair...params) throws Exception {
        TableInfo table = init(tableName);
        buildCondition(table,params);
        if(whereList.size()>0)
        {
            StringBuilder sb = new StringBuilder();
            sb.append("DELETE FROM ");
            sb.append(tableName);
            sb.append(" WHERE ");
            sb.append(StringExtension.Join(whereList," and "));
            return super.executeSQL(sb.toString());
        }
        else
        {
            throw new IllegalArgumentException("Please specific parameter or none of input parameter is valid for " + tableName);
        }
    }
    public int delete(String tableName,JSONObject jsonQuery) throws Exception {
        HashMap<String,ColumnInfo> columnList = super.getTableColumnList(tableName);
        String where = buildWhere(columnList,jsonQuery);
        if(StringExtension.IsNullOrEmpty(where))
        {
            throw new IllegalArgumentException("Please specific parameter or none of input parameter is valid for " + tableName);
        }
        StringBuilder sb = new StringBuilder();
        sb.append("DELETE FROM ");
        sb.append(tableName);
        sb.append(" WHERE ");
        sb.append(where);
        return super.executeSQL(sb.toString());
    }
    //endregion

    //region SELECT

    //TODO:known issue: query build doesn't support multiple same fields in where
    private String buildSelectQuery(String tableName,JSONObject jsonQuery) throws Exception {
        HashMap<String,ColumnInfo> columnList = super.getTableColumnList(tableName);
        List<String> queryColumnList = new ArrayList<>();
        Iterator<?> keys = jsonQuery.keys();

        String myWhere = null;
        Integer limit= 10;
        Integer offset= 0;
        while (keys.hasNext()){
            String key = (String)keys.next();
            if(!key.startsWith("$"))
            {
                boolean included = jsonQuery.get(key).toString().equalsIgnoreCase("true");
                if(included && columnList.containsKey(key))
                {
                    queryColumnList.add(key);
                }
            }
            if(key.equalsIgnoreCase("$where"))
            {
                myWhere = buildWhere(columnList,(JSONObject)jsonQuery.get(key));
            }
            if(key.equalsIgnoreCase("$limit"))
            {
                limit = (int)jsonQuery.get(key);
            }
            if(key.equalsIgnoreCase("$offset"))
            {
                offset = (int)jsonQuery.get(key);
            }
        }
        StringBuilder sb = new StringBuilder();
        sb.append("SELECT ");
        if(queryColumnList.size() == 0)
        {
            sb.append(" * ");
        }
        else
        {
            sb.append(StringExtension.Join(queryColumnList, ","));
        }
        if(limit>0)
        {
            if(DBType== DBTypeInfo.PostgreSQL)
            {
                sb.append(" ,count(1) OVER() AS ").append(Config.keyTotalCount);
                sb.append(" ,").append(limit).append(" AS ").append(Config.keyPageLimit);
                sb.append(" ,").append(offset).append(" AS ").append(Config.keyPageOffset);
            }
            if(DBType== DBTypeInfo.MySql)
            {
                sb.append(" ,FOUND_ROWS AS ").append(Config.keyTotalCount);
                sb.append(" ,").append(limit).append(" AS ").append(Config.keyPageLimit);
                sb.append(" ,").append(offset).append(" AS ").append(Config.keyPageOffset);
            }
        }
        sb.append(" FROM ");
        sb.append(tableName);
        if(!StringExtension.IsNullOrEmpty(myWhere))
        {
            sb.append(" WHERE ");
            sb.append(myWhere);
        }

        if(limit>0)
        {
            //add query pagination support
            if(DBType== DBTypeInfo.PostgreSQL)
            {
                sb.append(" LIMIT ").append(limit);
                sb.append(" OFFSET ").append(offset);
            }
            if(DBType== DBTypeInfo.MySql)
            {
                sb.append(" LIMIT ");
                sb.append(offset).append(",");
                sb.append(limit);
            }
        }

        return sb.toString();
    }
    private String buildWhere(HashMap<String,ColumnInfo> columnList,JSONObject jsonWhere) throws IllegalAccessException {
        Iterator<?> keys = jsonWhere.keys();
        List<String> localAndList = new ArrayList<>();
        List<String> localOrList = new ArrayList<>();
        while (keys.hasNext()){
            String key = (String)keys.next();
            Object jsonValue = jsonWhere.get(key);
            String inputColumnName = key;
            String subQuery=null;
            int subQueryIndex = inputColumnName.indexOf(".");
            if(subQueryIndex>0)
            {
                inputColumnName = key.substring(0,subQueryIndex);
                subQuery = key.substring(subQueryIndex+1);
            }
            if(!key.startsWith("$") && columnList.containsKey(inputColumnName))
            {
                ColumnInfo column = columnList.get(inputColumnName);
                if(jsonValue instanceof JSONObject)
                {
                    JSONObject compare = (JSONObject)jsonValue;
                    String operationString = compare.keySet().iterator().next().toString().toLowerCase();
                    Object operationValue = compare.get(operationString);
                    switch (operationString)
                    {
                        case "$gt":
                            if(!comparableTypes.contains(column.sqlType))
                            {
                                throw new IllegalArgumentException("Specific type does not support $gt operation");
                            }
                            localAndList.add(String.format("%s > $%s|%s", column.columnName, column.columnName, column.sqlType));
                            super.addParameter(column.columnName, operationValue);
                            break;
                        case "$gte":
                            if(!comparableTypes.contains(column.sqlType))
                            {
                                throw new IllegalArgumentException("Specific type does not support $gte operation");
                            }
                            localAndList.add(String.format("%s >= $%s|%s", column.columnName, column.columnName, column.sqlType));
                            super.addParameter(column.columnName, operationValue);
                            break;
                        case "$lt":
                            if(!comparableTypes.contains(column.sqlType))
                            {
                                throw new IllegalArgumentException("Specific type does not support $lt operation");
                            }
                            localAndList.add(String.format("%s < $%s|%s", column.columnName, column.columnName, column.sqlType));
                            super.addParameter(column.columnName, operationValue);
                            break;
                        case "$lte":
                            if(!comparableTypes.contains(column.sqlType))
                            {
                                throw new IllegalArgumentException("Specific type does not support $lte operation");
                            }
                            localAndList.add(String.format("%s <= $%s|%s", column.columnName, column.columnName, column.sqlType));
                            super.addParameter(column.columnName, operationValue);
                            break;
                        case "$in":
                            JSONArray inList = (JSONArray)operationValue;
                            List<String> inString = new ArrayList<>();
                            for (int i = 0; i < inList.length(); i++) {
                                inString.add("'"+ inList.get(i).toString() + "'");
                            }
                            localAndList.add(String.format("%s in (%s)", column.columnName, StringExtension.Join(inString, ",")));
                            break;
                    }
                }
                else
                {
                    localAndList.add(buildWhere(column, subQuery));
                    super.addParameter(column.columnName, jsonValue);
                }
            }
            if(key.equalsIgnoreCase("$or"))
            {
                JSONArray orArray = (JSONArray)jsonValue;
                for (int i = 0; i < orArray.length(); i++) {
                    localOrList.add(buildWhere(columnList, (JSONObject)orArray.get(i)));
                }
            }
        }
        StringBuilder sb = new StringBuilder();
        sb.append("(");
        sb.append(StringExtension.Join(localAndList, " AND "));
        //add or
        String orQuery = StringExtension.Join(localOrList, " OR ");
        if(localAndList.size()>0 && orQuery.length()>0)
        {
            sb.append(" OR ");
        }
        sb.append(orQuery);
        sb.append(")");
        String output = sb.toString();
        //if output = "()"
        return output.length()==2?Config.Empty:sb.toString();
    }

    private String buildSelectQuery(String tableName,List<String> queryColumnList,NameValuePair...params) throws Exception {
        TableInfo table = super.findTableInfo(tableName);
        buildCondition(table,params);
        Integer limit= 10;
        Integer offset= 0;
        for (NameValuePair item:params)
        {
            if(item.id.equalsIgnoreCase("limit"))
            {
                limit=Integer.parseInt(item.value.toString());
                continue;
            }
            if(item.id.equalsIgnoreCase("offset"))
            {
                offset=Integer.parseInt(item.value.toString());
            }
        }
        StringBuilder sb = new StringBuilder();
        sb.append("SELECT ");
        if(queryColumnList==null || queryColumnList.size()==0)
        {
            sb.append(" * ");
        }
        else
        {
            //query field
            List<String> filterColumnList = new ArrayList<>();
            for (ColumnInfo column:table.columnList)
            {
                for (String queryField:queryColumnList)
                {
                    if(column.columnName.equalsIgnoreCase(queryField))
                    {
                        filterColumnList.add(column.columnName);
                        break;
                    }
                }
            }
            if(filterColumnList.size()>0)
            {
                sb.append(StringExtension.Join(filterColumnList, ","));
            }
            else
            {
                sb.append(" * ");
            }
        }
        if(limit>0)
        {
            if(DBType== DBTypeInfo.PostgreSQL)
            {
                sb.append(" ,count(1) OVER() AS ").append(Config.keyTotalCount);
                sb.append(" ,").append(limit).append(" AS ").append(Config.keyPageLimit);
                sb.append(" ,").append(offset).append(" AS ").append(Config.keyPageOffset);
            }
            if(DBType== DBTypeInfo.MySql)
            {
                sb.append(" ,FOUND_ROWS AS ").append(Config.keyTotalCount);
                sb.append(" ,").append(limit).append(" AS ").append(Config.keyPageLimit);
                sb.append(" ,").append(offset).append(" AS ").append(Config.keyPageOffset);
            }
        }

        sb.append(" FROM ");
        sb.append(tableName);
        if(whereList.size()>0)
        {
            sb.append(" WHERE ");
            sb.append(StringExtension.Join(whereList," and "));
        }

        if(limit>0)
        {
            //add query pagination support
            if(DBType== DBTypeInfo.PostgreSQL)
            {
                sb.append(" LIMIT ").append(limit);
                sb.append(" OFFSET ").append(offset);
            }
            if(DBType== DBTypeInfo.MySql)
            {
                sb.append(" LIMIT ");
                sb.append(offset).append(",");
                sb.append(limit);
            }
        }
        return sb.toString();

    }
    public String selectToXml(String tableName,List<String> queryColumnList,Collection<NameValuePair> params) throws Exception {
        return selectToXml(tableName,queryColumnList, params.toArray(new NameValuePair[params.size()]));
    }
    public String selectToXml(String tableName,List<String> queryColumnList,NameValuePair...params) throws Exception {
        return super.executeToXml(buildSelectQuery(tableName,queryColumnList, params), CommandTypeInfo.TSQL);
    }
    public String selectToXml(String tableName,JSONObject jsonQuery) throws Exception {
        return super.executeToXml(buildSelectQuery(tableName,jsonQuery), CommandTypeInfo.TSQL);
    }

    public String selectToJson(String tableName,List<String> queryColumnList,Collection<NameValuePair> params) throws Exception {
        return selectToJson(tableName,queryColumnList, params.toArray(new NameValuePair[params.size()]));
    }
    public String selectToJson(String tableName,List<String> queryColumnList,NameValuePair...params) throws Exception {
        JSONObject output = super.executeToJson(buildSelectQuery(tableName,queryColumnList, params), CommandTypeInfo.TSQL);
        return StringExtension.IfNULLOrEmpty(output);
    }
    public String selectToJson(String tableName,JSONObject jsonQuery) throws Exception {
        JSONObject output = super.executeToJson(buildSelectQuery(tableName,jsonQuery), CommandTypeInfo.TSQL);
        return StringExtension.IfNULLOrEmpty(output);
    }
    //endregion

    //region INSERT
    public Object insert(String tableName,Collection<NameValuePair> params) throws Exception {
        return insert(tableName,params.toArray(new NameValuePair[params.size()]));
    }
    public Object insert(String tableName,NameValuePair...params) throws Exception {
        TableInfo table = init(tableName);
        buildCondition(table,params);
        if(whereList.size() >0 && insertList.size() >0)
        {
            StringBuilder sb = new StringBuilder();
            sb.append("INSERT INTO ");
            sb.append(tableName);
            sb.append("(");
            sb.append(StringExtension.Join(columnList, ","));
            sb.append(") VALUES(");
            sb.append(StringExtension.Join(insertList,","));
            sb.append(");");
            return super.insert(sb.toString());
        }
        throw new IllegalArgumentException("Please specific parameter or none of input parameter is valid for " + tableName);
    }
    //endregion

    @SuppressWarnings("ThrowableInstanceNeverThrown")
    private TableInfo init(String tableName) throws Exception {
        super.inputParameterList.clear();
        TableInfo table = super.findTableInfo(tableName);
        if(table == null || table.columnList.size()==0)
        {
            new IllegalAccessException("Specific table doesn't exist:"+tableName);
        }
        return table;
    }
}
