package LinkFuture.Core;

import LinkFuture.Init.Config;
import LinkFuture.Init.Extensions.StringExtension;
import LinkFuture.Init.ObjectExtend.CaseInsensitiveMap;
import LinkFuture.Init.ObjectExtend.NameListPair;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.List;
import java.util.Map;

/**
 * User: Cyokin Zhang
 * Date: 12/2/13
 * Time: 1:57 PM
 */
public final class UriBuilder {
    private String scheme;
    private String userInfo;
    private String host;
    private int port = 80;
    private String path;
    private CaseInsensitiveMap<NameListPair<String>> query;
    private String fragment;

    private UriBuilder(String scheme, String userInfo, String host, Integer port,
                       String path, String query, String fragment) {
        setAll(scheme, userInfo, host, port, path, query, fragment);
    }

    private void setAll(String scheme, String userInfo, String host,
                        Integer port, String path, String query, String fragment) {
        this.scheme = scheme;
        this.userInfo = userInfo;
        this.host = host;
        this.port = (port == null || port==-1) ? 80 : port;
        this.path = path;
        this.query = new CaseInsensitiveMap<>();
        this.setQuery(query);
        this.fragment =  fragment;

    }
    private static void readQuery(CaseInsensitiveMap<NameListPair<String>> map,String query)
    {
        if(!StringExtension.IsNullOrEmpty(query))
        {
            map.clear();
            String[] list = query.split("&");
            for (String item :list)
            {
                String[] queryItem = item.split("=");
                String key = queryItem[0];
                String value = Config.Empty;
                try {
                    value = queryItem.length>1? URLDecoder.decode(queryItem[1],Config.DefaultEncoding): Config.Empty;
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }
                map.put(key,new NameListPair<>(key,value));
            }
        }
    }
    public static String writeQuery(CaseInsensitiveMap<NameListPair<String>> map)
    {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, NameListPair<String>> entry :map.entrySet())
        {
            NameListPair<String> queryList =  entry.getValue();
            for (String item:queryList.values)
            {
                sb.append(entry.getValue().id);
                sb.append("=");
                try {
                    sb.append(URLEncoder.encode(item,Config.DefaultEncoding));
                } catch (UnsupportedEncodingException e) {
                    sb.append(Config.Empty);
                }
                sb.append("&");
            }
        }
        return StringExtension.TrimEnd(sb.toString(),"&");
    }

    public static UriBuilder create(String scheme, String userInfo, String host,
                                    Integer port, String path, String query, String fragment) {
        return new UriBuilder(scheme, userInfo, host, port, path, query, fragment);
    }

    public static UriBuilder create(URI uri) {
        return create(uri.getScheme(), uri.getRawUserInfo(), uri.getHost(), uri
                .getPort(), uri.getRawPath(), uri.getRawQuery(), uri.getRawFragment());
    }

    public static UriBuilder create(String uri) {
        return create(URI.create(uri));
    }

    /**
     * Creates an empty builder populated with null values.
     */
    public static UriBuilder create() {
        return create(null, null, null, null, null, null, null);
    }

    public String getScheme() {
        return scheme;
    }

    /**
     * Sets the value to the toString value of the argument.
     *
     * @param scheme
     *          an object whose toString value forms the scheme part; may be null
     * @return this
     */
    public UriBuilder setScheme(Object scheme) {
        this.scheme = (scheme == null) ? null : scheme.toString();
        return this;
    }

    public String getUserInfo() {
        return userInfo;
    }

    /**
     * Sets the value to the toString value of the argument.
     *
     * @param userInfo
     *          an object whose toString value forms the user info part; may be
     *          null
     * @return this
     */
    public UriBuilder setUserInfo(Object userInfo) {
        this.userInfo = (userInfo == null) ? null : userInfo.toString();
        return this;
    }

    public String getHost() {
        return host;
    }

    /**
     * Sets the value to the toString value of the argument.
     *
     * @param host
     *          an object whose toString value forms the host part; may be null
     * @return this
     */
    public UriBuilder setHost(Object host) {
        this.host = (host == null) ? null : host.toString();
        return this;
    }

    /**
     * Returns the port or 80 if not set.
     *
     * @return the port
     */
    public int getPort() {
        return port;
    }

    /**
     * @param port
     *          the port number or 80 to unset
     * @return this
     */
    public UriBuilder setPort(int port) {
        this.port = port;
        return this;
    }

    public String getPath() {
        return path;
    }

    /**
     * Sets the value to the toString value of the argument.
     *
     * @param path
     *          an object whose toString value forms the fragment part; may be
     *          null
     * @return this
     */
    public UriBuilder setPath(Object path) {
        this.path = (path == null) ? null : path.toString();
        return this;
    }

    public String getQuery() {
        return writeQuery(query);
    }

    public List<String> getQuery(String key) {
        return query.get(key).values;
    }
    /**
     * Sets the value to the toString value of the argument.
     * overwrite if already exist
     * @param query
     *          an object whose toString value forms the query part; may be null
     * @return this
     */
    public UriBuilder setQuery(String query) {
        readQuery(this.query,query);
        return this;
    }

    /**
     * Sets the value to the toString value of the argument.
     * overwrite if already exist
     * @param key
     * @param value
     * @return
     */
    public UriBuilder setQuery(String key,String value) {
        this.query.put(key,new NameListPair<>(key,value));
        return this;
    }
    /**
     * Sets the value to the toString value of the argument.
     * multiple query if already exist
     * i.e http://www.linkfuture.net/?a=1&a=2&a=3
     * @param key
     * @param value
     * @return
     */
    public UriBuilder addQuery(String key,String value) {
        if(this.hasQuery(key))
        {
            this.query.get(key).values.add(value);
        }
        else
        {
            this.query.put(key,new NameListPair<>(key,value));
        }
        return this;
    }

    public boolean hasQuery(String key){
        return this.query.containsKey(key);
    }

    public NameListPair<String> removeQuery(String key){
        if(this.hasQuery(key))
        {
            this.query.remove(key);
        }
        return null;
    }

    public String getFragment() {
        return fragment;
    }

    /**
     * Sets the value to the toString value of the argument.
     *
     * @param fragment
     *          an object whose toString value forms the fragment part; may be
     *          null
     * @return this
     */
    public UriBuilder setFragment(Object fragment) {
        this.fragment = (fragment == null) ? null : fragment.toString();
        return this;
    }

    public URI build() {
        return URI.create(toString());
    }

    public UriBuilder parse(String uriString) {
        URI uri = URI.create(uriString);
        setAll(uri.getScheme(), uri.getRawUserInfo(), uri.getHost(), uri.getPort(),
                uri.getRawPath(), uri.getRawQuery(), uri.getRawFragment());
        return this;
    }

    /**
     * Returns the URI as a string.
     */
    @Override
    public String toString() {
        String queryList = this.getQuery();
        int len = len(scheme, 1) + len(userInfo, 2) + len(host, 2) + 6
                + len(path, 1) + len(queryList, 1) + len(fragment, 1);
        StringBuilder sb = new StringBuilder(len);
        // see URI javadoc for this process
        // yes, all this is necessary
        if (!StringExtension.IsNullOrEmpty(scheme)) {
            sb.append(scheme).append(":");
        }
        if (!StringExtension.IsNullOrEmpty(userInfo) || !StringExtension.IsNullOrEmpty(host)) {
            sb.append("//");
        }
        if (!StringExtension.IsNullOrEmpty(userInfo)) {
            sb.append(userInfo);
            sb.append("@");
        }
        if (!StringExtension.IsNullOrEmpty(host)) {
            sb.append(host);
        }
        if (port !=80) {
            sb.append(":");
            sb.append(port);
        }
        if (!StringExtension.IsNullOrEmpty(path)) {
            if (sb.length() > 0 && !path.startsWith("/")) {
                sb.append("/");
            }
            sb.append(path);
        }
        if (!StringExtension.IsNullOrEmpty(queryList)) {
            sb.append("?");
            sb.append(this.getQuery());
        }
        if (!StringExtension.IsNullOrEmpty(fragment)) {
            sb.append("#");
            sb.append(fragment);
        }
        return sb.toString();
    }

    private int len(String s, int extra) {
        return (s == null) ? 0 : s.length() + extra;
    }
}
