package LinkFuture.Core.MongoDBHelper;

import LinkFuture.Init.Debugger;
import LinkFuture.Init.Extensions.StringExtension;
import com.mongodb.*;

import java.net.UnknownHostException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by Cyokin
 * on 12/31/2014.
 */
public class MongoDBHelper {
    private MongoClient _client;
    private static final Map<String,MongoClient> connectionPool = new ConcurrentHashMap<>();
    public static MongoClient getMongoClient(String mongoClientURI) throws UnknownHostException {
        MongoClient localClient;
        if(!connectionPool.containsKey(mongoClientURI))
        {
            localClient = connect(mongoClientURI);
            connectionPool.put(mongoClientURI,localClient);
        }
        else
        {
            Debugger.LogFactory.info("Use connection from pool {}",mongoClientURI);
            try
            {
                localClient = connectionPool.get(mongoClientURI);
                //verify connection state first.
                localClient.getConnectPoint();
            }
            catch (IllegalStateException e)
            {
                //lost connection? need reconnect
                Debugger.LogFactory.info("Lost connection from pool {}, reconnect",mongoClientURI);
                localClient = connect(mongoClientURI);
                connectionPool.put(mongoClientURI,localClient);
            }
        }
        return connectionPool.get(mongoClientURI);
    }
    private static MongoClient connect(String mongoClientURI) throws UnknownHostException {
        Debugger.LogFactory.info("Connecting to mongo DB {}",mongoClientURI);
        MongoClientURI connectionUri = new MongoClientURI(mongoClientURI);
        MongoClient localClient =  new MongoClient(connectionUri);
        String defaultDBName = connectionUri.getDatabase();
        Debugger.LogFactory.info("Connected to mongo DB, current version is {}", localClient.getVersion());
        if(!StringExtension.IsNullOrEmpty(defaultDBName))
        {
            localClient.getDB(defaultDBName);
            Debugger.LogFactory.info("Use {} database", defaultDBName);
        }
        return localClient;
    }
    public MongoDBHelper(String mongoClientURI) throws UnknownHostException {
        this(getMongoClient(mongoClientURI));
    }
    public MongoDBHelper(MongoClient client)
    {
        this.setClient(client);
    }

    public Collection<DB> getUsedDatabases() {
        return this.getClient().getUsedDatabases();
    }

    public MongoClient getClient() {
        return this._client;
    }
    public void setClient(MongoClient client) {
        this._client = client;
    }
    public DB getDefaultDB(){
        Collection<DB> list = this.getUsedDatabases();
        if(list==null || list.size() ==0)
        {
            throw new IllegalArgumentException("We don't have default DB, please specific one");
        }
        final Iterator itr = list.iterator();
        Object last = itr.next();
        while(itr.hasNext()) {
            last = itr.next();
        }
        return (DB)last;
    }

    //region Execute
    public String execute(String commandText, Object... args){
        return execute(getDefaultDB(), commandText,args);
    }
    public String execute(DB selectedDB,String commandText, Object... args){
        //use command as do eval don's support no lock option yet.
        CommandResult result = selectedDB.command(BasicDBObjectBuilder.start().add("$eval", commandText).add("args", args).add("nolock",true).get());
        // selectedDB.doEval(commandText,args);
        if (result.ok())
        {
            return result.toString();
        }
        return null;
    }
    public String execute(DB selectedDB,String commandText, HashMap<String,Object> args){
        List<String> paramList = readFunctionParameters(commandText);
        Object[] passedParam = new Object[paramList.size()];
        for (int i = 0;i<passedParam.length;i++)
        {
            passedParam[i] = args.get(paramList.get(i));
        }
        return execute(selectedDB,commandText,passedParam);
    }
    public String execute(String commandText,HashMap<String,Object> args){
        return execute(getDefaultDB(), commandText,args);
    }
    //endregion
    public static List<String> readFunctionParameters(String functionCode)
    {
        List<String> list = new ArrayList<>();
        functionCode = functionCode.trim();
        if(functionCode.startsWith("function("))
        {
            Pattern pattern = Pattern.compile("function\\(([^)]*)\\)", Pattern.CASE_INSENSITIVE);
            Matcher matcher = pattern.matcher(functionCode);
            if(matcher.find())
            {
                String param = matcher.group(1);
                for (String item:param.split(","))
                {
                    list.add(item.trim());
                }
            }
        }
        return list;
    }

    public void close(){
        if(this._client!=null)
        {
            Debugger.LogFactory.info("close Mongo DB connection: {}", this._client.getAddress().toString());
            this._client.close();
        }
    }
    public static void closeAll(){
        for (MongoClient client:connectionPool.values())
        {
            Debugger.LogFactory.info("close Mongo DB connection: {}",client.getAddress().toString());
            client.close();
        }
    }
}
