package LinkFuture.Core.DBHelper;
/**
 * User: Cyokin Zhang
 * Date: 10/1/13
 * Time: 10:48 AM
 */

import LinkFuture.Core.DBHelper.Model.*;
import LinkFuture.Core.MemoryManager.StaticMemoryCache.StaticMemoryCacheHelper;
import LinkFuture.Core.OperationManager.Operation;
import LinkFuture.Init.Debugger;
import LinkFuture.Init.Extensions.DateExtension;
import LinkFuture.Init.Extensions.StringExtension;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import javax.xml.parsers.ParserConfigurationException;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.sql.*;
import java.text.ParseException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


public class DBHelper implements AutoCloseable {
    //public static String DefaultDriverClassName =  "com.mysql.jdbc.Driver";
//    public static String DefaultDriverClassName =  "com.microsoft.sqlserver.jdbc.SQLServerDriver";
//    static {
//        try {
//            Class.forName(DefaultDriverClassName);
//        } catch (ClassNotFoundException e) {
//            throw new RuntimeException(e);
//        }
//    }

    public static Connection getConnection(String connectionString) throws SQLException, NamingException, ClassNotFoundException {
        Debugger.LogFactory.trace("Connect to {} ",connectionString);
        if(connectionString.startsWith("java:/"))
        {
            InitialContext cxt = new InitialContext();
            DataSource ds = (DataSource) cxt.lookup(connectionString);
            return ds.getConnection();
        }
        else
        {
            return DriverManager.getConnection(connectionString);
        }
    }

    private Connection conn;
    private static final Map<String, SPInfo> CachedSPMeta = new ConcurrentHashMap<>();

    public Map<String, SPParameterInfo> inputParameterList = new LinkedHashMap<>();
    public String dbUrl;
    public SPInfo SPMetaInfo = null;
    public Boolean IsMicrosoftSQLServer = false;
    public Statement statement;
    private boolean autoClose = true;

    //region DBHelper Structure
    public DBHelper(Connection conn) throws SQLException, ClassNotFoundException {
        //don't not close connection if connect come from others
        this.autoClose = false;
        this.init(conn);
    }
    public DBHelper(String connectionString) throws IOException, SQLException, ClassNotFoundException, NamingException {
        this.init(getConnection(connectionString));
    }
    //endregion

    private void init(Connection conn) throws SQLException {
        this.conn = conn;
        this.dbUrl = DBConnectionInfo.Parser(conn.getMetaData().getURL()).Url;
        this.IsMicrosoftSQLServer = this.conn.getMetaData().getDatabaseProductName().equalsIgnoreCase("Microsoft SQL Server");

    }

    //region Parameter
    public void addParameter(String name, Object value)
    {
        addParameter(name,value,null);
    }
    public void addOutParameter(String name)
    {
        addParameter(name,null, ParameterTypeInfo.procedureColumnOut);
    }
    private void addParameter(String name, Object value, ParameterTypeInfo type)
    {
        if(type==null) type = ParameterTypeInfo.procedureColumnIn;
        SPParameterInfo parameter = new SPParameterInfo();
        parameter.parameterName = name;
        parameter.parameterValue = value;
        parameter.parameterType = type;
        inputParameterList.put(buildParamKey(name), parameter);
    }
    //endregion

    //region Execute

    //region Execute SP
    public ArrayList<ArrayList<?>> executeSP(String commandText,Class<?>... outputType) throws Exception {
        this.setCommand(commandText,CommandTypeInfo.StoredProcedure);
        boolean hadResults =this.Execute(CommandTypeInfo.StoredProcedure);
        if(hadResults)
        {
            return new DBBeanReader(this.statement,this.SPMetaInfo).Read(outputType);
        }
        return null;
    }
    public<T> ArrayList<T> executeSP(String commandText,Class<T> outputType) throws Exception {
        this.setCommand(commandText,CommandTypeInfo.StoredProcedure);
        boolean hadResults =this.Execute(CommandTypeInfo.StoredProcedure);
        if(hadResults)
        {
            ArrayList<ArrayList<?>> results = new DBBeanReader(this.statement,this.SPMetaInfo).Read(outputType);
            if(results.size() >0)
            {
                //noinspection unchecked
                return (ArrayList<T>)results.get(0);
            }
        }
        return null;
    }
    public String executeToXml(String commandText,CommandTypeInfo commandType) throws SQLException, ParserConfigurationException, IOException, ParseException, NoSuchMethodException, IllegalAccessException, InvocationTargetException {
        this.setCommand(commandText,commandType);
        boolean hadResults =this.Execute(commandType);
        if(!hadResults)
        {
            return null;
        }
        return new DBXmlReader(this.statement,this.SPMetaInfo).Read();
    }
    //endregion

    //region Execute TSQL
    public ArrayList<ArrayList<?>> executeSQL(String sql,Class<?>... outputType) throws Exception {
        this.setCommand(sql,CommandTypeInfo.TSQL);
        boolean hadResults =this.Execute(CommandTypeInfo.TSQL);
        if(hadResults)
        {
            return new DBBeanReader(this.statement,this.SPMetaInfo).Read(outputType);
        }
        return null;
    }
    public<T> ArrayList<T> executeSQL(String sql,Class<T> outputType) throws Exception {
        this.setCommand(sql,CommandTypeInfo.TSQL);
        boolean hadResults =this.Execute(CommandTypeInfo.TSQL);
        if(hadResults)
        {
            ArrayList<ArrayList<?>> results = new DBBeanReader(this.statement,this.SPMetaInfo).Read(outputType);
            if(results.size() >0)
            {
                //noinspection unchecked
                return (ArrayList<T>)results.get(0);
            }
        }
        return null;

    }
    public int executeSQL(String sql) throws SQLException, IllegalAccessException, InstantiationException, NoSuchMethodException, InvocationTargetException, ParseException {
        this.setCommand(sql, CommandTypeInfo.TSQL);
        return this.TSQLExecuteUpdate();
    }
    //endregion

    //endregion

    public void close(){
        try {
            if(this.statement!=null && !this.statement.isClosed())
            {
                this.statement.close();
                this.statement = null;
            }
            if(this.autoClose && this.conn!=null && !this.conn.isClosed())
            {
                this.conn.close();
            }
        } catch (SQLException e) {
            Debugger.LogFactory.error("Close DB error", e);
        }
    }

    //region Private Method
    private void setCommand(String commandText,CommandTypeInfo commandType) throws SQLException, NoSuchMethodException, IllegalAccessException, InvocationTargetException {
        if(commandType == CommandTypeInfo.StoredProcedure)
        {
            this.SPMetaInfo = this.findSPInfo(commandText);
        }
        else
        {
            this.SPMetaInfo =  this.findTSQLInfo(commandText);
        }
    }
    private boolean Execute(CommandTypeInfo commandType) throws SQLException, ParseException {
        switch (commandType)
        {
            case TSQL:
                return TSQLExecute();
            case StoredProcedure:
                default:
                return SPExecute();
        }
    }
    private boolean SPExecute() throws SQLException, ParseException {
        CallableStatement callableStatement = this.conn.prepareCall(this.SPMetaInfo.sqlCall);
        this.statement =callableStatement;
        //fill
        this.fillParameter(this.SPMetaInfo, callableStatement);
        //exec
        return callableStatement.execute();
    }
    private boolean TSQLExecute() throws SQLException, ParseException {
        PreparedStatement preparedStatement= this.conn.prepareStatement(this.SPMetaInfo.sqlCall);

        this.statement =preparedStatement;
        //fill
        this.fillParameter(this.SPMetaInfo,preparedStatement);
        //exec
        return preparedStatement.execute();
    }
    private int TSQLExecuteUpdate() throws SQLException, ParseException {
        PreparedStatement preparedStatement= this.conn.prepareStatement(this.SPMetaInfo.sqlCall);

        this.statement =preparedStatement;
        //fill
        this.fillParameter(this.SPMetaInfo,preparedStatement);
        //exec
        return preparedStatement.executeUpdate();
    }
    private void fillParameter(SPInfo spMeta, PreparedStatement stmt) throws SQLException, ParseException {
        if(spMeta.parameterList.size()>0)
        {
            boolean indexModel = spMeta.parameterList.get(0).parameterName.equalsIgnoreCase("?");
            int length =spMeta.parameterList.size();
            List keys = new ArrayList(inputParameterList.keySet());
            for (int i=0;i<length;i++)
            {
                if(indexModel)
                {
                    String paramKey = ((String)keys.get(i));
                    SPParameterInfo passedParam = inputParameterList.get(buildParamKey(paramKey));
                    stmt.setObject(i+1, passedParam.parameterValue);
                }
                else
                {
                    SPParameterInfo param = spMeta.parameterList.get(i);
                    SPParameterInfo passedParam = inputParameterList.get(buildParamKey(param.parameterName));
                    stmt.setObject(i+1, buildPassedValue(param.sqlTypes, passedParam.parameterValue),param.sqlTypes);
                }
            }
        }
    }
    private Object buildPassedValue(int sqlTypes,Object passValue) throws ParseException {
        if((sqlTypes == Types.TIMESTAMP || sqlTypes == Types.DATE) && passValue instanceof String )
        {
            //need pass string to Date
            return DateExtension.Parse((String)passValue);
        }
        else
        {
            return passValue;
        }
    }
    private void fillParameter(SPInfo spMeta, CallableStatement stmt) throws SQLException, ParseException {
        if(spMeta.parameterList.size()>0)
        {
            for (SPParameterInfo param:spMeta.parameterList){
                String paramKey = buildParamKey(param.parameterName) ;
                String passParamName=param.parameterName;
                if(inputParameterList.containsKey(paramKey) && param.parameterType ==ParameterTypeInfo.procedureColumnIn)
                {
                    //Types.TIMESTAMP
                    Object passValue = inputParameterList.get(paramKey).parameterValue;
                    stmt.setObject(passParamName, buildPassedValue(param.sqlTypes,passValue));
                }
                else
                {
                    //append null value as default
                    if(param.parameterType ==ParameterTypeInfo.procedureColumnIn)
                    {
                        stmt.setNull(passParamName,param.sqlTypes);
                    }
                }
                if(param.parameterType == ParameterTypeInfo.procedureColumnOut
                        || param.parameterType ==ParameterTypeInfo.procedureColumnInOut
                        || param.parameterType == ParameterTypeInfo.procedureColumnReturn
                        )
                {
                    stmt.registerOutParameter(passParamName, param.sqlTypes);
                }
            }
        }
    }
    //endregion

    //region SPMetaInfo
    public SPInfo findSPInfo(String spName) throws SQLException {
    /*
     http://technet.microsoft.com/en-us/library/ms378416.aspx
     AS mySQL not support default in SP, so just ignore it
     PROCEDURE_NAME: The name of the stored procedure.
     COLUMN_NAME:  The name of the column.
     COLUMN_TYPE:  The type of the column. It can be one of the following values:
         procedureColumnUnknown (0)
         procedureColumnIn (1)
         procedureColumnInOut (2)
         procedureColumnOut (4)
         procedureColumnReturn (5)
         procedureColumnResult (3)
    DATA_TYPE: java.sql.Types (smallint)
    TYPE_NAME: The name of the sql data type. (String)
    NULLABLE:  Indicates if the column can contain a null value. It can be one of the following values:
         procedureNoNulls (0)
         procedureNullable (1)
         procedureNullableUnknown (2)
    COLUMN_DEF: The default value of the column.
     */
        String key = (dbUrl + "_" + spName).toLowerCase();
        if(!CachedSPMeta.containsKey(key))
        {
            Debugger.LogFactory.trace("Read SP Info:{}",key);
            SPInfo spMeta = new SPInfo();
            spMeta.dbName = dbUrl;
            spMeta.spName = spName;
            DatabaseMetaData dbmd = this.conn.getMetaData();
            ResultSet rs = dbmd.getProcedureColumns(null, null, spName, null);
            while (rs.next()) {
                SPParameterInfo parameter = new SPParameterInfo();
                parameter.parameterName = removeMSSQLPre(rs.getString("COLUMN_NAME"));
                parameter.parameterType = ParameterTypeInfo.convert(rs.getInt("COLUMN_TYPE"));
                if(parameter.parameterType ==ParameterTypeInfo.procedureColumnReturn)
                {
                    spMeta.hasReturn = true;
                }
                parameter.sqlTypes = rs.getInt("DATA_TYPE");
                spMeta.parameterList.add(parameter);
            }
            spMeta.sqlCall =  buildSPCallString(spMeta);
            CachedSPMeta.put(key,spMeta);
        }
        return CachedSPMeta.get(key);
    }
    private String removeMSSQLPre(String paramName){
        if(IsMicrosoftSQLServer && paramName.startsWith("@"))
        {
            return paramName.substring(1);
        }
        return paramName;
    }
    private static String buildParamKey(String paramName){
        if(paramName.startsWith("@") || paramName.startsWith("$"))
        {
            return paramName.substring(1).toLowerCase();
        }
        return paramName.toLowerCase();
    }
    private String buildSPCallString(SPInfo spMeta) throws SQLException {
        String paramTemplete = "";
        if(spMeta.parameterList.size()>0)
        {
            StringBuilder sb = new StringBuilder();
            sb.append("(");
            for (SPParameterInfo ignored :spMeta.parameterList){
                if(ignored.parameterType !=ParameterTypeInfo.procedureColumnReturn)
                {
                    sb.append("?,");
                }
            }
            //delete last comma ","
            sb.deleteCharAt(sb.length()-1);
            sb.append(")");
            paramTemplete = sb.toString();
        }
        if(spMeta.hasReturn)
        {
            return String.format("{? = call %s%s}", spMeta.spName, paramTemplete);
        }
        return String.format("{call %s%s}", spMeta.spName, paramTemplete);
    }

    public SPInfo findTSQLInfo(String commandText) throws SQLException, NoSuchMethodException, IllegalAccessException, InvocationTargetException {
        String key = (dbUrl + "_" + commandText).toLowerCase();
        if(!CachedSPMeta.containsKey(key))
        {
            SPInfo spMeta = new SPInfo();
            spMeta.dbName = dbUrl;
            spMeta.sqlCall = commandText;
            spMeta.spName = "TSQL";
            ArrayList<String> list = findTSQLParameters(commandText);
            if(list.size()>0)
            {
                for (String param :list)
                {
                    SPParameterInfo parameter = new SPParameterInfo();
                    String[] paramMeta = param.split("\\|");
                    String paramName = paramMeta[0];
                    parameter.parameterName =paramName;
                    parameter.parameterType = ParameterTypeInfo.procedureColumnIn;
                    if(paramMeta.length>1)
                    {
                        parameter.sqlTypes = Integer.parseInt(paramMeta[1]);
                    }
                    else
                    {
                        parameter.sqlTypes = Types.VARCHAR;
                    }
                    spMeta.parameterList.add(parameter);
                    //reset TSQL string, as this.conn.prepareStatement don't understand ":Param"
                    spMeta.sqlCall = spMeta.sqlCall.replace("$"+param,"?");
                }
            }
            else
            {
                PreparedStatement preparedStatement= this.conn.prepareStatement(spMeta.sqlCall);
                ParameterMetaData metaList = preparedStatement.getParameterMetaData();
                for (int i = 1;i<=metaList.getParameterCount();i++)
                {
                    SPParameterInfo parameter = new SPParameterInfo();
                    parameter.parameterName ="?";
                    parameter.parameterType = ParameterTypeInfo.procedureColumnIn;
                    parameter.sqlTypes = metaList.getParameterType(i);
                    spMeta.parameterList.add(parameter);
                }
            }
            CachedSPMeta.put(key,spMeta);
        }
        return CachedSPMeta.get(key);
    }

    public synchronized static SPInfo findSPMetaInfo(String connectionString,String spName,CommandTypeInfo commandType) throws SQLException, ClassNotFoundException, IOException, NamingException, NoSuchMethodException, IllegalAccessException, InvocationTargetException {
        //double CachedSPMeta for reduce DB connection
        String key = ("findSPMetaInfo_" + connectionString + "_" + spName).toLowerCase();
        if(!CachedSPMeta.containsKey(key))
        {
            try(DBHelper helper = new DBHelper(connectionString)){
                switch (commandType)
                {
                    case TSQL:
                        CachedSPMeta.put(key,helper.findTSQLInfo(spName));
                        break;
                    case StoredProcedure:
                        CachedSPMeta.put(key,helper.findSPInfo(spName));
                        break;
                }

            }
        }
        return  CachedSPMeta.get(key);
    }

    /**
     * Search TSQL script find parameter with following format $param, it will find $name and $size
     *     SELECT * from world.city where name=$name limit 0,$size"
     * @param content
     * @return
     * @throws NoSuchMethodException
     * @throws IllegalAccessException
     * @throws InvocationTargetException
     */
    public static ArrayList<String> findTSQLParameters(String content) throws NoSuchMethodException, IllegalAccessException, InvocationTargetException {
        Pattern pattern = Pattern.compile("(\\$[^\",;\\) ]*)", Pattern.CASE_INSENSITIVE);
        Matcher matcher = pattern.matcher(content);
        ArrayList<String> list = new ArrayList<>();
        while (matcher.find()) {
            String param = matcher.group();
            list.add(param.substring(1));
        }
        return list;
    }

    //endregion

    public static TableInfo findResultsColumnInfo(ResultSetMetaData rsMeta) throws SQLException {
        int numberOfColumns = rsMeta.getColumnCount();
        TableInfo tableMeta = new TableInfo();
        for (int i = 1; i <= numberOfColumns; i++) {
            ColumnInfo columnMeta = new ColumnInfo();
            //use label name instead of columnName if specific label.
            String label = rsMeta.getColumnLabel(i);
            columnMeta.columnName = StringExtension.IsNullOrEmpty(label)?rsMeta.getColumnName(i):label;
            columnMeta.javaClassName =rsMeta.getColumnClassName(i);
            columnMeta.sqlTypeName = rsMeta.getColumnTypeName(i);
            columnMeta.sqlType = rsMeta.getColumnType(i);
            tableMeta.columnList.add(columnMeta);
        }
        return tableMeta;
    }
    public synchronized static HashMap<String,FieldInfo> findClassFieldInfo(Class<?> type) throws Exception {
        String classKey = "$DBHelper$FindClassFieldInfo".concat(type.getName());
        return StaticMemoryCacheHelper.AddNeverExpiredMemoryCache(classKey, new Operation<HashMap<String, FieldInfo>>(type) {
            @Override
            public HashMap<String, FieldInfo> call() throws Exception {
                HashMap<String, FieldInfo> fieldList = new HashMap<>();
                Class<?> runType = (Class<?>) this.params[0];
                Debugger.LogFactory.trace(runType.getName());
                for (Field field : runType.getDeclaredFields()) {
                    //ignore directly;
                    IgnoreDBColumnAttribute ignoreAttribute = field.getAnnotation(IgnoreDBColumnAttribute.class);
                    if (ignoreAttribute != null) {
                        continue;
                    }
                    DBColumnAttribute columnAttribute = field.getAnnotation(DBColumnAttribute.class);
                    FieldInfo fieldMeta = new FieldInfo();
                    fieldMeta.field = field;
                    if (columnAttribute != null) {
                        String columnName = columnAttribute.columnName().length() == 0 ? field.getName() : columnAttribute.columnName();
                        if (columnAttribute.isEntity()) {
                            //find sub field, save into Memory
                            findClassFieldInfo(field.getType());
                        }
                        fieldMeta.columnAttribute = columnAttribute;
                        fieldList.put(columnName.toLowerCase(), fieldMeta);
                    } else {
                        fieldList.put(field.getName().toLowerCase(), fieldMeta);
                    }
                }
                return fieldList;
            }
        });
    }

    public static Map<String,Object>  ReadOutputParameterList(SPInfo spMetaInfo, CallableStatement callableStatement) throws SQLException
    {
        Map<String,Object> outputParameterList = new HashMap<>();
        if(spMetaInfo.parameterList!=null && spMetaInfo.parameterList.size()>0)
        {
            for (SPParameterInfo param:spMetaInfo.parameterList){
                if(param.parameterType == ParameterTypeInfo.procedureColumnOut
                        ||   param.parameterType == ParameterTypeInfo.procedureColumnInOut
                        ||   param.parameterType == ParameterTypeInfo.procedureColumnReturn
                        )
                {
                    outputParameterList.put(buildParamKey(param.parameterName), callableStatement.getObject(param.parameterName));
                }
            }
        }
        return outputParameterList;
    }
    public Map<String,Object> getOutputParameterList() throws SQLException {
        return ReadOutputParameterList(this.SPMetaInfo,(CallableStatement)this.statement);
    }
}





