package LinkFuture.Core.ContentManager.ContentResource;

import LinkFuture.Core.ContentManager.ContentEvent.IContentEvent;
import LinkFuture.Core.ContentManager.ContentSelector.ContentSelectorController;
import LinkFuture.Core.ContentManager.Model.*;
import LinkFuture.Core.MemoryManager.MemoryFactory;
import LinkFuture.Core.MemoryManager.Meta.MemoryCacheMetaInfo;
import LinkFuture.Core.MemoryManager.StaticMemoryCache.StaticMemoryCacheHelper;
import LinkFuture.Core.OperationManager.Operation;
import LinkFuture.Init.Config;
import LinkFuture.Init.Debugger;
import LinkFuture.Init.Extensions.StringExtension;


/**
 * User: Cyokin Zhang
 * Date: 10/12/13
 * Time: 8:14 PM
 */
public abstract class ContentBaseResource implements IContentResource
 {
     public ResourceInfo CurrentResource;
     private String resourceIdentity;
     public ContentBaseResource(ResourceInfo currentResource)
     {
         this.CurrentResource = currentResource;
     }

     public abstract ContentResultInfo RetrieveResource(ContentItemInfo content, ContentParameterCollectionInfo parameters) throws Exception;

     public String GerResourceIdentity(ContentItemInfo content, ContentParameterCollectionInfo parameters) throws Exception {
         if(StringExtension.IsNullOrEmpty(resourceIdentity))
         {
             this.resourceIdentity = BuildResourceIdentity(content,parameters).replaceAll("\n", Config.Empty);
             Debugger.LogFactory.trace("Generated ResourceIdentity ContentBaseResource {} => {}",this.CurrentResource.Name, this.resourceIdentity );
         }
         return this.resourceIdentity;
     }

     public abstract String BuildResourceIdentity(ContentItemInfo content, ContentParameterCollectionInfo parameters) throws Exception;
     public abstract void Verify(ResourceInfo currentResource) throws Exception;

     public ContentResultInfo GerResource(ContentItemInfo content, ContentParameterCollectionInfo parameters) throws Exception {
         AddInfo add = null;
         if(content!=null)
         {
             add = content.FindResourceAddInfo(this.CurrentResource.Name);
         }
         ContentResultInfo result=null;
         long startTime = System.currentTimeMillis();
         try
         {
             //must build selector before cache logic
             ContentSelectorController.BuilderSelector(add, parameters,AddCycleType.Before);
             MemoryCacheMetaInfo meta = MemoryFactory.FindMemoryCache(this.CurrentResource.CacheSettingKey);
             Object[] passArgs= new Object[] {content,parameters,add};
             if(meta==null || !meta.Enable)
             {
                 result = Callback(passArgs);
             }
             else
             {
                 meta.Key  =this.GerResourceIdentity(content, parameters);
                 meta.Action = new Operation<Object>(passArgs) {
                     @Override
                     public Object call() throws Exception {
                         return Callback(this.params);
                     }
                 };
                 result = StaticMemoryCacheHelper.AddMemoryCache(meta);
             }

         }
         catch (Exception e)
         {
             result = new ContentResultInfo();
             result.Name  = this.CurrentResource.Name;
             result.Success = false;
             result.Error = e;
             OnCompletedResource(add,parameters,result);
         }
         finally {
             if(result!=null)
             {
                 result.LoadTimeInMs = System.currentTimeMillis() - startTime;
                 if(result.Success)
                 {
                     return result;
                 }
                 //fail
                 if(add!=null && add.IgnoreException)
                 {
                     return result;
                 }
                 else
                 {
                     throw result.Error;
                 }
             }
             throw new RuntimeException(String.format("Unexpected error on retrieve resource: %s",this.CurrentResource.Name));
         }
     }

     public ContentResultInfo Callback(Object... args) throws Exception {
         ContentItemInfo content = (ContentItemInfo)args[0];
         ContentParameterCollectionInfo parameters = (ContentParameterCollectionInfo)args[1];
         AddInfo add = (AddInfo)args[2];
         ContentResultInfo result=null;
         Debugger.LogFactory.trace("Retrieve Resource: {}", this.CurrentResource.Name);
         result = RetrieveResource(content, parameters);
         result.Name = this.CurrentResource.Name;
         OnCompletedResource(add,parameters,result);
         if(result.Success)
         {
             return result;
         }
         else
         {
             //so no cache
             throw result.Error;
         }
     }

     private void OnCompletedResource(AddInfo add,ContentParameterCollectionInfo parameters,ContentResultInfo result) throws Exception {
         if(add!=null)
         {
             IContentEvent event = add.getOnCompletedEvent();
             if(event!=null)
             {
                 Object[] context = new Object[]{result,this};
                 event.OnTrigger(add,context,parameters);
             }
         }
     }

     @Override
     public boolean MatchCondition(ContentItemInfo content, ContentParameterCollectionInfo parameters) {
         AddInfo add;
         if(content!=null)
         {
             add = content.FindResourceAddInfo(this.CurrentResource.Name);
             if(add!=null && !StringExtension.IsNullOrEmpty(add.Condition))
             {
                 Debugger.LogFactory.trace("MatchCondition:{}",add.Condition);
                 return LinkFuture.Core.Utility.stringExpression(parameters, add.Condition);
             }
         }
         return true;
     }
 }