package net.leadware.velocity.maven.plugin.mojo;

/*
 * #%L
 * Velocity Maven Plugin Mojo
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2013 - 2016 Leadware
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.nio.charset.Charset;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;

import org.apache.maven.model.FileSet;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader;
import org.apache.velocity.runtime.resource.loader.FileResourceLoader;
import org.codehaus.plexus.util.FileUtils;

/**
 * Classe representant le MOJO de resolution de template Velocity 
 * @author <a href="mailto:jetune@leadware.net">Jean-Jacques ETUNE NGI (Leadware Enterprise Architect)</a>
 * @since 17 févr. 2016 - 22:50:36
 */
@Mojo(name = "velocity", 
	  defaultPhase = LifecyclePhase.GENERATE_TEST_RESOURCES, 
	  threadSafe = true, 
	  requiresProject = true, 
	  requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME)
public class VelocityMojo extends AbstractMojo {

	/**
	 * Projet Maven en cours
	 */
	@Component
	private MavenProject project;
	
	/**
	 * Encodage
	 */
	@Parameter(required = false, defaultValue = "")
	private String encoding;

	/**
	 * Repertoire de sortie
	 */
	@Parameter(required = true)
	private File outputDirectory;
	
	/**
	 * Ensemble de fichiers template
	 */
	@Parameter(required = true)
	private FileSet templateFiles;
	
	/**
	 * Valeurs de substitution
	 */
	@Parameter(required = true)
	private Properties templateValues;
	
	
	/* (non-Javadoc)
	 * @see org.apache.maven.plugin.Mojo#execute()
	 */
	@Override
	public void execute() throws MojoExecutionException, MojoFailureException {
		
		// Un Log
		getLog().info("Demarrage de la cible velocity....");
		
		// Ensemble de caracteres
		Charset characterSet = null;
		
		// Si l'encodage est null
		if (encoding == null) {
			
			// Positionnement de l'encodage projet
            encoding = project.getProperties().getProperty("project.build.sourceEncoding");
            
            // Si l'encodage projet est null
            if (encoding == null) {
            	
            	// Un Warning
                getLog().warn("Utilisation de l'ensemble de caracteres par defaut");
                
                // Construction du charset par defaut
                characterSet = Charset.defaultCharset();
                
            } else {
            	
            	// Construction de l'ensemble de caracteres base sur l'encodage
                characterSet = Charset.forName(encoding);
            }
        }
		
		// Affichage de l'ensemble de caracteres a utiliser
		getLog().debug("Charset utilisé" + characterSet.displayName());
		
		// Moteur Velocity
		VelocityEngine engine = new VelocityEngine();

		// Positionnement du LOG System
		engine.setProperty(Velocity.RUNTIME_LOG_LOGSYSTEM, new LogHandler(this));
		
		// Proprietes du moteur
		Properties engineConfig = new Properties();
		
        // Positionnement du chargeur de ressource
        engineConfig.setProperty(Velocity.RESOURCE_LOADER, "classpath,file");
        
		// Positionnement de la classe du chargeur de ressources
        engineConfig.setProperty("classpath.resource.loader.class", ClasspathResourceLoader.class.getName());
        
        // Positionnement de la classe du chargeur de ressources de type file
        engineConfig.setProperty("file.resource.loader.class", FileResourceLoader.class.getName());
        
        // Positionnement du chemin du chargeur de ressource File
        engineConfig.setProperty(Velocity.FILE_RESOURCE_LOADER_PATH, project.getBasedir().getAbsolutePath());
		
        // Initialisation du moteur
        engine.init(engineConfig);
        
        // Contexte velocity
        VelocityContext context = new VelocityContext();
        
        // Ajout du contexte projet
        context.put("project", project);
        
        // Ajout du contexte proprietes systeme
        context.put("system", System.getProperties());
        
        // Ajout du contexte Proprietes Environnementales
        context.put("env", System.getenv());
        
        // Ajout des proprietes au contexte
        addPropertiesToContext(context, templateValues);
        
        try {
        	
        	// Obtention des noms de fichiers en entree
			List<?> fileNames = expandFileSet();
			
			// Si la liste des fichiers est vide
			if(fileNames == null || fileNames.isEmpty()) {
				
				// Un log
				getLog().warn("Aucun fichier a traiter");
				
			} else {
				
				// Traitement des fichiers
				getLog().debug("Traitement des fichiers");
				
				// Parcours
				for (Object object : fileNames) {
					
					// On caste
					String templateFile = (String) object;
					
					// Un log
					getLog().debug( "Template -> " + templateFile );
					
					// Transformation
					translateFile(engine, templateFiles.getDirectory(), templateFile, context);
				}
			}
			
		} catch (Exception e) {
			
			// On relance
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}
	
	/**
	 * Methode permettant de rajouter des proprietes au contexte 
	 * @param context	Contexte
	 * @param properties	Proprietes a rajouter
	 */
	private void addPropertiesToContext(VelocityContext context, Properties properties) {
		
		// Un log
		getLog().debug("Importation des proprietes utilisateur dans le contexte" + properties);
		
		// Si les proprirtes sont nulle
		if(properties == null || properties.isEmpty()) return;
		
		// Enumeration des noms de propeietes
		Enumeration< ? > propEnumeration = properties.propertyNames();
		
		// Parcours de l'enumeration
		while (propEnumeration.hasMoreElements()) {
			
			// Obtention du nom courant
			String key = (String) propEnumeration.nextElement();
			
			// Obtention de la valeur courante
			String value = properties.getProperty(key, "").trim();
			
			// Un log
			getLog().debug(key + "=" + value);
			
			// Ajout dans le contexte s'il ne contient pas deja cette cle
			if(!context.containsKey(key)) context.put(key, value);
		}
	}
	
	/**
	 * Methode permettant d'obtenir la liste des fichiers en entree 
	 * @return	Liste des fichiers en entree
	 * @throws IOException	Exception potentielle
	 */
	private List< ? > expandFileSet() throws IOException {
		
		// Repertoire de base
		File baseDir = new File(project.getBasedir().getAbsolutePath() + File.separator + templateFiles.getDirectory());
		
		// Affichage du repertoire de base
		getLog().debug(baseDir.getAbsolutePath());
		
		// Chargement du pattern des fichiers a inclure
		String includes = listToCsvString(templateFiles.getIncludes());
		
		// Log du pattern include
		getLog().debug("Pattern Includes: " + includes);
		
		String excludes = listToCsvString(templateFiles.getExcludes());
		
		// Log du pattern Exclude
		getLog().debug("excludes: " + excludes);
		
		// Obtention de la liste de fichiers
		return FileUtils.getFileNames(baseDir, includes, excludes, false);
	}
	
	/**
	 * Methode permettant de charger transformer la liste de patter de fichiers en chaine CSV
	 * @param patterns	Liste de pattern
	 * @return	Chaine au format CSV
	 */
	private String listToCsvString(List< ? > patterns) {
		
		// Delimiteur
		String delim = "";
		
		// Buffer
		StringBuffer buf = new StringBuffer();
		
		// Si la liste des pattern est non vide
		if (patterns != null) {
			
			// Parcours
			for (Object object : patterns) {
				
				// Ajout du pattern
				buf.append(delim).append(object);
				
				// Positionnement du delimiteur
				delim = ", ";
			}
		}
		
		// On retourne la chaine
		return buf.toString();
	}
	
	/**
	 * Methode permettant de transformer un template en resolvant ses variables 
	 * @param basedir	Repertoire de base
	 * @param templateFile	Fichier template
	 * @param context	Contexte Velocity
	 * @throws Exception Exception potentielle	
	 */
	private void translateFile(VelocityEngine engine, String basedir, String templateFile,
	        VelocityContext context) throws Exception {
		
		// Template Velocity
		Template template = null;
		
		// Fichier en entree
		String inputFile = basedir + File.separator + templateFile;
		
		// Un log
		getLog().debug( "#translateFile - Fichier Input -> " + inputFile );
		
		try {
			
			// Construction du template
			template = engine.getTemplate(inputFile, encoding == null ? "UTF-8" : encoding);
			
		} catch (Exception e) {
			
			// Un log
			getLog().info("Erreur de charement du template : " + inputFile);
			
			// On leve une exception
			throw new MojoExecutionException("Erreur de charement du template : " + inputFile, e);
		}
		
		// Writer de chaine
		StringWriter sw = new StringWriter();
		
		try {
			
			// Resolution des variables
			template.merge(context, sw);
			
		} catch (Exception e) {
			
			// Un log
			getLog().info("Erreur lors de la fusion du template et des variables" + inputFile + ":" + e.getMessage());
			
			// On leve une exception
			throw new MojoExecutionException("Erreur lors de la fusion du template et des variables" + inputFile, e);

		}
		
		// Fichier resultat
		File result = new File(outputDirectory.getAbsoluteFile() + File.separator + templateFile);
		
		// Repertoire parent du fichier resultat
		File dir = result.getParentFile();
		
		// Si le repertoire parent n'existe pas
		if (!dir.exists()) {
			
			// Si la tentative de creation du repertoire parent echoue
			if (!dir.mkdirs()) {
				
				// On leve un exception
				throw new MojoExecutionException("Echec lors de la creation du repertoire de dortie");
			}
		}
		
		// Stream de sortie
		FileOutputStream os = new FileOutputStream(result);
		
		try {
			
			// Ecriture du resultat dans le stream de sortie
			os.write(sw.toString().getBytes(encoding == null ? "UTF-8" : encoding));
			
		} finally {
			
			// fermeture du stream
			os.close();
		}
	}
}
