/*
 * Decompiled with CFR 0.152.
 */
package net.kafujo.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;
import net.kafujo.base.RequirementException;
import net.kafujo.config.KafuLog;
import net.kafujo.config.SystemProperty;
import net.kafujo.text.KafuText;
import net.kafujo.units.DataSize;
import net.kafujo.units.KafuDateTime;
import net.kafujo.units.KafuNumber;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KafuFile {
    private static final Logger lgr = LoggerFactory.getLogger(KafuFile.class);
    public static final char UNIX_SEPARATOR = '/';
    public static final char WINDOWS_SEPARATOR = '\\';
    public static final Pattern VALID_FILENAME_CHARS = Pattern.compile("[^a-zA-Z0-9._\\-$]+");
    static final String KAFUJO_TEMP_PREFIX = "kafujo_";
    private static final String KAFUJO_TEMP_SUFFIX_DEFAULT = ".tmp";

    private KafuFile() {
    }

    public static boolean exists(String path) {
        if (path == null) {
            return false;
        }
        return Files.exists(Paths.get(path, new String[0]), new LinkOption[0]);
    }

    public static Path requireExists(Path path) {
        Objects.requireNonNull(path, "REQUIRE path");
        if (!Files.exists(path, new LinkOption[0])) {
            throw new RequirementException("PATH DOES NOT EXIST: " + path);
        }
        return path;
    }

    public static Path requireExists(String path) {
        Objects.requireNonNull(path, "REQUIRE path string");
        return KafuFile.requireExists(Paths.get(path, new String[0]));
    }

    public static Path requireNotExists(Path path) {
        Objects.requireNonNull(path, "REQUIRE path");
        if (Files.exists(path, new LinkOption[0])) {
            if (Files.isDirectory(path, new LinkOption[0])) {
                throw new RequirementException("EXISTS AS DIRECTORY: " + path);
            }
            throw new RequirementException("EXISTS AS FILE: " + path);
        }
        return path;
    }

    public static Path requireNotExists(String path) {
        Objects.requireNonNull(path, "REQUIRE path string");
        return KafuFile.requireNotExists(Paths.get(path, new String[0]));
    }

    public static Path requireReadable(Path path) {
        KafuFile.requireExists(path);
        if (!Files.isReadable(path)) {
            throw new RequirementException("PATH NOT READABLE: " + path);
        }
        return path;
    }

    public static Path requireDirectory(Path path) {
        KafuFile.requireExists(path);
        if (!Files.isDirectory(path, new LinkOption[0])) {
            throw new RequirementException("IS NOT A DIRECTORY: " + path);
        }
        return path;
    }

    public static Path requireNotDirectory(Path path) {
        KafuFile.requireExists(path);
        if (Files.isDirectory(path, new LinkOption[0])) {
            throw new RequirementException("IS A DIRECTORY: " + path);
        }
        return path;
    }

    public static Path requireExecutable(Path path) {
        KafuFile.requireExists(path);
        if (!Files.isExecutable(path)) {
            throw new RequirementException("IS NOT EXECUTABLE: " + path);
        }
        return path;
    }

    public static Path requireWritable(Path path) {
        KafuFile.requireExists(path);
        if (!Files.isWritable(path)) {
            throw new RequirementException("IS NOT WRITABLE: " + path);
        }
        return path;
    }

    public static Path requireWritableEmptyFile(Path path) {
        KafuFile.requireNotDirectory(path);
        KafuFile.requireWritable(path);
        DataSize size = KafuFile.sizeOfFile(path);
        if (!size.isZero()) {
            throw new RequirementException("IS NOT EMPTY: " + path + "; SIZE=" + size);
        }
        return path;
    }

    public static Path requireWritableEmptyDirectory(Path directory) {
        KafuFile.requireDirectory(directory);
        KafuFile.requireWritable(directory);
        long count = KafuFile.countRootEntries(directory);
        if (count != 0L) {
            throw new RequirementException("DIRECTORY NOT EMPTY: " + directory + "; CONTAINS " + count + " ENTRIES");
        }
        return directory;
    }

    public static Path createFileOrUseEmpty(Path path) {
        if (Files.exists(path, new LinkOption[0])) {
            KafuFile.requireWritableEmptyFile(path);
            lgr.debug("going to use EXISTING, empty file: {}", (Object)path);
        } else {
            KafuFile.createFile(path);
            lgr.debug("created NEW file: {}", (Object)path);
        }
        return path;
    }

    public static Path createFile(Path file) {
        KafuFile.requireNotExists(file);
        try {
            return Files.createFile(file, new FileAttribute[0]);
        }
        catch (IOException io) {
            throw new UncheckedIOException("Problem creating file", io);
        }
    }

    public static DataSize createFile(Path file, byte[] content) {
        Objects.requireNonNull(content, "REQUIRE content");
        KafuFile.requireNotExists(file);
        try {
            Files.write(file, content, new OpenOption[0]);
        }
        catch (IOException io) {
            throw new UncheckedIOException(io);
        }
        return DataSize.of(content.length);
    }

    public static Path createTempFile() {
        try {
            return Files.createTempFile(KAFUJO_TEMP_PREFIX, KAFUJO_TEMP_SUFFIX_DEFAULT, new FileAttribute[0]);
        }
        catch (IOException io) {
            throw new UncheckedIOException("Problem creating temp file", io);
        }
    }

    public static Path createTempFile(InputStream inputStream) {
        Path path;
        block8: {
            InputStream inputStream2 = inputStream;
            try {
                Path p = KafuFile.createTempFile();
                Files.copy(inputStream, p, StandardCopyOption.REPLACE_EXISTING);
                path = p;
                if (inputStream2 == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (inputStream2 != null) {
                        try {
                            inputStream2.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException io) {
                    throw new UncheckedIOException("Problem writing InputStream to temp file", io);
                }
            }
            inputStream2.close();
        }
        return path;
    }

    public static Path createTempFile(CharSequence content) {
        try {
            Path p = Files.createTempFile(KAFUJO_TEMP_PREFIX, ".txt", new FileAttribute[0]);
            Files.writeString(p, content, StandardOpenOption.TRUNCATE_EXISTING);
            return p;
        }
        catch (IOException io) {
            throw new UncheckedIOException("Problem creating temp file of CharSequence", io);
        }
    }

    public static Path createTempFile(Iterable<? extends CharSequence> content) {
        try {
            Path p = Files.createTempFile(KAFUJO_TEMP_PREFIX, "txt", new FileAttribute[0]);
            Files.write(p, content, StandardOpenOption.TRUNCATE_EXISTING);
            return p;
        }
        catch (IOException io) {
            throw new UncheckedIOException("Problem creating temp file of lines", io);
        }
    }

    public static Path createTempFile(byte[] bytes) {
        try {
            Path p = Files.createTempFile(KAFUJO_TEMP_PREFIX, "bin", new FileAttribute[0]);
            Files.write(p, bytes, StandardOpenOption.TRUNCATE_EXISTING);
            return p;
        }
        catch (IOException io) {
            throw new UncheckedIOException("Problem writing bytes to temp file", io);
        }
    }

    public static Path createTempDirectory() {
        try {
            return Files.createTempDirectory(KAFUJO_TEMP_PREFIX, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static boolean deleteFileQuietly(Path path) {
        if (!Files.exists(path = path.toAbsolutePath(), new LinkOption[0])) {
            lgr.info("quietly did nothing, path didn't exist {}", (Object)path);
            return false;
        }
        KafuFile.requireNotDirectory(path);
        try {
            Files.delete(path);
            lgr.info("quietly deleted {}", (Object)path);
            return true;
        }
        catch (IOException | RuntimeException ioAndRt) {
            if (lgr.isDebugEnabled()) {
                lgr.debug("QUIETLY FAILED TO DELETE FILE " + path, ioAndRt);
            } else {
                lgr.info("QUIETLY FAILED TO DELETE FILE {} cause {}", (Object)path, (Object)ioAndRt.toString());
            }
            return false;
        }
    }

    public static boolean deleteQuietly(Path ... paths) {
        int failCount = 0;
        for (Path path : paths) {
            if (KafuFile.deleteQuietly(path)) continue;
            ++failCount;
        }
        return failCount == 0;
    }

    public static boolean deleteFileQuietly(String path) {
        Objects.requireNonNull(path, "REQUIRE path");
        return KafuFile.deleteFileQuietly(Paths.get(path, new String[0]));
    }

    public static int deleteDirectory(Path dir) {
        KafuFile.requireDirectory(dir);
        var visitor = new SimpleFileVisitor<Path>(){
            int count = 0;

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                ++this.count;
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                Files.delete(dir);
                ++this.count;
                return FileVisitResult.CONTINUE;
            }
        };
        try {
            Files.walkFileTree(dir, visitor);
        }
        catch (IOException io) {
            throw new UncheckedIOException(io);
        }
        return visitor.count;
    }

    public static boolean deleteDirectoryQuietly(Path path) {
        if (!Files.exists(path = path.toAbsolutePath(), new LinkOption[0])) {
            lgr.info("quietly did nothing, path didn't exist {}", (Object)path);
            return false;
        }
        KafuFile.requireDirectory(path);
        try {
            int count = KafuFile.deleteDirectory(path);
            lgr.info("quietly deleted {} files from {}", (Object)count, (Object)path);
            return true;
        }
        catch (RuntimeException anyRt) {
            KafuLog.logThrowableOnDebug(lgr, "QUIETLY FAILED TO DELETE DIRECTORY " + path, anyRt);
            return false;
        }
    }

    public static boolean deleteDirectoryQuietly(String path) {
        Objects.requireNonNull(path, "REQUIRE path");
        return KafuFile.deleteDirectoryQuietly(Paths.get(path, new String[0]));
    }

    public static boolean deleteQuietly(Path path) {
        Objects.requireNonNull(path, "REQUIRES: path to be deleted");
        if (!Files.exists(path, new LinkOption[0])) {
            lgr.info("quietly did nothing, path didn't exist {}", (Object)path.toAbsolutePath());
            return false;
        }
        if (Files.isDirectory(path, new LinkOption[0])) {
            return KafuFile.deleteDirectoryQuietly(path);
        }
        return KafuFile.deleteFileQuietly(path);
    }

    public static int deleteQuietly(List<Path> paths) {
        Objects.requireNonNull(paths, "REQUIRES: paths to be deleted");
        int count = 0;
        for (Path path : paths) {
            if (!KafuFile.deleteQuietly(path)) continue;
            ++count;
        }
        return count;
    }

    public static boolean deleteQuietly(String path) {
        Objects.requireNonNull(path, "REQUIRE path");
        return KafuFile.deleteFileQuietly(Paths.get(path, new String[0]));
    }

    public static Path getTempDir() {
        Path tmp = SystemProperty.JAVA_IO_TMPDIR.asPath().toAbsolutePath();
        KafuFile.requireDirectory(tmp);
        KafuFile.requireExecutable(tmp);
        KafuFile.requireWritable(tmp);
        return tmp;
    }

    public static Path getUserHome() {
        Path home = SystemProperty.USER_HOME.asPath().toAbsolutePath();
        KafuFile.requireDirectory(home);
        return home;
    }

    public static Path getUserDir() {
        Path home = SystemProperty.USER_DIR.asPath().toAbsolutePath();
        KafuFile.requireDirectory(home);
        return home;
    }

    public static Path getWorkingDir() {
        Path working = Paths.get(".", new String[0]).toAbsolutePath().normalize();
        KafuFile.requireDirectory(working);
        return working;
    }

    public static FileStore getFileStore(Path path) {
        KafuFile.requireExists(path);
        try {
            return Files.getFileStore(path.toRealPath(new LinkOption[0]));
        }
        catch (IOException io) {
            throw new UncheckedIOException("KafuFile.getFileSore failed: " + io.getMessage(), io);
        }
    }

    public static DataSize getUsableSpace(Path path) {
        try {
            long bytes = KafuFile.getFileStore(path).getUsableSpace();
            return DataSize.of(bytes);
        }
        catch (IOException io) {
            throw new UncheckedIOException(io);
        }
    }

    public static DataSize getUsableSpace(String path) {
        return KafuFile.getUsableSpace(Paths.get(path, new String[0]));
    }

    public static String getUsableSpace(Path path, String def) {
        Objects.requireNonNull(path, "REQUIRE path");
        if (!Files.exists(path, new LinkOption[0])) {
            return def;
        }
        return KafuFile.getUsableSpace(path).toString();
    }

    public static boolean isEmptyDirectory(Path directory) {
        boolean bl;
        block8: {
            KafuFile.requireDirectory(directory);
            DirectoryStream<Path> dirStream = Files.newDirectoryStream(directory);
            try {
                boolean bl2 = bl = !dirStream.iterator().hasNext();
                if (dirStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (dirStream != null) {
                        try {
                            dirStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException io) {
                    throw new UncheckedIOException("Problem checking directory for emptiness", io);
                }
            }
            dirStream.close();
        }
        return bl;
    }

    public static boolean isEmptyDirectory(String directory) {
        Objects.requireNonNull(directory, "REQUIRE directory as String");
        return KafuFile.isEmptyDirectory(Paths.get(directory, new String[0]));
    }

    public static boolean isEmptyFile(Path file) {
        return KafuFile.sizeOfFile(file).equals(DataSize.ZERO);
    }

    public static boolean isEmptyFile(String file) {
        Objects.requireNonNull(file, "REQUIRE file as String");
        return KafuFile.isEmptyFile(Paths.get(file, new String[0]));
    }

    public static boolean isEmpty(Path fileOrDirectory) {
        Objects.requireNonNull(fileOrDirectory, "REQUIRE file or directory");
        if (Files.isDirectory(fileOrDirectory, new LinkOption[0])) {
            return KafuFile.isEmptyDirectory(fileOrDirectory);
        }
        return KafuFile.isEmptyFile(fileOrDirectory);
    }

    public static boolean isEmpty(String fileOrDirectory) {
        Objects.requireNonNull(fileOrDirectory, "REQUIRE file or directory as String");
        return KafuFile.isEmpty(Paths.get(fileOrDirectory, new String[0]));
    }

    public static long countRootEntries(Path directory) {
        KafuFile.requireDirectory(directory);
        try {
            return Files.list(directory).count();
        }
        catch (IOException io) {
            throw new UncheckedIOException("Problem counting elements in directory", io);
        }
    }

    public static long countAllEntries(Path directory) {
        long l;
        block10: {
            KafuFile.requireDirectory(directory);
            DirectoryStream<Path> dirStream = Files.newDirectoryStream(directory);
            try {
                long count = 0L;
                for (Path path : dirStream) {
                    if (Files.isDirectory(path, new LinkOption[0])) {
                        count += KafuFile.countAllEntries(path);
                        continue;
                    }
                    ++count;
                }
                l = count;
                if (dirStream == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (dirStream != null) {
                        try {
                            dirStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException io) {
                    throw new UncheckedIOException("Problem counting elements in directory", io);
                }
            }
            dirStream.close();
        }
        return l;
    }

    public static long summateBytesOfDiectory(Path directory) {
        long l;
        block10: {
            Objects.requireNonNull(directory, "REQUIRE directory");
            KafuFile.requireDirectory(directory);
            DirectoryStream<Path> dirStream = Files.newDirectoryStream(directory);
            try {
                long size = 0L;
                for (Path fileOrDirectory : dirStream) {
                    if (Files.isDirectory(fileOrDirectory, new LinkOption[0])) {
                        size += KafuFile.summateBytesOfDiectory(fileOrDirectory);
                        continue;
                    }
                    size += Files.size(fileOrDirectory);
                }
                l = size;
                if (dirStream == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (dirStream != null) {
                        try {
                            dirStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException io) {
                    throw new UncheckedIOException("Problem traversing elements in directory", io);
                }
            }
            dirStream.close();
        }
        return l;
    }

    public static DataSize sizeOfDirectory(Path directory) {
        return DataSize.of(KafuFile.summateBytesOfDiectory(directory));
    }

    public static DataSize size(Path path) {
        if (Files.isDirectory(path, new LinkOption[0])) {
            return KafuFile.sizeOfDirectory(path);
        }
        return KafuFile.sizeOfFile(path);
    }

    public static DataSize sizeOfFile(Path file) {
        KafuFile.requireNotDirectory(file);
        try {
            return DataSize.of(Files.size(file));
        }
        catch (IOException io) {
            throw new UncheckedIOException("IO problem fetching size of file " + file.toAbsolutePath(), io);
        }
    }

    public static DataSize sizeOfFile(String file) {
        Objects.requireNonNull(file, "REQUIRE file to be used as path");
        return KafuFile.sizeOfFile(Paths.get(file, new String[0]));
    }

    public static DataSize sizeQuietly(Path fileOrDirectory) {
        KafuFile.requireExists(fileOrDirectory);
        final AtomicLong size = new AtomicLong(0L);
        try {
            Files.walkFileTree(fileOrDirectory, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    size.addAndGet(attrs.size());
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) {
                    KafuLog.logThrowableOnDebug(lgr, "QUIETLY SKIPPED PATH", exc);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                    if (exc != null) {
                        KafuLog.logThrowableOnDebug(lgr, "QUIETLY HAVING TROUBLE TRAVERSING", exc);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            throw new AssertionError((Object)"walkFileTree will not throw IOException if the FileVisitor does not");
        }
        return DataSize.of(size.get());
    }

    public static DataSize sizeQuietly(String fileOrDirectory) {
        Objects.requireNonNull(fileOrDirectory, "REQUIRE fileOrDirectory to be used as path");
        return KafuFile.sizeQuietly(Paths.get(fileOrDirectory, new String[0]));
    }

    public static Map<FileStore, DataSize> getUseableSpace(boolean silent) {
        HashMap<FileStore, DataSize> map;
        block3: {
            map = new HashMap<FileStore, DataSize>();
            try {
                for (FileStore store : FileSystems.getDefault().getFileStores()) {
                    map.put(store, DataSize.of(store.getUsableSpace()));
                }
            }
            catch (IOException io) {
                if (silent) break block3;
                throw new UncheckedIOException(io);
            }
        }
        return map;
    }

    public static String getFileStoreStats(boolean silent) {
        StringBuilder sb = new StringBuilder(String.format("%-25s %-10s %10s %12s %10s%n", "filestore", "type", "total", "useable", "used"));
        try {
            for (FileStore store : FileSystems.getDefault().getFileStores()) {
                long total = store.getTotalSpace();
                long usable = store.getUsableSpace();
                double used = 100.0;
                if (total != 0L) {
                    used = (double)(total - usable) * 100.0 / (double)total;
                }
                sb.append(String.format("%-25s %-10s %10s %12s   %8.2f%% %s%n", store, store.type(), DataSize.toString(total), DataSize.toString(usable), used, store.isReadOnly() ? "READONLY" : ""));
            }
        }
        catch (IOException io) {
            if (silent) {
                sb.append("problem fetching file store information").append(io);
            }
            throw new UncheckedIOException(io);
        }
        return sb.toString();
    }

    public static Path createDirectories(String dir) {
        return KafuFile.createDirectories(Paths.get(dir, new String[0]));
    }

    public static Path createDirectories(Path dir) {
        if (!Files.exists(dir, new LinkOption[0])) {
            try {
                return Files.createDirectories(dir, new FileAttribute[0]);
            }
            catch (IOException io) {
                throw new UncheckedIOException(io);
            }
        }
        return KafuFile.requireDirectory(dir);
    }

    public static Path sanitizeFilename(CharSequence rawName) {
        Objects.requireNonNull(rawName, "REQUIRE filename");
        String out = rawName.toString().strip();
        if (out.isEmpty()) {
            throw new IllegalArgumentException("cannot use blank string to form a sane filename");
        }
        if (out.startsWith("-")) {
            out = out.replaceFirst("-", "_");
        }
        out = KafuText.replaceUmlauts(out);
        out = VALID_FILENAME_CHARS.matcher(out).replaceAll("_");
        return Paths.get(out, new String[0]);
    }

    public static Path sanitizeFilename(CharSequence connector, CharSequence ... filenameFragments) {
        Objects.requireNonNull(filenameFragments, "REQUIRE connector. Use emypty String, if you don't need one");
        Objects.requireNonNull(filenameFragments, "REQUIRE filename");
        if (StringUtils.isAllBlank(filenameFragments)) {
            throw new IllegalArgumentException("All filename fragments are blank - would result in empty file name");
        }
        if (VALID_FILENAME_CHARS.matcher(connector).find()) {
            throw new IllegalArgumentException("Connector contains invalid chars '" + connector + "'; regex rule: " + VALID_FILENAME_CHARS);
        }
        StringBuilder combinedFragments = new StringBuilder();
        int countNonBlanks = 0;
        for (int idx = 0; idx < filenameFragments.length; ++idx) {
            String fragment = filenameFragments[idx].toString().strip();
            if (fragment.isBlank()) continue;
            ++countNonBlanks;
            if (idx == filenameFragments.length - 1 && fragment.startsWith(".")) {
                combinedFragments.append(fragment);
                continue;
            }
            if (countNonBlanks > 1) {
                combinedFragments.append(connector);
            }
            combinedFragments.append(fragment);
        }
        return KafuFile.sanitizeFilename(combinedFragments);
    }

    public static Path timestampFilename(String rawName, ZoneId zone) {
        Objects.requireNonNull(rawName, "REQUIRE filename");
        Object output = rawName.strip();
        if (((String)output).length() == 0) {
            throw new IllegalArgumentException("cannot use an empty filename");
        }
        output = LocalDateTime.now(zone).format(KafuDateTime.FILENAME) + "-" + rawName;
        return KafuFile.sanitizeFilename((CharSequence)output);
    }

    public static Path timestampFilename(String rawName) {
        return KafuFile.timestampFilename(rawName, ZoneId.systemDefault());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean equalContent(Path file1, Path file2) {
        KafuFile.requireNotDirectory(file1);
        KafuFile.requireNotDirectory(file2);
        if (file1.equals(file2)) {
            return true;
        }
        try {
            long size1 = Files.size(file1);
            if (size1 != Files.size(file2)) {
                return false;
            }
            if (size1 < 2048L) {
                return Arrays.equals(Files.readAllBytes(file1), Files.readAllBytes(file2));
            }
        }
        catch (IOException ioTrouble) {
            throw new UncheckedIOException("trouble comparing small files", ioTrouble);
        }
        try (InputStream is1 = Files.newInputStream(file1, new OpenOption[0]);
             InputStream is2 = Files.newInputStream(file2, new OpenOption[0]);){
            int data;
            do {
                if ((data = is1.read()) == -1) return true;
            } while (data == is2.read());
            boolean bl = false;
            return bl;
        }
        catch (IOException ioTrouble) {
            throw new UncheckedIOException("trouble comparing input streams", ioTrouble);
        }
    }

    public static Path createRandomBinaryTempFile(DataSize dataSize) {
        Objects.requireNonNull(dataSize, "REQUIRE size of random binary file");
        long bytes = dataSize.longValue();
        if (bytes < 1L) {
            throw new IllegalArgumentException("data size must be at least one byte");
        }
        int chunkSize = KafuNumber.randomInt(1984, 13742);
        byte[] chunk = new byte[chunkSize];
        long bytesWritten = 0L;
        Path destination = KafuFile.createTempFile();
        try (OutputStream os = Files.newOutputStream(destination, new OpenOption[0]);){
            do {
                long missing;
                if ((missing = bytes - bytesWritten) >= (long)chunkSize) {
                    os.write(KafuNumber.randomBytes(chunk));
                    bytesWritten += (long)chunkSize;
                    continue;
                }
                os.write(KafuNumber.randomBytes((int)missing));
                bytesWritten += missing;
            } while (bytesWritten != bytes);
        }
        catch (IOException ioTrouble) {
            throw new UncheckedIOException("Problem writing random temp file", ioTrouble);
        }
        return destination;
    }

    public static String sizeAndName(Path path) {
        Objects.requireNonNull(path, "REQUIRE path");
        return path.getFileName() + " (" + KafuFile.size(path) + ")";
    }

    public static String sizeAndPath(Path path) {
        Objects.requireNonNull(path, "REQUIRE path");
        return path.toAbsolutePath().normalize() + " (" + KafuFile.size(path) + ")";
    }

    public static boolean isAbsoluteDosPath(String path) {
        Objects.requireNonNull(path, "REQUIRE path to be checked");
        if (path.length() < 2) {
            return false;
        }
        return path.substring(0, 2).matches("[a-zA-Z]:");
    }

    public static String separatorsToUnix(Path path) {
        Objects.requireNonNull(path, "REQUIRE path to transform to unix style");
        return KafuFile.separatorsToUnix(path.toString());
    }

    public static String separatorsToUnix(String path) {
        Objects.requireNonNull(path, "REQUIRE path to transform to unix style");
        if (path.indexOf(92) == -1) {
            return path;
        }
        if (KafuFile.isAbsoluteDosPath(path)) {
            throw new IllegalArgumentException("cannot convert absolute dos path to unix style path");
        }
        return path.replace('\\', '/');
    }
}

