/*
 * Copyright 2009 Markus KARG
 *
 * This file is part of webdav-addressbook.
 *
 * webdav-addressbook is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * webdav-addressbook is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with webdav-addressbook.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.java.dev.webdav.addressbook;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;

/**
 * This implementation of {@link HttpServletResponseWrapper} allows to
 * externally inspect the wrapped response.
 * 
 * @author Markus KARG (mkarg@users.dev.java.net)
 */
public final class ReadableHttpRequestWrapper extends HttpServletResponseWrapper {

	private int statusCode;

	private String statusMessage;

	private Map<String, Collection<String>> headers = new HashMap<String, Collection<String>>();

	public ReadableHttpRequestWrapper(final HttpServletResponse httpServletResponse) {
		super(httpServletResponse);
	}

	@Override
	public final void addDateHeader(final String name, final long date) {
		super.addDateHeader(name, date);

		final String value = new Date(date).toString();

		if (this.containsHeader(name))
			this.headers.get(name).add(value);
		else
			this.headers.put(name, new LinkedList<String>(Collections.singleton(value)));
	}

	@Override
	public final void addHeader(final String name, final String value) {
		super.addHeader(name, value);

		if (this.headers.containsKey(name))
			this.headers.get(name).add(value);
		else
			this.headers.put(name, new LinkedList<String>(Collections.singleton(value)));
	}

	@Override
	public final void addIntHeader(String name, int integer) {
		super.addIntHeader(name, integer);

		final String value = Integer.toString(integer);

		if (this.headers.containsKey(name))
			this.headers.get(name).add(value);
		else
			this.headers.put(name, new LinkedList<String>(Collections.singleton(value)));
	}

	@Override
	public final void setDateHeader(final String name, final long date) {
		super.setDateHeader(name, date);
		this.headers.put(name, new LinkedList<String>(Collections.singleton(new Date(date).toString())));
	}

	@Override
	public final void setHeader(final String name, final String value) {
		super.setHeader(name, value);
		this.headers.put(name, new LinkedList<String>(Collections.singleton(value)));
	}

	@Override
	public final void setIntHeader(final String name, final int value) {
		super.setIntHeader(name, value);
		this.headers.put(name, new LinkedList<String>(Collections.singleton(Integer.toString(value))));
	}

	@Override
	public final void setStatus(final int statusCode, final String statusMessage) {
		super.setStatus(statusCode, statusMessage);
		this.statusCode = statusCode;
		this.statusMessage = statusMessage;
	}

	@Override
	public final void setStatus(final int statusCode) {
		super.setStatus(statusCode);
		this.statusCode = statusCode;
	}

	@Override
	public final void sendError(final int statusCode) throws IOException {
		super.sendError(statusCode);
		this.statusCode = statusCode;
	}

	@Override
	public final void sendError(final int statusCode, final String statusMessage) throws IOException {
		super.sendError(statusCode, statusMessage);
		this.statusCode = statusCode;
		this.statusMessage = statusMessage;
	}

	public final String getStatusMessage() {
		return this.statusMessage;
	}

	public final int getStatusCode() {
		return this.statusCode;
	}

	public final Collection<String> getHeaderNames() {
		return this.headers.keySet();
	}

	public final Collection<String> getHeaders(final String name) {
		return this.headers.get(name);
	}
}