/*
 * Copyright 2009 Markus KARG
 *
 * This file is part of webdav-addressbook.
 *
 * webdav-addressbook is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * webdav-addressbook is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with webdav-addressbook.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.java.dev.webdav.addressbook;

import java.io.IOException;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * This implementation of {@link Filter} logs request and response.
 * 
 * @author Markus KARG (mkarg@users.dev.java.net) *
 */
public final class LoggingFilter implements Filter {

	@Override
	public void init(final FilterConfig config) throws ServletException {
		// Intentionally left blank.
	}

	@Override
	public final void destroy() {
		// Intentionally left blank.
	}

	@Override
	public final void doFilter(final ServletRequest request, final ServletResponse response, final FilterChain chain) throws IOException, ServletException {
		/*
		 * This filter is only able to handle HTTP, so we bypass anything else.
		 */
		if (!(request instanceof HttpServletRequest) || !(response instanceof HttpServletResponse)) {
			chain.doFilter(request, response);
			return;
		}

		this.doFilter((HttpServletRequest) request, (HttpServletResponse) response, chain);
	}

	private final void doFilter(final HttpServletRequest request, final HttpServletResponse response, final FilterChain chain) throws IOException,
			ServletException {
		this.logHttpServletRequest(request);

		final ReadableHttpRequestWrapper wrapper = new ReadableHttpRequestWrapper(response);

		chain.doFilter(request, wrapper);

		this.logHttpServletResponse(wrapper);
	}

	@SuppressWarnings("unchecked")
	private final void logHttpServletRequest(final HttpServletRequest request) {
		System.out.print(String.format("%1$tF %1$tT --> %2$s %3$s", new Date(), request.getMethod(), request.getPathInfo()));

		for (final String headerName : (List<String>) Collections.list(request.getHeaderNames()))
			for (final String headerValue : (List<String>) Collections.list(request.getHeaders(headerName)))
				System.out.print(String.format(" %s=%s", headerName, headerValue));

		System.out.println();
	}

	private final void logHttpServletResponse(final ReadableHttpRequestWrapper response) {
		System.out.print(String.format("%1$tF %1$tT <-- %2$d %3$s", new Date(), response.getStatusCode(), response.getStatusMessage()));

		for (final String headerName : response.getHeaderNames())
			for (final String headerValue : response.getHeaders(headerName))
				System.out.print(String.format(" %s=%s", headerName, headerValue));

		System.out.println();
	}

}