package joo {

/**
 * A class loader that provides special treatment of resource bundle classes.
 * It maintains a current locale and loads a locale-specific subclass of any
 * class ending with <code>_properties</code>.
 * <p>The locale is saved in and retrieved from a Cookie whose name, path, and domain
 * can be configured.</p>
 * <p>In your JavaScript code, after including the Jangaroo Runtime, activate
 * locale-specific resource bundle class loading as follows:</p>
 * <pre>
 * new joo.ResourceBundleAwareClassLoader(["en", "de"]);
 * </pre>
 * <p>You can create resource bundle classes by creating <code>*_<i>locale</i>.properties</code>
 * files below the <code>src/main/joo</code> directory. These files are translated to AS3
 * code by the Jangaroo properties compiler (Maven goal: <code>properties</code>).
 * Each resource bundle must consist of properties files for each supported locale, where
 * the default locale is suppressed.
 * For example for supported locales <code>["en", "de"]</code> (which makes <code>"en"</code>
 * the default locale), you need properties files <code>My.properties</code> and <code>My_de.properties</code>.</p>
 * <p>To change the locale, your application code must call</p>
 * <pre>
 * joo.ResourceBundleAwareClassLoader.INSTANCE.setLocale(newLocale);
 * </pre>
 *
 * @see #ResourceBundleAwareClassLoader
 * @see #getLocale
 * @see #setLocale
 *
 * @author Manuel Ohlendorf
 * @author Frank Wienberg
 */
public class ResourceBundleAwareClassLoader extends DynamicClassLoader {
  /**
   * The ResourceBundleAwareClassLoader singleton, which is created or overwritten
   * by invoking the constructor.
   * @see #ResourceBundleAwareClassLoader
   */
  public static var INSTANCE:ResourceBundleAwareClassLoader;

  /**
   * Set before any resource bundle is loaded.
   * @see #getLocale
   */
  public var localeCookieDomain:String;

  /**
   * Create a new ResourceBundleAwareClassLoader with the given supported locales, default locale,
   * and locale Cookie name, and set it as the default Jangaroo class loader (<code>joo.classLoader</code>)
   * and as the singleton ResourceBundleAwareClassLoader (<code>INSTANCE</code>).
   * @param supportedLocales (default: <code>["en"]</code>) An Array of supported locales (String).
   *   The current locale is guaranteed to be an item of this list.
   *   The first element of this list is used as the default locale to use when all other attempts to determine
   *   a locale fail.
   * @param localeCookieName (default: <code>"joo.locale"</code>) The name of the Cookie to load and store locale
   *   information on the client.
   * @param localeCookiePath (default: <code>window.location.pathname</code>) The path of the Cookie to load and
   *   store locale information on the client.
   * @param localeCookieDomain (default: <code>null</code>) The domain of the Cookie to load and store locale
   *   information on the client.
   * @param preferredLocales the locales preferred by the current user, in order of preference, which will be used
   *   to determine the best supported locale if no locale Cookie is set.
   * @see joo.classLoader
   * @see #INSTANCE
   * @see #getLocale
   */
  public function ResourceBundleAwareClassLoader(supportedLocales:Array = undefined, localeCookieName:String = undefined, localeCookiePath:String = undefined, localeCookieDomain:String = undefined, preferredLocales:Array = undefined) {
    super();
  }

  public native function getSupportedLocales():Array;

  public native function getDefaultLocale():String;

  override protected native function createClassDeclaration(packageDef:String, metadata:Object, classDef:String, inheritanceLevel:int, memberFactory:Function, publicStaticMethodNames:Array, dependencies:Array):JooClassDeclaration;

  /**
   * Used internally by code generated by the properties compiler.
   * In case your want to implement a custom resource bundle, use the following code to
   * generate a locale-specific instance of your bundle class:
   * <pre>
   * public static const INSTANCE:My_properties = My_properties(ResourceBundleAwareClassLoader.INSTANCE.createSingleton(My_properties));
   * </pre>
   * @param resourceBundle the resource bundle class for which the subclass corresponding to the current locale
   *   is to be instantiated.
   * @return Object an instance of the resource bundle class or the subclass corresponding to the current locale
   * @see #getLocale
   */
  public native function createSingleton(resourceBundle:Class):Object;

  /**
   * Sets the current locale to the given locale or, if the given locale is not supported,
   * to the longest match in the supported locales, or the default locale if there is no match.
   * @param newLocale the locale to use for resource bundle class loading
   * @return String the supported locale that has actually been set
   * @see #ResourceBundleAwareClassLoader
   */
  public native function setLocale(newLocale:String):String;

  public native function findSupportedLocale(locale:String):String;

  /**
   * Return the locale currently used for resource bundle class loading.
   * On the first call of this method, the locale is read from the Cookie given by
   * <code>localeCookieName</code> (default "joo.locale"), <code>localeCookiePath</code>
   * (<code>window.location.pathname</code>) and <code>localeCookieDomain</code>
   * (<code>null</code>).
   * If there is no such Cookie, the browser's <code>navigator</code> object is asked for its
   * language. If the locale still is not determined, the <code>defaultLocale</code> is used.
   * This value if stored using <code>setLocale()</code>.
   *
   * @return the locale currently used for resource bundle class loading.
   *
   * @see #ResourceBundleAwareClassLoader
   * @see #setLocale
   */
  public native function getLocale():String;
}
}