/*
 * Copyright 2008 CoreMedia AG
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an "AS
 * IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either 
 * express or implied. See the License for the specific language 
 * governing permissions and limitations under the License.
 */

package net.jangaroo.jooc.ast;

import net.jangaroo.jooc.CompilerError;
import net.jangaroo.jooc.JooSymbol;
import net.jangaroo.jooc.Scope;
import net.jangaroo.jooc.sym;

import java.io.IOException;
import java.util.List;

/**
 * @author Frank Wienberg
 */
public class AnnotationParameter extends NodeImplBase {

  private Ide optName;
  private JooSymbol optSymEq;
  private AstNode value;
  private CompilationUnit compilationUnit;
  private Annotation parentAnnotation;

  public AnnotationParameter(Ide optName, JooSymbol optSymEq, AstNode value) {
    this.optName = optName;
    this.optSymEq = optSymEq;
    this.value = value;
  }

  @Override
  public List<? extends AstNode> getChildren() {
    return makeChildren(super.getChildren(), optName, value);
  }

  @Override
  public void visit(AstVisitor visitor) throws IOException {
    visitor.visitAnnotationParameter(this);
  }

  public Annotation getParentAnnotation() {
    return parentAnnotation;
  }

  public void setParentAnnotation(Annotation parentAnnotation) {
    this.parentAnnotation = parentAnnotation;
  }

  @Override
  public void scope(final Scope scope) {
    if (getValue() != null) {
      // TODO: is value really optional?
      getValue().scope(scope);
      compilationUnit = scope.getCompilationUnit();
    }
  }

  public void analyze(AstNode parentNode) {
    super.analyze(parentNode);
    AstNode value = getValue();
    if (value != null) {
      value.analyze(this);
      if (value instanceof Ide) {
        ((Ide) value).analyzeAsExpr(this, null);
      }
      String metaName = parentAnnotation.getMetaName();
      if ("Embed".equals(metaName) && getOptName() != null && "source".equals(getOptName().getName())) {
        JooSymbol valueSymbol = value.getSymbol();
        if (valueSymbol.sym != sym.STRING_LITERAL) {
          throw new CompilerError(valueSymbol, "The source parameter of an [Embed] annotation must be a string literal");
        }
        String text = valueSymbol.getText();
        String quote = text.substring(0, 1);
        String source = (String) valueSymbol.getJooValue();
        String absoluteSource = compilationUnit.addResourceDependency(source);
        this.value = new LiteralExpr(new JooSymbol(valueSymbol.sym, valueSymbol.getFileName(),
                valueSymbol.getLine(), valueSymbol.getColumn(), valueSymbol.getWhitespace(),
                quote + absoluteSource + quote,
                absoluteSource));
      } else if ("ExtConfig".equals(metaName) && optName != null && "target".equals(optName.getName())) {
        // special case: [ExtConfig(target="...")] must trigger a dependency to the target class!
        Object targetClassName = value.getSymbol().getJooValue();
        if (targetClassName instanceof String) {
          compilationUnit.addDependency((String) targetClassName);
        }
      }
    }
  }

  public JooSymbol getSymbol() {
    return getOptName() == null ? getValue().getSymbol() : getOptName().getSymbol();
  }

  public Ide getOptName() {
    return optName;
  }

  public JooSymbol getOptSymEq() {
    return optSymEq;
  }

  public AstNode getValue() {
    return value;
  }

}
