package net.jahhan.jdbc.mybaitssession;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.ibatis.binding.BindingException;
import org.apache.ibatis.cursor.Cursor;
import org.apache.ibatis.exceptions.PersistenceException;
import org.apache.ibatis.exceptions.TooManyResultsException;
import org.apache.ibatis.executor.BatchResult;
import org.apache.ibatis.executor.ErrorContext;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.executor.result.DefaultMapResultHandler;
import org.apache.ibatis.executor.result.DefaultResultContext;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.apache.ibatis.session.SqlSession;

import net.jahhan.common.extension.constant.JahhanErrorCode;
import net.jahhan.exception.JahhanException;

public class DefaultSqlSessionHelper implements SqlSession {

	private final Configuration configuration;

	private final Executor executor;

	private boolean dirty;

	public DefaultSqlSessionHelper(Configuration configuration, Executor executor) {
		this.configuration = configuration;
		this.executor = executor;
		this.dirty = false;
	}

	@Override
	public <T> T selectOne(String statement) {
		return this.<T> selectOne(statement, null);
	}

	@Override
	public <T> T selectOne(String statement, Object parameter) {
		List<T> list = this.<T> selectList(statement, parameter);
		if (list.size() == 1) {
			return list.get(0);
		} else if (list.size() > 1) {
			throw new TooManyResultsException(
					"Expected one result (or null) to be returned by selectOne(), but found: " + list.size());
		} else {
			return null;
		}
	}

	@Override
	public <K, V> Map<K, V> selectMap(String statement, String mapKey) {
		return this.selectMap(statement, null, mapKey, RowBounds.DEFAULT);
	}

	@Override
	public <K, V> Map<K, V> selectMap(String statement, Object parameter, String mapKey) {
		return this.selectMap(statement, parameter, mapKey, RowBounds.DEFAULT);
	}

	@Override
	public <K, V> Map<K, V> selectMap(String statement, Object parameter, String mapKey, RowBounds rowBounds) {
		final List<?> list = selectList(statement, parameter, rowBounds);
		final DefaultMapResultHandler<K, V> mapResultHandler = new DefaultMapResultHandler<K, V>(mapKey,
				configuration.getObjectFactory(), configuration.getObjectWrapperFactory(),
				configuration.getReflectorFactory());
		final DefaultResultContext context = new DefaultResultContext();
		for (Object o : list) {
			context.nextResultObject(o);
			mapResultHandler.handleResult(context);
		}
		Map<K, V> selectedMap = mapResultHandler.getMappedResults();
		return selectedMap;
	}

	@Override
	public <E> List<E> selectList(String statement) {
		return this.selectList(statement, null);
	}

	@Override
	public <E> List<E> selectList(String statement, Object parameter) {
		return this.selectList(statement, parameter, RowBounds.DEFAULT);
	}

	@Override
	public <E> List<E> selectList(String statement, Object parameter, RowBounds rowBounds) {
		try {
			MappedStatement ms = configuration.getMappedStatement(statement);
			List<E> result = executor.query(ms, wrapCollection(parameter), rowBounds, Executor.NO_RESULT_HANDLER);
			return result;
		} catch (Exception e) {
			throw SQLExceptionFactory.wrapException("Error querying database.  Cause: " + e, e);
		} finally {
			ErrorContext.instance().reset();
		}
	}

	@Override
	public void select(String statement, Object parameter, ResultHandler handler) {
		select(statement, parameter, RowBounds.DEFAULT, handler);
	}

	@Override
	public void select(String statement, ResultHandler handler) {
		select(statement, null, RowBounds.DEFAULT, handler);
	}

	@Override
	public void select(String statement, Object parameter, RowBounds rowBounds, ResultHandler handler) {
		try {
			MappedStatement ms = configuration.getMappedStatement(statement);
			executor.query(ms, wrapCollection(parameter), rowBounds, handler);
		} catch (Exception e) {
			throw SQLExceptionFactory.wrapException("Error querying database.  Cause: " + e, e);
		} finally {
			ErrorContext.instance().reset();
		}
	}

	@Override
	public int insert(String statement) {
		return insert(statement, null);
	}

	@Override
	public int insert(String statement, Object parameter) {
		return update(statement, parameter);
	}

	@Override
	public int update(String statement) {
		return update(statement, null);
	}

	@Override
	public int update(String statement, Object parameter) {
		try {
			dirty = true;
			MappedStatement ms = configuration.getMappedStatement(statement);
			return executor.update(ms, wrapCollection(parameter));
		} catch (Exception e) {
			throw SQLExceptionFactory.wrapException("Error updating database.  Cause: " + e, e);
		} finally {
			ErrorContext.instance().reset();
		}
	}

	@Override
	public int delete(String statement) {
		return update(statement, null);
	}

	@Override
	public int delete(String statement, Object parameter) {
		return update(statement, parameter);
	}

	@Override
	public void commit() {
		commit(false);
	}

	@Override
	public void commit(boolean force) {
		try {
			executor.commit(isCommitOrRollbackRequired(force));
			dirty = false;
		} catch (Exception e) {
			throw SQLExceptionFactory.wrapException("Error committing transaction.  Cause: " + e, e);
		} finally {
			ErrorContext.instance().reset();
		}
	}

	@Override
	public void rollback() {
		rollback(false);
	}

	@Override
	public void rollback(boolean force) {
		try {
			executor.rollback(isCommitOrRollbackRequired(force));
			dirty = false;
		} catch (Exception e) {
			throw SQLExceptionFactory.wrapException("Error rolling back transaction.  Cause: " + e, e);
		} finally {
			ErrorContext.instance().reset();
		}
	}

	@Override
	public List<BatchResult> flushStatements() {
		try {
			return executor.flushStatements();
		} catch (Exception e) {
			throw SQLExceptionFactory.wrapException("Error flushing statements.  Cause: " + e, e);
		} finally {
			ErrorContext.instance().reset();
		}
	}

	@Override
	public void close() {
		try {
			executor.close(isCommitOrRollbackRequired(false));
			dirty = false;
		} finally {
			ErrorContext.instance().reset();
		}
	}

	@Override
	public Configuration getConfiguration() {
		return configuration;
	}

	@Override
	public <T> T getMapper(Class<T> type) {
		return configuration.<T> getMapper(type, this);
	}

	@Override
	public Connection getConnection() {
		try {
			return executor.getTransaction().getConnection();
		} catch (SQLException e) {
			throw SQLExceptionFactory.wrapException("Error getting a new connection.  Cause: " + e, e);
		}
	}

	@Override
	public void clearCache() {
		executor.clearLocalCache();
	}

	private boolean isCommitOrRollbackRequired(boolean force) {
		return dirty || force;
	}

	private Object wrapCollection(final Object object) {
		if (object instanceof List) {
			StrictMap<Object> map = new StrictMap<Object>();
			map.put("list", object);
			return map;
		} else if (object != null && object.getClass().isArray()) {
			StrictMap<Object> map = new StrictMap<Object>();
			map.put("array", object);
			return map;
		}
		return object;
	}

	public static class StrictMap<V> extends HashMap<String, V> {

		private static final long serialVersionUID = -5741767162221585340L;

		@Override
		public V get(Object key) {
			if (!super.containsKey(key)) {
				throw new BindingException(
						"Parameter '" + key + "' not found. Available parameters are " + this.keySet());
			}
			return super.get(key);
		}

	}

	static class SQLExceptionFactory {
		public static RuntimeException wrapException(String message, Exception e) {
			if (SQLException.class.isInstance(e) && message != null && message.contains("Incorrect string value")) {
				return new JahhanException(1, "内容含有特殊字符", e);
			}
			return new PersistenceException(ErrorContext.instance().message(message).cause(e).toString(), e);
		}
	}

	@Override
	public <T> Cursor<T> selectCursor(String statement) {
		JahhanException.throwException( JahhanErrorCode.UNSUPPORT_ERROR, "不支持");
		return null;
	}

	@Override
	public <T> Cursor<T> selectCursor(String statement, Object parameter) {
		JahhanException.throwException( JahhanErrorCode.UNSUPPORT_ERROR, "不支持");
		return null;
	}

	@Override
	public <T> Cursor<T> selectCursor(String statement, Object parameter, RowBounds rowBounds) {
		JahhanException.throwException( JahhanErrorCode.UNSUPPORT_ERROR, "不支持");
		return null;
	}

}
