/*
 * Decompiled with CFR 0.152.
 */
package net.hasor.cobble.i18n;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;
import java.util.function.Function;
import net.hasor.cobble.ClassUtils;
import net.hasor.cobble.ResourcesUtils;
import net.hasor.cobble.StringUtils;
import net.hasor.cobble.i18n.I18nResource;
import net.hasor.cobble.logging.Logger;
import net.hasor.cobble.ref.LinkedCaseInsensitiveMap;
import net.hasor.cobble.text.token.GenericTokenParser;

public class I18nUtils {
    private static final Logger logger = Logger.getLogger(I18nUtils.class);
    public static final I18nUtils DEFAULT = new I18nUtils();
    private static final Map<String, Locale> LOCAL_CACHE = new LinkedCaseInsensitiveMap<Locale>();
    private String defaultI18nKey = I18nUtils.toI18nKey(Locale.getDefault());
    private final I18nMessageSource messageSource;
    private final Function<String, String> variablesSource;
    private final Map<String, ClassLoader> i18nSource;
    private final Set<String> i18nLoaded;

    public static I18nUtils initI18n(String ... i18nResources) throws IOException {
        I18nUtils utils = new I18nUtils();
        utils.loadResources(i18nResources);
        return utils;
    }

    public static I18nUtils initI18n(ClassLoader resourceLoader, String ... i18nResources) throws IOException {
        I18nUtils utils = new I18nUtils();
        utils.loadResources(resourceLoader, i18nResources);
        return utils;
    }

    public static Locale getLocale(String i18nKey) {
        return LOCAL_CACHE.get(i18nKey);
    }

    public static Locale getLocale(String language, String country) {
        return LOCAL_CACHE.computeIfAbsent(language + "_" + country, s -> new Locale(language, country));
    }

    public static String toI18nKey(String language, String country) {
        if (StringUtils.isNotBlank(language) && StringUtils.isNotBlank(country)) {
            return language + "_" + country;
        }
        if (StringUtils.isNotBlank(country)) {
            return country;
        }
        return language;
    }

    public static String toI18nKey(Locale locale) {
        if (locale == null) {
            return null;
        }
        if (StringUtils.isNotBlank(locale.getLanguage()) && StringUtils.isNotBlank(locale.getCountry())) {
            return locale.getLanguage() + "_" + locale.getCountry();
        }
        if (StringUtils.isNotBlank(locale.getCountry())) {
            return locale.getCountry();
        }
        return locale.getLanguage();
    }

    private I18nUtils() {
        this.messageSource = new I18nMessageSourceImpl();
        this.variablesSource = new VariablesSourceImpl();
        this.i18nSource = new LinkedHashMap<String, ClassLoader>();
        this.i18nLoaded = new LinkedHashSet<String>();
    }

    public I18nUtils(I18nMessageSource messageSource) {
        this.messageSource = Objects.requireNonNull(messageSource, "messageSource is null.");
        this.variablesSource = new VariablesSourceImpl();
        this.i18nSource = new LinkedHashMap<String, ClassLoader>();
        this.i18nLoaded = new LinkedHashSet<String>();
    }

    public I18nUtils(I18nMessageSource messageSource, Function<String, String> variablesSource) {
        this.messageSource = Objects.requireNonNull(messageSource, "messageSource is null.");
        this.variablesSource = variablesSource == null ? new VariablesSourceImpl() : variablesSource;
        this.i18nSource = new LinkedHashMap<String, ClassLoader>();
        this.i18nLoaded = new LinkedHashSet<String>();
    }

    public String getDefaultI18nKey() {
        return this.defaultI18nKey;
    }

    public void setDefaultI18nKey(String defaultI18nKey) {
        this.defaultI18nKey = defaultI18nKey;
    }

    public void setDefaultI18nKey(Locale defaultLocale) {
        this.defaultI18nKey = I18nUtils.toI18nKey(defaultLocale);
    }

    public Set<String> getI18nSources() {
        return Collections.unmodifiableSet(this.i18nSource.keySet());
    }

    protected void loadResources(String ... i18nResources) throws IOException {
        ClassLoader classLoader = ClassUtils.getClassLoader(Thread.currentThread().getContextClassLoader());
        this.loadResources(classLoader, i18nResources);
    }

    protected void loadResources(ClassLoader resourceLoader, String ... i18nResources) throws IOException {
        if (!(this.messageSource instanceof I18nMessageSourceImpl)) {
            throw new UnsupportedOperationException("I18nMessageSource is external.");
        }
        resourceLoader = resourceLoader == null ? ClassUtils.getClassLoader(Thread.currentThread().getContextClassLoader()) : resourceLoader;
        for (String i18n : i18nResources) {
            if (this.i18nSource.containsKey(i18n)) continue;
            this.i18nSource.put(i18n, resourceLoader);
            String i18nResource = i18n + ".properties";
            try (InputStream stream = ResourcesUtils.getResourceAsStream(resourceLoader, i18nResource);){
                if (stream == null) continue;
                Properties properties = new Properties();
                properties.load(new InputStreamReader(stream, StandardCharsets.UTF_8));
                properties.forEach((BiConsumer<? super Object, ? super Object>)((BiConsumer<Object, Object>)(key, value) -> ((I18nMessageSourceImpl)this.messageSource).putDictionary(key.toString(), value.toString())));
            }
        }
    }

    public void loadResources(Class<?> ... i18nResources) throws IOException {
        HashSet<Class> loaded = new HashSet<Class>();
        for (Class<?> i18nType : i18nResources) {
            List<Class<?>> interfaces = ClassUtils.getAllInterfaces(i18nType);
            List<Class<?>> superclasses = ClassUtils.getAllSuperclasses(i18nType);
            Collections.reverse(interfaces);
            Collections.reverse(superclasses);
            ArrayList loadTypes = new ArrayList();
            loadTypes.addAll(interfaces);
            loadTypes.addAll(superclasses);
            loadTypes.add(i18nType);
            for (Class clazz : loadTypes) {
                if (loaded.contains(clazz)) continue;
                loaded.add(clazz);
                I18nResource i18nResource = clazz.getAnnotation(I18nResource.class);
                if (i18nResource == null) continue;
                this.loadResources(clazz.getClassLoader(), i18nResource.value());
            }
        }
    }

    public void addVariables(String varName, String varValue) {
        if (!(this.variablesSource instanceof VariablesSourceImpl)) {
            throw new UnsupportedOperationException("variablesSource is external.");
        }
        ((VariablesSourceImpl)this.variablesSource).put(varName, varValue);
    }

    public void putVariables(Map<String, String> variables) {
        if (variables == null || !(this.variablesSource instanceof VariablesSourceImpl)) {
            throw new UnsupportedOperationException("variablesSource is external.");
        }
        ((VariablesSourceImpl)this.variablesSource).putAll(variables);
    }

    public String getMessage(String code) {
        return this.getMessage(code, null, this.defaultI18nKey);
    }

    public String getMessage(String code, Object[] args) {
        return this.getMessage(code, args, this.defaultI18nKey);
    }

    public String getMessage(String code, Object[] args, String i18nLocal) {
        if (StringUtils.isBlank(code)) {
            return code;
        }
        Locale locale = LOCAL_CACHE.containsKey(i18nLocal) ? LOCAL_CACHE.get(i18nLocal) : Locale.getDefault();
        return this.getMessage(code, args, locale);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String getMessage(String code, Object[] args, Locale locale) {
        String message;
        String i18nKey = I18nUtils.toI18nKey(locale);
        if (!this.i18nLoaded.contains(i18nKey)) {
            I18nUtils i18nUtils = this;
            synchronized (i18nUtils) {
                if (!this.i18nLoaded.contains(i18nKey)) {
                    for (String i18nResourcePath : this.i18nSource.keySet()) {
                        String i18nResource = i18nResourcePath + "_" + i18nKey + ".properties";
                        ClassLoader i18nLoader = this.i18nSource.get(i18nResourcePath);
                        try {
                            InputStream stream = ResourcesUtils.getResourceAsStream(i18nLoader, i18nResource);
                            Throwable throwable = null;
                            try {
                                if (stream == null) continue;
                                Properties properties = new Properties();
                                properties.load(new InputStreamReader(stream, StandardCharsets.UTF_8));
                                properties.forEach((BiConsumer<? super Object, ? super Object>)((BiConsumer<Object, Object>)(key, value) -> ((I18nMessageSourceImpl)this.messageSource).putDictionary(key.toString(), locale, value.toString())));
                            }
                            catch (Throwable throwable2) {
                                throwable = throwable2;
                                throw throwable2;
                            }
                            finally {
                                if (stream == null) continue;
                                if (throwable != null) {
                                    try {
                                        stream.close();
                                    }
                                    catch (Throwable throwable3) {
                                        throwable.addSuppressed(throwable3);
                                    }
                                    continue;
                                }
                                stream.close();
                            }
                        }
                        catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                    }
                    this.i18nLoaded.add(i18nKey);
                }
            }
        }
        if ((message = this.messageSource.getMessage(code, args, locale)) == null) {
            return code;
        }
        message = this.resolveMessageArgs(message);
        if (args == null || args.length == 0) {
            return message;
        }
        return MessageFormat.format(message, args);
    }

    private String resolveMessageArgs(String msg) {
        return new GenericTokenParser(new String[]{"${"}, "}", (builder, openToken, closeToken, content) -> {
            String var;
            String varKey = content;
            String varDefault = "";
            int defaultIndexOf = content.indexOf(":");
            if (defaultIndexOf != -1) {
                varDefault = content.substring(defaultIndexOf + 1);
                varKey = content.substring(0, defaultIndexOf);
            }
            if (StringUtils.isBlank(var = this.resolveArg(varKey)) && StringUtils.isNotBlank(varDefault)) {
                var = varDefault;
            }
            if (varKey.equalsIgnoreCase(var)) {
                return varKey;
            }
            return var;
        }).parse(msg);
    }

    protected String resolveArg(String argName) {
        if (this.variablesSource != null) {
            return this.variablesSource.apply(argName);
        }
        return argName;
    }

    public void checkDifference(String localName) throws IOException {
        for (String resource : this.getI18nSources()) {
            ClassLoader classLoader = this.i18nSource.get(resource);
            this.checkDifference(resource, classLoader, localName);
        }
    }

    private void checkDifference(String resource, ClassLoader classLoader, String localName) throws IOException {
        if (StringUtils.isBlank(localName)) {
            throw new IllegalArgumentException("localName is null.");
        }
        String srcPropFile = resource + ".properties";
        String diffPropFile = resource + "_" + localName + ".properties";
        this.checkDifference(classLoader, srcPropFile, diffPropFile);
    }

    protected void checkDifference(ClassLoader loader, String srcFile, String dstFile) throws IOException {
        try (InputStream srcStream = ResourcesUtils.getResourceAsStream(loader, srcFile);
             InputStream dstStream = ResourcesUtils.getResourceAsStream(loader, dstFile);){
            Properties srcProps = new Properties();
            srcProps.load(srcStream);
            Properties dstProps = new Properties();
            dstProps.load(dstStream);
            for (String key : srcProps.stringPropertyNames()) {
                String msgUs = dstProps.getProperty(key, null);
                if (msgUs != null) continue;
                logger.warn("I18N: " + dstFile + " not exist " + key + " entry.");
            }
        }
        catch (IOException e) {
            logger.error("I18N: open i18n file " + srcFile + "or" + dstFile + " failed.");
            throw e;
        }
    }

    static {
        for (Locale locale : Locale.getAvailableLocales()) {
            String i18nKey = locale.getLanguage() + "_" + locale.getCountry();
            LOCAL_CACHE.put(i18nKey, locale);
        }
    }

    private static class VariablesSourceImpl
    extends ConcurrentHashMap<String, String>
    implements Function<String, String> {
        private VariablesSourceImpl() {
        }

        @Override
        public String apply(String s) {
            return super.getOrDefault(s, s);
        }
    }

    private static class I18nMessageSourceImpl
    implements I18nMessageSource {
        private final Map<String, String> defaultDictionary = new ConcurrentHashMap<String, String>();
        private final Map<String, Map<Locale, String>> i18nDictionary = new ConcurrentHashMap<String, Map<Locale, String>>();

        private I18nMessageSourceImpl() {
        }

        @Override
        public String getMessage(String code, Object[] args, Locale locale) {
            Map<Locale, String> localeMap;
            if (locale == null) {
                locale = Locale.getDefault();
            }
            if ((localeMap = this.i18nDictionary.get(code)) == null || !localeMap.containsKey(locale)) {
                if (this.defaultDictionary.containsKey(code)) {
                    return this.defaultDictionary.get(code);
                }
                return null;
            }
            return localeMap.get(locale);
        }

        public void putDictionary(String code, String message) {
            this.defaultDictionary.put(code, message);
        }

        public void putDictionary(String code, Locale locale, String message) {
            if (locale == null) {
                locale = Locale.getDefault();
            }
            if (!this.i18nDictionary.containsKey(code)) {
                this.i18nDictionary.put(code, new ConcurrentHashMap());
            }
            this.i18nDictionary.get(code).put(locale, message);
        }
    }

    public static interface I18nMessageSource {
        public String getMessage(String var1, Object[] var2, Locale var3);
    }
}

