/*
 *  Copyright © 2016 Amichai Rothman
 *
 *  This file is part of JScrollPhat - the Java Scroll pHAT package.
 *
 *  JScrollPhat is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  JScrollPhat is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with JScrollPhat.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  For additional info see http://www.freeutils.net/source/jscrollphat/
 */

package net.freeutils.scrollphat;

import java.io.*;
import java.util.ArrayList;
import java.util.List;

/**
 * Provides General-purpose utility methods.
 */
public class Utils {

    // prevent instantiation
    private Utils() {}

    /**
     * Returns an input stream for reading the content of a file or resource.
     * <p>
     * First an attempt is made to read from a file with the given name (full path).
     * If such a file does not exist, an attempt is made to load a resource with
     * the same name preceded by a slash using {@link Class#getResourceAsStream}.
     *
     * @param filename the file (or resource) name.
     * @return the input stream
     * @throws FileNotFoundException if a file or resource with the given name
     *         is not found
     */
    public static InputStream getInputStream(String filename) throws FileNotFoundException {
        InputStream in;
        if (new File(filename).exists())
            in = new FileInputStream(filename);
        else
            in = Utils.class.getResourceAsStream("/" + filename);
        if (in == null)
            throw new FileNotFoundException(filename);
        return new BufferedInputStream(in);
    }

    /**
     * Reads the entire contents of the given stream into a byte array.
     * The stream is closed by this method.
     *
     * @param in an input stream
     * @return a byte array containing the full contents of the stream
     * @throws IOException if an error occurs
     */
    public static byte[] readBytes(InputStream in) throws IOException {
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream(4096);
            byte[] buf = new byte[4096];
            int read;
            while ((read = in.read(buf)) > -1)
                out.write(buf, 0, read);
            return out.toByteArray();
        } finally {
            in.close();
        }
    }

    /**
     * Reads an entire text file into an array of lines.
     *
     * @param file the file to read
     * @return the text lines
     * @throws IOException if an error occurs
     */
    public static String[] readLines(File file) throws IOException {
        return readLines(new FileInputStream(file));
    }

    /**
     * Reads an entire text stream into an array of lines.
     * The stream is closed by this method.
     *
     * @param in the steam to read from
     * @return the text lines
     * @throws IOException if an error occurs
     */
    public static String[] readLines(InputStream in) throws IOException {
        List<String> lines = new ArrayList<String>();
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(in));
            String line;
            while ((line = reader.readLine()) != null)
                lines.add(line);
        } finally {
            if (reader != null)
                reader.close();
            in.close();
        }
        return lines.toArray(new String[lines.size()]);
    }

    /**
     * Reads a varint (variable-length integer) from a stream.
     *
     * @param in the stream to read from
     * @return the integer value
     * @throws IOException if an error occurs
     */
    public static long readVarint(InputStream in) throws IOException {
        long val = 0;
        int bits = 0;
        int b;
        do {
            b = in.read();
            val |= (b & 0x7f) << bits;
            bits += 7;
            if (b < 0)
                throw new EOFException("unexpected end of stream");
        } while ((b & ~0x7f) != 0);
        return val;
    }

    /**
     * Writes a varint (variable-length integer) to a stream.
     *
     * @param out the stream to write to
     * @param val the value to write
     * @throws IOException if an error occurs
     */
    public static void writeVarint(OutputStream out, long val) throws IOException {
        while ((val & ~0x7f) != 0) {
            out.write((int)(val | 0x80));
            val >>>= 7;
        }
        out.write((int)val);
    }

    /**
     * Appends the hex string representing the given byte value to a StringBuffer.
     *
     * @param sb the StringBuffer to append to
     * @param b the byte value whose hex string representation is to be appended
     * @return the StringBuffer
     */
    public static StringBuilder append(StringBuilder sb, byte b) {
        String s = Integer.toHexString(b & 0xff);
        if (s.length() == 1)
            sb.append('0');
        return sb.append(s);
    }

    /**
     * Appends the hex string representing the given byte values to a StringBuffer.
     *
     * @param sb the StringBuffer to append to
     * @param bytes the byte values whose hex string representation is to be appended
     * @param offset the offset within the byte array where the values start
     * @param length the number of values
     * @return the StringBuffer
     */
    public static StringBuilder append(StringBuilder sb, byte[] bytes, int offset, int length) {
        sb.append('[');
        for (int i = 0; i < length; i++)
            append(sb, bytes[offset + i]).append(' ');
        sb.setCharAt(sb.length() - 1, ']');
        return sb;
    }
}
