/*
 *  Copyright © 2016 Amichai Rothman
 *
 *  This file is part of JScrollPhat - the Java Scroll pHAT package.
 *
 *  JScrollPhat is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  JScrollPhat is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with JScrollPhat.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  For additional info see http://www.freeutils.net/source/jscrollphat/
 */

package net.freeutils.scrollphat;

import java.io.IOException;

/**
 * A Device implementation that writes to the console instead
 * of to a real device. All writes to the device are logged,
 * and display updates are printed out as ASCII art.
 * <p>
 * This class is extremely convenient to use for debugging,
 * during development on a desktop PC, or on a Raspberry Pi
 * which does not (yet) have a real device connected to it.
 */
public class MockDevice extends Device {

    byte[] registers = new byte[256]; // holds the registers state

    @Override
    protected Device openImpl(int busNumber, int address) throws IOException {
        StringBuilder sb = new StringBuilder(64);
        sb.append("opening device on bus #").append(busNumber).append(" at address 0x");
        Utils.append(sb, (byte)address);
        System.out.println(sb);
        return this;
    }

    @Override
    protected void closeImpl() {
        System.out.println("closing device");
    }

    @Override
    protected void writeImpl(byte register, byte[] data, int offset, int length) throws IOException {
        System.arraycopy(data, offset, registers, register & 0xff, length);
        StringBuilder sb = new StringBuilder(32);
        sb.append("writing to register 0x");
        Utils.append(sb, register).append(": ");
        Utils.append(sb, data, offset, length);
        System.out.println(sb);
    }

    @Override
    public void update() throws IOException {
        String art = Canvas.toAsciiArt(registers, REG_MATRIX1_DATA_START, getWidth(), getHeight());
        System.out.println(art);
    }

    /**
     * The main command-line utility entry point.
     *
     * @param args the arguments
     * @throws IOException if an error occurs
     * @throws InterruptedException if the thread is interrupted
     */
    public static void main(String[] args) throws IOException, InterruptedException {
        new MockDevice().init().displayTestPatterns();
    }
}
