/*
 *  Copyright © 2016 Amichai Rothman
 *
 *  This file is part of JScrollPhat - the Java Scroll pHAT package.
 *
 *  JScrollPhat is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  JScrollPhat is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with JScrollPhat.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  For additional info see http://www.freeutils.net/source/jscrollphat/
 */

package net.freeutils.scrollphat;

import java.io.IOException;

/**
 * Provides common animation utilities.
 */
public class Animation {

    /**
     * Gradually changes the display brightness.
     *
     * @param device the device
     * @param startValue the brightness value to start at (0-128)
     * @param endValue the brightness value to end at (0-128)
     * @param duration the duration in milliseconds of the gradual change
     * @throws IOException if an error occurs
     * @throws InterruptedException if the thread is interrupted
     */
    public static void gradualBrightness(Device device, int startValue, int endValue, int duration)
            throws IOException, InterruptedException {
        int steps = Math.abs(endValue - startValue);
        int interval = steps == 0 || duration <= 0 ? 1 : duration / steps;
        for (int i = startValue; i < endValue; i++) {
            device.setBrightness(i);
            Thread.sleep(interval);
        }
        device.setBrightness(endValue);
    }

    /**
     * Scrolls the display from left to right.
     *
     * @param device the device
     * @param matrix the pixel matrix containing the full display
     * @param interval the interval in milliseconds between consecutive scrolling steps
     * @param iterations the number of times the entire scrolled display is repeated
     *        (a negative value means indefinite scrolling)
     * @param spaceBefore the amount of space (empty columns) to add before the matrix
     * @param spaceAfter the amount of space (empty columns) to add after the matrix
     * @throws IOException if an error occurs
     * @throws InterruptedException if the thread is interrupted
     */
    public static void scroll(Device device, byte[] matrix, int interval, int iterations,
            int spaceBefore, int spaceAfter) throws InterruptedException, IOException {
        int width = device.getWidth();
        spaceAfter = Math.max(spaceAfter, 0);
        spaceBefore = Math.max(spaceBefore, 0);
        int len = matrix.length - width + 1 + spaceBefore + spaceAfter;
        int count = iterations * len;
        Canvas canvas = new Canvas(matrix, device.getHeight());
        for (int i = 0; i < count || iterations < 0; i++) {
            if (i % len == 0)
                canvas.setOffset(-spaceBefore);
            device.update(canvas.render());
            if (interval > 0)
                Thread.sleep(interval);
            canvas.scroll(1);
        }
    }

    /**
     * Scrolls the display from left to right (with spaces before and after text)
     *
     * @param device the device
     * @param matrix the pixel matrix containing the full display
     * @param interval the interval in milliseconds between consecutive scrolling steps
     * @param iterations the number of times the entire scrolled display is repeated
     *        (a negative value means infinite iterations)
     * @throws IOException if an error occurs
     * @throws InterruptedException if the thread is interrupted
     */
    public static void scroll(Device device, byte[] matrix, int interval, int iterations)
            throws InterruptedException, IOException {
        int width = device.getWidth();
        scroll(device, matrix, interval, iterations, width, width);
    }

    public static void transitionHorizontally(Device device, byte[] matrix1, byte[] matrix2,
            int interval) throws IOException, InterruptedException {
        int width = device.getWidth();
        Canvas canvas = new Canvas(2 * width, device.getHeight());
        canvas.setColumns(0, matrix1, Math.max(0, matrix1.length - width), Math.min(width, matrix1.length));
        canvas.setColumns(width, matrix2, 0, Math.min(width, matrix2.length));
        scroll(device, canvas.getMatrix(), interval, 1, 0, 0);
    }

    public static void transitionVertically(Device device, byte[] matrix1, byte[] matrix2, int interval) throws IOException, InterruptedException {
        int height = device.getHeight();
        int width = device.getWidth();
        byte[] buf = new byte[width];
        for (int i = 0; i <= height; i++) {
            for (int j = 0; j < width; j++) {
                buf[j] = 0;
                if (j < matrix1.length)
                    buf[j] |= matrix1[j] >>> i;
                if (j < matrix2.length)
                    buf[j] |= matrix2[j] << (height - i);
            }
            device.update(buf);
            Thread.sleep(interval);
        }
    }
}
