/*
 * Decompiled with CFR 0.152.
 */
package net.e6tech.elements.security.vault;

import java.io.IOException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.spec.RSAPrivateKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Set;
import java.util.function.Function;
import javax.crypto.SecretKey;
import net.e6tech.elements.common.logging.Logger;
import net.e6tech.elements.common.util.SystemException;
import net.e6tech.elements.security.AsymmetricCipher;
import net.e6tech.elements.security.Hex;
import net.e6tech.elements.security.RNG;
import net.e6tech.elements.security.SymmetricCipher;
import net.e6tech.elements.security.vault.ClearText;
import net.e6tech.elements.security.vault.Credential;
import net.e6tech.elements.security.vault.DualEntry;
import net.e6tech.elements.security.vault.FileStore;
import net.e6tech.elements.security.vault.KeyProtected;
import net.e6tech.elements.security.vault.PasswordProtected;
import net.e6tech.elements.security.vault.Secret;
import net.e6tech.elements.security.vault.Vault;
import net.e6tech.elements.security.vault.VaultManagerState;
import net.e6tech.elements.security.vault.VaultStore;

public class VaultManager {
    private static Logger logger = Logger.getLogger();
    public static final String KEY_VAULT = "key-vault";
    public static final String USER_VAULT = "user-vault";
    public static final String DATA_VAULT = "data-vault";
    public static final String LOCAL_VAULT = "local-vault";
    private static final String CANNOT_FIND_USER = "Cannot find user ";
    private static final String USER = "USER ";
    private static final String IS_NOT_GUARDIAN = " is not a guardian";
    private static final String GUARDIA_HAS_BEEN_TEMPERED = " guardian property has been tempered";
    private static final GeneralSecurityException NOT_OPEN_EXCEPTION = new GeneralSecurityException("user local store is not open.  Please call open() first");
    private static final Function<Exception, GeneralSecurityException> BAD_USER_PASSWORD = ex -> new GeneralSecurityException("Bad user name or password to unlock the vault", (Throwable)ex);
    private SymmetricCipher symmetricCipher;
    private AsymmetricCipher asymmetricCipher;
    private VaultStore userLocalStore;
    private boolean userLocalOpened = false;
    private VaultStore keyDataStore;
    private boolean keyDataOpened = false;
    private PasswordProtected pwd = new PasswordProtected();
    private KeyProtected keyEncryption = new KeyProtected();
    private VaultManagerState state = new VaultManagerState();

    public VaultManager() {
        this.symmetricCipher = SymmetricCipher.getInstance("AES");
        this.symmetricCipher.setBase64(false);
        this.asymmetricCipher = AsymmetricCipher.getInstance("RSA");
        this.userLocalStore = this.keyDataStore = new FileStore();
        this.userLocalStore.manage(USER_VAULT, LOCAL_VAULT);
        this.keyDataStore.manage(KEY_VAULT, DATA_VAULT);
    }

    public VaultStore getKeyDataStore() {
        return this.keyDataStore;
    }

    public void setKeyDataStore(VaultStore keyDataStore) {
        this.keyDataStore = keyDataStore;
    }

    public VaultStore getUserLocalStore() {
        return this.userLocalStore;
    }

    public void setUserLocalStore(VaultStore userLocalStore) {
        this.userLocalStore = userLocalStore;
    }

    public SymmetricCipher getSymmetricCipher() {
        return this.symmetricCipher;
    }

    public AsymmetricCipher getAsymmetricCipher() {
        return this.asymmetricCipher;
    }

    private ClearText generateInternalKeyPair(String alias) throws GeneralSecurityException {
        KeyPair keyPair = this.asymmetricCipher.generateKeySpec();
        KeyFactory fact = this.asymmetricCipher.getKeyFactory();
        RSAPublicKeySpec pub = fact.getKeySpec(keyPair.getPublic(), RSAPublicKeySpec.class);
        RSAPrivateKeySpec priv = fact.getKeySpec(keyPair.getPrivate(), RSAPrivateKeySpec.class);
        ClearText ct = new ClearText();
        ct.alias(alias);
        BigInteger mod = priv.getModulus();
        BigInteger exp = priv.getPrivateExponent();
        String encoded = mod.toString(16) + "$" + exp.toString(16);
        ct.setBytes(encoded.getBytes(StandardCharsets.UTF_8));
        ct.setProperty("type", "key-pair");
        ct.setProperty("algorithm", this.asymmetricCipher.getAlgorithm());
        ct.setProperty("public-key-mod", pub.getModulus().toString(16));
        ct.setProperty("public-key-exp", pub.getPublicExponent().toString(16));
        ct.protect();
        return ct;
    }

    private ClearText generateSignature() {
        ClearText ct = new ClearText();
        byte[] bytes = RNG.generateSeed(16);
        ct.setBytes(bytes);
        ct.alias("signature");
        ct.setProperty("signature", Hex.toString(bytes));
        ct.setProperty("type", "signature");
        ct.setProperty("signature-format", "1.0");
        ct.protect();
        return ct;
    }

    private ClearText getPassphrase() throws GeneralSecurityException {
        Secret secret = this.getLocal("passphrase", null);
        return this.pwd.unsealUserOrPassphrase(secret, this.state.getCurrentPassphrase());
    }

    private ClearText passphraseForSecret(Secret secret) throws GeneralSecurityException {
        String[] components = secret.getSecret().split("\\$");
        String keyVersion = null;
        if (components.length >= 5) {
            keyVersion = components[4];
        }
        if (keyVersion != null && keyVersion.trim().equals("0")) {
            keyVersion = null;
        }
        Secret passphraseSecret = this.getLocal("passphrase", keyVersion);
        return this.pwd.unsealUserOrPassphrase(passphraseSecret, this.state.getCurrentPassphrase());
    }

    private ClearText passphraseClearText(char[] passphrase) {
        ClearText ct = new ClearText();
        ct.setBytes(new String(passphrase).getBytes(StandardCharsets.UTF_8));
        ct.alias("passphrase");
        ct.setProperty("type", "passphrase");
        ct.protect();
        return ct;
    }

    private ClearText generateInternalKey(String alias) {
        SecretKey secretKey = this.symmetricCipher.generateKeySpec();
        ClearText ct = new ClearText();
        ct.setBytes(secretKey.getEncoded());
        ct.alias(alias);
        ct.setProperty("type", "key");
        ct.setProperty("algorithm", "AES");
        ct.protect();
        return ct;
    }

    private ZonedDateTime setCreationTimeVersion(ClearText ... clearTexts) {
        ZonedDateTime now = ZonedDateTime.now();
        String dateTime = now.format(DateTimeFormatter.ISO_ZONED_DATE_TIME);
        for (ClearText ct : clearTexts) {
            if (ct.version() != null) continue;
            ct.setProperty("creation-date-time", dateTime);
            ct.setProperty("creation-time", Long.toString(now.toInstant().toEpochMilli()));
            ct.version(Long.toString(now.toInstant().toEpochMilli()));
        }
        return now;
    }

    public RSAPublicKeySpec getPublicKey() throws GeneralSecurityException {
        KeyFactory fact = this.asymmetricCipher.getKeyFactory();
        ClearText key = this.getKey("asy-key", null);
        if (key == null) {
            return null;
        }
        return fact.getKeySpec(key.asKeyPair().getPublic(), RSAPublicKeySpec.class);
    }

    public boolean validateUser(String user, char[] password) {
        try {
            if (user == null || password == null) {
                return false;
            }
            ClearText ct1 = this.getUser(new Credential(user, password));
            if (ct1 == null) {
                return false;
            }
            return ct1.getProperty("type").equals("user");
        }
        catch (Exception th) {
            Logger.suppress((Throwable)th);
            return false;
        }
    }

    public void addKeyPair(DualEntry dualEntry, String alias) throws GeneralSecurityException {
        this.checkAccess(dualEntry);
        ClearText ct = this.generateInternalKeyPair(alias);
        this.addData(ct);
    }

    public void addSecretData(DualEntry dualEntry, String alias, ClearText ct) throws GeneralSecurityException {
        this.checkAccess(dualEntry);
        ct.alias(alias);
        ct.setProperty("type", "secret");
        ct.setProtectedProperty("type", "secret");
        this.addData(ct);
    }

    public ClearText getSecretData(Credential credential, String alias) throws GeneralSecurityException {
        return this.getSecretData(credential, alias, null);
    }

    public ClearText getSecretData(Credential credential, String alias, String version) throws GeneralSecurityException {
        this.checkAccess(credential);
        Secret secret = this.getData(alias, version);
        if (secret == null) {
            return null;
        }
        String[] components = this.encryptedComponents(secret.getSecret());
        String keyAlias = components[2];
        String keyVersion = components[3];
        return this.keyEncryption.unseal(secret, this.getKey(keyAlias, keyVersion));
    }

    private String[] encryptedComponents(String encoded) {
        String[] components = encoded.split("\\$");
        if (components.length != 4) {
            throw new IllegalStateException("Invalid encryption format");
        }
        return components;
    }

    public String generateKey(DualEntry dualEntry) throws GeneralSecurityException {
        return this.generateKey(dualEntry, false);
    }

    public String generateKey(DualEntry dualEntry, boolean asymmetricKey) throws GeneralSecurityException {
        return asymmetricKey ? this.generateAsymmetricKey(dualEntry) : this.generateSymmetricKey(dualEntry);
    }

    private String generateSymmetricKey(DualEntry dualEntry) throws GeneralSecurityException {
        byte[] plain = this.symmetricCipher.generateKeySpec().getEncoded();
        this.checkAccess(dualEntry);
        return this.internalEncrypt("m-key", null, plain);
    }

    private String generateAsymmetricKey(DualEntry dualEntry) throws GeneralSecurityException {
        byte[] plain = VaultManager.generateEncodedAsymmetricKey(this.asymmetricCipher);
        this.checkAccess(dualEntry);
        return this.internalEncrypt("m-key", null, plain);
    }

    public static byte[] generateEncodedAsymmetricKey(AsymmetricCipher asymmetricCipher) throws GeneralSecurityException {
        KeyPair keyPair = asymmetricCipher.generateKeySpec();
        KeyFactory fact = asymmetricCipher.getKeyFactory();
        RSAPublicKeySpec pub = fact.getKeySpec(keyPair.getPublic(), RSAPublicKeySpec.class);
        RSAPrivateKeySpec priv = fact.getKeySpec(keyPair.getPrivate(), RSAPrivateKeySpec.class);
        String encoded = "RSA$M=" + priv.getModulus().toString(16) + "$D=" + priv.getPrivateExponent().toString(16) + "$E=" + pub.getPublicExponent().toString(16);
        return encoded.getBytes();
    }

    public String importKey(DualEntry dualEntry, String plainKey, String iv) throws GeneralSecurityException {
        return this.importKey(dualEntry, plainKey, iv, false);
    }

    public String importKey(DualEntry dualEntry, String plainKey, String iv, String version) throws GeneralSecurityException {
        return this.importKey(dualEntry, plainKey, iv, false, version);
    }

    public String importKey(DualEntry dualEntry, String plainKey, String iv, boolean asymmetricKey) throws GeneralSecurityException {
        return this.importKey(dualEntry, plainKey, iv, asymmetricKey, null);
    }

    public String importKey(DualEntry dualEntry, String plainKey, String iv, boolean asymmetricKey, String version) throws GeneralSecurityException {
        return asymmetricKey ? this.importAsymmetricKey(dualEntry, plainKey, iv, version) : this.importSymmetricKey(dualEntry, plainKey, iv, version);
    }

    private String importAsymmetricKey(DualEntry dualEntry, String plainKey, String iv, String version) throws GeneralSecurityException {
        byte[] plain = this.symmetricCipher.toBytes(plainKey);
        String[] keyValuesParts = new String(plain).split("\\$");
        if (keyValuesParts.length != 4) {
            throw new GeneralSecurityException("Invalid key format: expecting the key to be 4 components separated by \\$");
        }
        try {
            RSAPrivateKeySpec privateKeySpec = new RSAPrivateKeySpec(new BigInteger(keyValuesParts[1].substring(2), 16), new BigInteger(keyValuesParts[2].substring(2), 16));
            RSAPublicKeySpec publicKeySpec = new RSAPublicKeySpec(new BigInteger(keyValuesParts[1].substring(2), 16), new BigInteger(keyValuesParts[3].substring(2), 16));
            this.asymmetricCipher.getKeyFactory().generatePrivate(privateKeySpec);
            this.asymmetricCipher.getKeyFactory().generatePublic(publicKeySpec);
        }
        catch (Exception e) {
            throw new SystemException("Can't create private and public key", (Throwable)e);
        }
        this.checkAccess(dualEntry);
        return this.internalEncrypt("m-key", version, plain, iv);
    }

    private String importSymmetricKey(DualEntry dualEntry, String plainKey, String iv, String version) throws GeneralSecurityException {
        byte[] plain = this.symmetricCipher.toBytes(plainKey);
        byte[] plain2 = this.symmetricCipher.generateKeySpec().getEncoded();
        String iv2 = this.symmetricCipher.generateIV();
        if (plain2.length != plain.length) {
            throw new GeneralSecurityException("Invalid key length: expecting the key to be " + plain2.length + " bytes.");
        }
        if (iv2.length() != plain.length) {
            int ivLength = this.symmetricCipher.toBytes(iv2).length;
            throw new GeneralSecurityException("Invalid IV length: expecting the IV to be " + ivLength + " bytes.");
        }
        this.checkAccess(dualEntry);
        return this.internalEncrypt("m-key", version, plain, iv);
    }

    public String encrypt(Credential credential, String key, byte[] data, String iv) throws GeneralSecurityException {
        this.checkAccess(credential);
        byte[] keyBytes = this.internalDecrypt(key);
        SecretKey secretKey = this.symmetricCipher.getKeySpec(keyBytes);
        return this.symmetricCipher.encrypt(secretKey, data, iv);
    }

    public byte[] decrypt(Credential credential, String key, String secret, String iv) throws GeneralSecurityException {
        this.checkAccess(credential);
        byte[] keyBytes = this.internalDecrypt(key);
        SecretKey secretKey = this.symmetricCipher.getKeySpec(keyBytes);
        return this.symmetricCipher.decrypt(secretKey, secret, iv);
    }

    public byte[] decrypt(Credential credential, String secret) throws GeneralSecurityException {
        this.checkAccess(credential);
        return this.internalDecrypt(secret);
    }

    public String encryptPublic(byte[] data) throws GeneralSecurityException {
        ClearText key = this.getKey("asy-key", null);
        if (key == null) {
            throw new GeneralSecurityException("Public/Private key not set up");
        }
        return this.asymmetricCipher.encrypt(key.asKeyPair().getPublic(), data);
    }

    public byte[] decryptPrivate(String secret) throws GeneralSecurityException {
        ClearText key = this.getKey("asy-key", null);
        if (key == null) {
            throw new GeneralSecurityException("Public/Private key not set up");
        }
        return this.asymmetricCipher.decrypt(key.asKeyPair().getPrivate(), secret);
    }

    private ClearText getUser(Credential credential) throws GeneralSecurityException {
        Secret user = this.getUser(credential.getUser(), null);
        if (user == null) {
            return null;
        }
        return this.pwd.unsealUserOrPassphrase(user, credential.getPassword());
    }

    private void newUser(Credential credential, byte[] component, String group) throws GeneralSecurityException {
        Secret user = this.getUser(credential.getUser(), null);
        if (user != null) {
            throw new GeneralSecurityException("User exists: " + credential.getUser());
        }
        ClearText ct1 = new ClearText();
        ct1.alias(credential.getUser());
        ct1.setBytes(component);
        ct1.setProperty("username", credential.getUser());
        ct1.setProperty("guardian", group);
        ct1.setProperty("type", "user");
        this.setCreationTimeVersion(ct1);
        ct1.protect();
        this.addUser(this.pwd.sealUser(ct1, credential.getPassword()));
    }

    public void addUser(Credential newUser, Credential existingUser) throws GeneralSecurityException {
        this.checkAccess(existingUser);
        ClearText ct1 = this.getUser(existingUser);
        if (ct1 == null) {
            throw new GeneralSecurityException("Existing user not found: " + existingUser.getUser());
        }
        this.newUser(newUser, ct1.getBytes(), ct1.getProperty("guardian"));
    }

    public void changePassword(String user, char[] oldPwd, char[] newPwd) throws GeneralSecurityException {
        Vault userVault = this.userLocalStore.getVault(USER_VAULT);
        Set<Long> versions = userVault.versions(user);
        for (Long version : versions) {
            Secret secret = this.getUser(user, "" + version);
            ClearText ct = this.pwd.unsealUserOrPassphrase(secret, oldPwd);
            this.addUser(this.pwd.sealUser(ct, newPwd));
        }
    }

    public void passphraseLock(DualEntry dualEntry, String alias, ClearText ct) throws GeneralSecurityException {
        ClearText passphrase;
        ct.alias(alias);
        if (alias.equals("passphrase")) {
            passphrase = this.passphraseClearText(this.getUserComponents(dualEntry));
            passphrase.version("0");
        } else {
            Secret secret = this.getLocal("passphrase", null);
            passphrase = this.pwd.unsealUserOrPassphrase(secret, this.getUserComponents(dualEntry));
        }
        this.addLocal(ct, passphrase);
    }

    public ClearText passphraseUnlock(Credential credential, String alias) throws GeneralSecurityException {
        this.checkAccess(credential);
        Secret secret = this.getLocal(alias, null);
        if (secret == null) {
            return null;
        }
        return this.pwd.unseal(secret, this.getPassphrase());
    }

    public void newMasterKey(DualEntry dualEntry) throws GeneralSecurityException {
        this.checkAccess(dualEntry);
        this.addKey(this.generateInternalKey("m-key"));
        Vault dataVault = this.keyDataStore.getVault(DATA_VAULT);
        Set<String> aliases = dataVault.aliases();
        for (String alias : aliases) {
            Set<Long> versions = dataVault.versions(alias);
            for (Long version : versions) {
                ClearText ct = this.getSecretData(dualEntry.getUser1(), alias, "" + version);
                if (ct == null) continue;
                this.addSecretData(dualEntry, alias, ct);
            }
        }
    }

    private void checkAccess(Credential credential) throws GeneralSecurityException {
        ClearText ct = this.getUser(credential);
        if (ct == null) {
            throw new GeneralSecurityException(CANNOT_FIND_USER + credential.getUser());
        }
        if (!ct.getProperty("guardian").equals("group-1") && !ct.getProperty("guardian").equals("group-2")) {
            throw new GeneralSecurityException(USER + ct.getProperty("username") + IS_NOT_GUARDIAN);
        }
        if (!ct.getProperty("guardian").equals(ct.getProtectedProperty("guardian"))) {
            throw new GeneralSecurityException(USER + ct.getProperty("username") + GUARDIA_HAS_BEEN_TEMPERED);
        }
    }

    private void checkAccess(DualEntry dualEntry) throws GeneralSecurityException {
        ClearText ct1 = this.getUser(dualEntry.getUser1());
        ClearText ct2 = this.getUser(dualEntry.getUser2());
        if (ct1 == null) {
            throw new GeneralSecurityException("Bad user name " + dualEntry.getUser1());
        }
        if (ct2 == null) {
            throw new GeneralSecurityException("Bad user name " + dualEntry.getUser2());
        }
        if (ct1.getProperty("guardian") == null || !ct1.getProperty("guardian").equals("group-1") && !ct1.getProperty("guardian").equals("group-2")) {
            throw new SystemException(USER + ct1.getProperty("username") + IS_NOT_GUARDIAN);
        }
        if (ct2.getProperty("guardian") == null || !ct2.getProperty("guardian").equals("group-1") && !ct2.getProperty("guardian").equals("group-2")) {
            throw new GeneralSecurityException(USER + ct2.getProperty("username") + IS_NOT_GUARDIAN);
        }
        if (ct1.getProperty("guardian").equals(ct2.getProperty("guardian"))) {
            throw new SystemException("User1 " + ct1.getProperty("username") + " and user 2 " + ct2.getProperty("username") + " cannot be in the same group");
        }
        if (!ct1.getProperty("guardian").equals(ct1.getProtectedProperty("guardian"))) {
            throw new GeneralSecurityException(USER + ct1.getProperty("username") + GUARDIA_HAS_BEEN_TEMPERED);
        }
        if (!ct2.getProperty("guardian").equals(ct2.getProtectedProperty("guardian"))) {
            throw new GeneralSecurityException(USER + ct2.getProperty("username") + GUARDIA_HAS_BEEN_TEMPERED);
        }
    }

    private char[] getUserComponents(DualEntry dualEntry) throws GeneralSecurityException {
        ClearText ct1 = this.getUser(dualEntry.getUser1());
        if (ct1 == null) {
            throw new GeneralSecurityException(CANNOT_FIND_USER + dualEntry.getUser1());
        }
        ClearText ct2 = this.getUser(dualEntry.getUser2());
        if (ct2 == null) {
            throw new GeneralSecurityException(CANNOT_FIND_USER + dualEntry.getUser2());
        }
        this.checkAccess(dualEntry);
        byte[] comp1 = ct1.getBytes();
        byte[] comp2 = ct2.getBytes();
        return this.xor(comp1, comp2);
    }

    private char[] xor(byte[] comp1, byte[] comp2) {
        byte[] pwdBytes = new byte[comp1.length];
        for (int i = 0; i < comp1.length; ++i) {
            pwdBytes[i] = (byte)((comp1[i] ^ comp2[i]) & 0xFF);
        }
        return Hex.toString(pwdBytes).toCharArray();
    }

    public ClearText getSignature() {
        return this.state.getSignature();
    }

    public void changePassphrase(DualEntry dualEntry) throws GeneralSecurityException {
        ClearText newPassphrase;
        this.checkAccess(dualEntry);
        VaultManagerState oldState = this.state.clone();
        String currentVersion = this.state.getSignature().version();
        try {
            this.userLocalStore.backup(currentVersion);
            this.keyDataStore.backup(currentVersion);
        }
        catch (IOException ex) {
            throw new GeneralSecurityException(ex);
        }
        ClearText newSignature = this.generateSignature();
        try {
            newPassphrase = this.userLocalPassphraseChange(dualEntry, newSignature);
        }
        catch (Exception ex) {
            this.state = oldState;
            throw new GeneralSecurityException(ex);
        }
        try {
            this.userLocalStore.save();
        }
        catch (Exception ex) {
            this.state = oldState;
            try {
                this.userLocalStore.restore(currentVersion);
            }
            catch (IOException io) {
                throw new GeneralSecurityException(io);
            }
            throw new GeneralSecurityException(ex);
        }
        try {
            this.keyDataPassphraseChange(newSignature, newPassphrase);
        }
        catch (Exception ex) {
            this.state = oldState;
            try {
                this.userLocalStore.restore(currentVersion);
            }
            catch (IOException io) {
                throw new GeneralSecurityException(io);
            }
            throw new GeneralSecurityException(ex);
        }
        try {
            this.keyDataStore.save();
        }
        catch (Exception ex) {
            try {
                this.keyDataStore.restore(currentVersion);
            }
            catch (IOException io) {
                throw new GeneralSecurityException(io);
            }
            this.state = oldState;
            try {
                this.userLocalStore.restore(currentVersion);
            }
            catch (IOException io) {
                throw new GeneralSecurityException(io);
            }
            throw new GeneralSecurityException(ex);
        }
    }

    private ClearText userLocalPassphraseChange(DualEntry dualEntry, ClearText newSignature) throws GeneralSecurityException {
        byte[] comp1 = RNG.generateSeed(16);
        byte[] comp2 = RNG.generateSeed(16);
        char[] newRandomPassphrase = this.xor(comp1, comp2);
        ClearText newPassphrase = this.passphraseClearText(newRandomPassphrase);
        this.setCreationTimeVersion(newPassphrase, newSignature);
        try {
            Vault localVault = this.userLocalStore.getVault(LOCAL_VAULT);
            localVault.removeSecret("signature", null);
            Set<String> aliases = localVault.aliases();
            for (String alias : aliases) {
                Set<Long> versions = localVault.versions(alias);
                for (Long version : versions) {
                    Secret secret = this.getLocal(alias, "" + version);
                    ClearText oldPassphrase = this.passphraseForSecret(secret);
                    ClearText ct = this.pwd.unseal(secret, oldPassphrase);
                    this.addLocal(ct, newPassphrase);
                }
            }
            ClearText ct1 = this.getUser(dualEntry.getUser1());
            ClearText ct2 = this.getUser(dualEntry.getUser2());
            if (ct1 == null || ct2 == null) {
                throw new IllegalStateException();
            }
            ct1.setBytes(comp1);
            ct2.setBytes(comp2);
            this.addUser(this.pwd.sealUser(ct1, dualEntry.getUser1().getPassword()));
            this.addUser(this.pwd.sealUser(ct2, dualEntry.getUser2().getPassword()));
            this.passphraseLock(dualEntry, "passphrase", newPassphrase);
            this.state.setCurrentPassphrase(this.getUserComponents(dualEntry));
            this.addLocal(newSignature, newPassphrase);
            aliases = this.listUsers();
            for (String alias : aliases) {
                if (dualEntry.getUser1().getUser().equals(alias) || dualEntry.getUser2().getUser().equals(alias)) continue;
                this.removeUser(alias, null);
            }
        }
        catch (Exception ex) {
            throw new GeneralSecurityException(ex);
        }
        return newPassphrase;
    }

    private void keyDataPassphraseChange(ClearText newSignature, ClearText newPassphrase) throws GeneralSecurityException {
        Vault keyVault = this.keyDataStore.getVault(KEY_VAULT);
        Set<String> aliases = keyVault.aliases();
        for (String alias : aliases) {
            Set<Long> versions = keyVault.versions(alias);
            for (Long version : versions) {
                ClearText ct = this.getKey(alias, "" + version);
                Secret secret = this.pwd.seal(ct, newPassphrase);
                keyVault.addSecret(secret);
            }
        }
        this.keyDataStore.getVault(DATA_VAULT).removeSecret("signature", null);
        this.addData(newSignature);
        this.state.setSignature(newSignature);
    }

    public void restore(DualEntry dualEntry, String version) throws GeneralSecurityException {
        String currentVersion = this.state.getSignature().version();
        try {
            this.userLocalStore.backup(currentVersion);
            this.keyDataStore.backup(currentVersion);
        }
        catch (IOException ex) {
            throw new GeneralSecurityException(ex);
        }
        try {
            this.keyDataStore.restore(version);
            this.userLocalStore.restore(version);
        }
        catch (IOException io) {
            throw new GeneralSecurityException(io);
        }
        try {
            this.state.setCurrentPassphrase(this.getUserComponents(dualEntry));
        }
        catch (GeneralSecurityException ex) {
            throw BAD_USER_PASSWORD.apply(ex);
        }
        this.state.setSignature(this.passphraseUnlock(dualEntry.getUser1(), "signature"));
    }

    private String internalEncrypt(String keyAlias, String version, byte[] plain) throws GeneralSecurityException {
        return this.internalEncrypt(keyAlias, version, plain, null);
    }

    private String internalEncrypt(String keyAlias, String version, byte[] plain, String iv) throws GeneralSecurityException {
        ClearText ct = this.getKey(keyAlias, version);
        if (ct == null) {
            throw new GeneralSecurityException("No key for keyAlias=" + keyAlias);
        }
        if (iv == null) {
            iv = this.symmetricCipher.generateIV();
        }
        String encrypted = this.symmetricCipher.encrypt(ct.asSecretKey(), plain, iv);
        return iv + "$" + encrypted + "$" + keyAlias + "$" + ct.version();
    }

    public byte[] internalDecrypt(String encoded) throws GeneralSecurityException {
        String version;
        String[] components = this.encryptedComponents(encoded);
        String alias = components[2];
        ClearText ct = this.getKey(alias, version = components[3]);
        if (ct == null) {
            throw new GeneralSecurityException("No key for keyAlias=" + alias);
        }
        return this.symmetricCipher.decrypt(ct.asSecretKey(), components[1], components[0]);
    }

    public void save() throws IOException {
        this.userLocalStore.save();
        this.keyDataStore.save();
    }

    public void close() throws IOException {
        this.userLocalStore.close();
        this.keyDataStore.close();
    }

    public void open(DualEntry dualEntry) throws GeneralSecurityException {
        if (this.userLocalOpened) {
            return;
        }
        try {
            this.userLocalStore.open();
        }
        catch (IOException e) {
            throw new GeneralSecurityException(e);
        }
        this.userLocalOpened = true;
        boolean modified = false;
        if (this.userLocalStore.getVault(USER_VAULT).size() == 0) {
            byte[] comp1 = RNG.generateSeed(16);
            byte[] comp2 = RNG.generateSeed(16);
            this.newUser(dualEntry.getUser1(), comp1, "group-1");
            this.newUser(dualEntry.getUser2(), comp2, "group-2");
            char[] newRandomPassphrase = this.getUserComponents(dualEntry);
            try {
                this.state.setCurrentPassphrase(this.getUserComponents(dualEntry));
            }
            catch (GeneralSecurityException ex) {
                throw BAD_USER_PASSWORD.apply(ex);
            }
            ClearText passphrase = this.passphraseClearText(newRandomPassphrase);
            this.passphraseLock(dualEntry, "passphrase", passphrase);
            ClearText signature = this.generateSignature();
            this.passphraseLock(dualEntry, "signature", signature);
            modified = true;
        }
        if (modified) {
            try {
                this.userLocalStore.save();
            }
            catch (IOException e) {
                throw new SystemException((Throwable)e);
            }
        }
        try {
            this.state.setCurrentPassphrase(this.getUserComponents(dualEntry));
        }
        catch (GeneralSecurityException ex) {
            throw BAD_USER_PASSWORD.apply(ex);
        }
        this.state.setSignature(this.passphraseUnlock(dualEntry.getUser1(), "signature"));
    }

    private void openKeyData() throws GeneralSecurityException {
        if (this.keyDataOpened) {
            return;
        }
        if (!this.userLocalOpened) {
            throw new GeneralSecurityException("user local store is not open.  Please call open() first");
        }
        try {
            this.keyDataStore.open();
        }
        catch (IOException e) {
            throw new GeneralSecurityException(e);
        }
        this.keyDataOpened = true;
        boolean modified = false;
        Secret keyStoreSigSecret = this.getData("signature", null);
        if (keyStoreSigSecret == null) {
            this.initKeys();
            this.addData(this.state.getSignature());
            modified = true;
        } else {
            String signatureFormat = this.state.getSignature().getProperty("signature-format");
            String dataSignatureFormat = keyStoreSigSecret.getProperty("signature-format");
            boolean restore = false;
            if (signatureFormat != null) {
                if (!signatureFormat.equals(dataSignatureFormat) || !this.state.getSignature().getProperty("signature").equals(keyStoreSigSecret.getProperty("signature"))) {
                    restore = true;
                }
            } else if (dataSignatureFormat != null) {
                restore = true;
            }
            if (restore) {
                try {
                    if (logger.isWarnEnabled()) {
                        logger.warn("Restoring key data store to version {}", (Object)this.state.getSignature().version());
                    }
                    this.keyDataStore.restore(this.state.getSignature().version());
                    keyStoreSigSecret = this.getData("signature", null);
                }
                catch (IOException e) {
                    throw new GeneralSecurityException("Property signature do not match.  Local store vault signature does not match key store signature", e);
                }
            }
            modified = this.initKeys();
            String[] components = this.encryptedComponents(keyStoreSigSecret.getSecret());
            String keyAlias = components[2];
            String keyVersion = components[3];
            ClearText keyStoreSig = this.keyEncryption.unseal(keyStoreSigSecret, this.getKey(keyAlias, keyVersion));
            if (!Arrays.equals(this.state.getSignature().getBytes(), keyStoreSig.getBytes())) {
                try {
                    if (logger.isWarnEnabled()) {
                        logger.warn("Restoring key data store to version {}", (Object)this.state.getSignature().version());
                    }
                    this.keyDataStore.restore(this.state.getSignature().version());
                }
                catch (IOException e) {
                    throw new GeneralSecurityException("Local store vault signature does not match key store signature", e);
                }
            }
        }
        if (modified) {
            try {
                this.keyDataStore.save();
            }
            catch (IOException e) {
                throw new SystemException((Throwable)e);
            }
        }
    }

    private boolean initKeys() throws GeneralSecurityException {
        boolean modified = false;
        if (this.getKey("m-key", null) == null) {
            this.addKey(this.generateInternalKey("m-key"));
            modified = true;
        }
        if (this.getKey("auth-key", null) == null) {
            this.addKey(this.generateInternalKey("auth-key"));
            modified = true;
        }
        if (this.getKey("asy-key", null) == null) {
            this.addKey(this.generateInternalKeyPair("asy-key"));
            modified = true;
        }
        return modified;
    }

    public ClearText getKey(DualEntry dualEntry, String keyAlias, String version) throws GeneralSecurityException {
        this.checkAccess(dualEntry);
        return this.getKey(keyAlias, version);
    }

    protected ClearText getKey(String keyAlias, String version) throws GeneralSecurityException {
        this.openKeyData();
        Secret key = this.keyDataStore.getVault(KEY_VAULT).getSecret(keyAlias, version);
        if (key == null) {
            return null;
        }
        return this.pwd.unseal(key, this.passphraseForSecret(key));
    }

    protected void addKey(ClearText ct) throws GeneralSecurityException {
        this.openKeyData();
        this.setCreationTimeVersion(ct);
        ClearText passphrase = this.getPassphrase();
        Secret secret = this.pwd.seal(ct, passphrase);
        this.keyDataStore.getVault(KEY_VAULT).addSecret(secret);
    }

    private void addData(ClearText ct) throws GeneralSecurityException {
        this.openKeyData();
        this.setCreationTimeVersion(ct);
        ClearText key = this.getKey("m-key", null);
        Secret secret = this.keyEncryption.seal(ct.alias(), ct, key);
        this.keyDataStore.getVault(DATA_VAULT).addSecret(secret);
    }

    private Secret getData(String alias, String version) throws GeneralSecurityException {
        this.openKeyData();
        return this.keyDataStore.getVault(DATA_VAULT).getSecret(alias, version);
    }

    private void addUser(Secret secret) throws GeneralSecurityException {
        if (!this.userLocalOpened) {
            throw NOT_OPEN_EXCEPTION;
        }
        this.userLocalStore.getVault(USER_VAULT).addSecret(secret);
    }

    private Secret getUser(String alias, String version) throws GeneralSecurityException {
        if (!this.userLocalOpened) {
            throw NOT_OPEN_EXCEPTION;
        }
        return this.userLocalStore.getVault(USER_VAULT).getSecret(alias, version);
    }

    private void removeUser(String alias, String version) throws GeneralSecurityException {
        if (!this.userLocalOpened) {
            throw NOT_OPEN_EXCEPTION;
        }
        this.userLocalStore.getVault(USER_VAULT).removeSecret(alias, version);
    }

    public Set<String> listUsers() throws GeneralSecurityException {
        if (!this.userLocalOpened) {
            throw NOT_OPEN_EXCEPTION;
        }
        return this.userLocalStore.getVault(USER_VAULT).aliases();
    }

    private void addLocal(ClearText ct, ClearText passphrase) throws GeneralSecurityException {
        if (!this.userLocalOpened) {
            throw NOT_OPEN_EXCEPTION;
        }
        this.setCreationTimeVersion(ct);
        Secret secret = this.pwd.seal(ct, passphrase);
        this.userLocalStore.getVault(LOCAL_VAULT).addSecret(secret);
    }

    private Secret getLocal(String alias, String version) throws GeneralSecurityException {
        if (!this.userLocalOpened) {
            throw NOT_OPEN_EXCEPTION;
        }
        return this.userLocalStore.getVault(LOCAL_VAULT).getSecret(alias, version);
    }
}

