/*
 * Copyright 2016 Digital Receipt Exchange Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.dreceiptx.receipt.document;

import net.dreceiptx.receipt.validation.ReceiptValidation;
import java.util.ArrayList;
import java.util.List;
import com.google.gson.annotations.SerializedName;
import net.dreceiptx.receipt.validation.ValidationErrors;

public class StandardBusinessDocumentHeader {
    @SerializedName("sender") private List<DocumentOwner> _sender;
    @SerializedName("receiver") private List<DocumentOwner> _receiver;
    @SerializedName("documentIdentification") private DocumentIdentification _documentIdentification;
    private DocumentOwner _merchant;
    private DocumentOwner _dRx;
    private DocumentOwner _user;
    
    public StandardBusinessDocumentHeader()
    {
        _sender = new ArrayList<DocumentOwner>();
        _receiver = new ArrayList<DocumentOwner>();
        _documentIdentification = new DocumentIdentification();

        _merchant = new DocumentOwner();
        _merchant.getIdentifier().setAuthority("GS1");
        _merchant.getIdentifier().setValue(null);
        // TODO: Is there a reason we set the index to 0?
        _sender.add(0, _merchant);

        _dRx = new DocumentOwner();
        _dRx.getIdentifier().setAuthority("GS1");
        _dRx.getIdentifier().setValue(null);
        // TODO: Is there a reason we set the index to 0?
        _receiver.add(0, _dRx);

        _user = new DocumentOwner();
        _user.getIdentifier().setAuthority("dRx");
        _user.getIdentifier().setValue(null);
        // TODO: Is there a reason we set the index to 1?
        _receiver.add(1, _user);
    }

    public void setMerchantGLN(String merchantGLN) {
        _merchant.setValue(merchantGLN);
    }
    
    public String getMerchantGLN() {
        return _merchant.getValue();
    }
    
    public void setdRxGLN(String dRxGLN) {
        _dRx.setValue(dRxGLN);
    }
    
    public String getdRxGLN() {
        return _dRx.getValue();
    }
    
    public void setUserIdentifier(String userIdentifier) {
        _user.setValue(userIdentifier);
    }
    
    public String getUserIdentifier() {
        return _user.getValue();
    }
    
    public List<ReceiptContact> getClientContacts() {
        return _user.getDocumentOwnerContact();
    }
    
    public void addMerchantContact(ReceiptContact contact) {
        _merchant.addDocumentOwnerContact(contact);
    }
    
    public void addRMSContact(ReceiptContact contact) {
        _user.addDocumentOwnerContact(contact);
    }

    public List<DocumentOwner> getReceiver() {
        return _receiver;
    }
    
    public List<DocumentOwner> getSender() {
        return _sender;
    }

    public void addReceiver(DocumentOwner receiver) {
        _receiver.add(receiver);
    }

    public DocumentIdentification getDocumentIdentification() {
        return _documentIdentification;
    }

    public void setDocumentIdentification(DocumentIdentification documentIdentification) {
        _documentIdentification = documentIdentification;
    }
    
    public ReceiptValidation validate(ReceiptValidation receiptValidation) {
        if (_sender.isEmpty()) {
            receiptValidation.AddError(ValidationErrors.MerchantGLNMustBeSet);
        }

        return receiptValidation;
    }
}
