/*
 * Copyright 2016 Digital Receipt Exchange Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.dreceiptx.client;

import net.dreceiptx.receipt.DigitalReceipt;
import net.dreceiptx.receipt.DigitalReceiptGenerator;
import net.dreceiptx.receipt.config.ConfigManager;
import net.dreceiptx.receipt.merchant.Merchant;
import net.dreceiptx.receipt.serialization.json.NewUsersSerializer;
import net.dreceiptx.users.NewUser;
import net.dreceiptx.users.NewUserRegistrationResult;
import net.dreceiptx.users.User;
import net.dreceiptx.users.UserIdentifierType;
import net.dreceiptx.users.Users;
import net.dreceiptx.client.exception.ExchangeClientException;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.apache.commons.codec.binary.Base64;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class DRXClient implements ExchangeClient {
    private static final int BUFFER_SIZE = 4096;
    private final ConfigManager _configManager;
    private static final String CONTENT_TYPE_JSON = "application/json";
    private static final String CONTENT_TYPE_XML = "application/xml";
    private static final String CONTENT_TYPE_PDF = "application/pdf";
    private final EnvironmentType _environmentType;
    private String _exchangeProtocol = "https";
    private final String _exchangeHostname;
    private final String _requesterId;
    private final String _receiptVersion;
    private String _directoryProtocol = "https";
    private final String _directoryHostname;
    private final String _userVersion;
    private final String _downloadDirectory;
    private int _responseErrorCode = 500;
    private String _responseErrorMessage = "Unknown Error";

    /**
     * Creates instance of ExchangeClient using the given ConfigManager
     *
     * @param configManager the ConfigManager to be used by the ExchangeClient
     * @throws ExchangeClientException
     */
    public DRXClient(ConfigManager configManager) throws ExchangeClientException {
        _configManager = configManager;
        _exchangeHostname = this.validateConfigOption("exchange.hostname");
        _directoryHostname = this.validateConfigOption("directory.hostname");
        _requesterId = this.validateConfigOption("api.requesterId");
        _receiptVersion = this.validateConfigOption("receipt.version");
        _userVersion = this.validateConfigOption("user.version");
        _downloadDirectory = this.validateConfigOption("download.directory");
        
        if(_configManager.exists("exchange.protocol")){
            _environmentType = EnvironmentType.valueOf(_configManager.getConfigValue("environment.type"));
        }else{
            _environmentType = EnvironmentType.PROD;
        }
        
        if(_configManager.exists("exchange.protocol")){
            _exchangeProtocol = _configManager.getConfigValue("exchange.protocol");
        }
        
        if(_configManager.exists("directory.protocol")){
            _directoryProtocol = _configManager.getConfigValue("directory.protocol");
        }
        
        switch (_environmentType) {
            case PROD:
                _exchangeProtocol = "https";
                _directoryProtocol = "https";
            break;
        }
    }

    @Override
    public User searchUser(UserIdentifierType identiferType, String identifer) throws ExchangeClientException {
        try {
            HttpURLConnection connection = createConnection(_directoryProtocol, _directoryHostname,
                    "/users/" + identiferType.getValue().toLowerCase()+ "/" + createHashedIdentifier(identiferType, identifer),
                    CONTENT_TYPE_JSON, "GET", _userVersion);
            connection.connect();
            int responseCode = connection.getResponseCode();
            if (responseCode == HttpCodes.HTTP_200_OK) {
                JsonObject exchangeResponse = getReponseJsonObject(connection);
                User user = new User();
                if (exchangeResponse.get("success").getAsBoolean()) {
                    JsonObject responseData = exchangeResponse.get("responseData").getAsJsonObject();
                    user.setGUID(responseData.get("GUID").getAsString());
                    user.setRMS(responseData.get("RMS").getAsString());
                    connection.disconnect();
                    return user;
                } else {
                    throw new ExchangeClientException(exchangeResponse.get("code").getAsInt(), exchangeResponse.get("exceptionMessage").getAsString());
                }

            }else if (responseCode == HttpCodes.HTTP_400_BAD_REQUEST || responseCode == HttpCodes.HTTP_401_UNAUTHORIZED) {
                loadErrorReponseJsonObject(connection);
                connection.disconnect();
                throw new ExchangeClientException(_responseErrorCode, _responseErrorMessage);
            } else {
                String errorMessage = connection.getResponseMessage();
                connection.disconnect();
                throw new ExchangeClientException(responseCode, errorMessage);
            }
        }
        catch (ExchangeClientException dRxE) {
            throw dRxE;
        } catch (Exception e) {
            throw new ExchangeClientException(500, e.toString(), e);
        }
    }
    
    @Override
    public Users searchUsers(UserIdentifierType identiferType, ArrayList<String> userIdentifiers, Boolean hashed) throws ExchangeClientException {
        StringBuilder urlparameters = new StringBuilder();
        urlparameters.append("?type=");
        urlparameters.append(identiferType.getValue().toLowerCase());
        urlparameters.append("&identifiers=");
        Boolean firstIteration = true;
        for (String userIdentifier : userIdentifiers) {
            if(!firstIteration){
                urlparameters.append(";");
            }
            firstIteration = false;
            urlparameters.append(userIdentifier);
        }
        urlparameters.append("&hashed=");
        urlparameters.append(hashed.toString());
        try {
            HttpURLConnection connection = createConnection(_directoryProtocol, _directoryHostname,
                    "/users" + urlparameters.toString(),
                    CONTENT_TYPE_JSON, "GET", _userVersion);
            connection.connect();
            int responseCode = connection.getResponseCode();
            if (responseCode == HttpCodes.HTTP_200_OK) {
                JsonObject exchangeResponse = getReponseJsonObject(connection);
                Users users = new Users();
                if (exchangeResponse.get("success").getAsBoolean()) {
                    JsonObject responseData = exchangeResponse.get("responseData").getAsJsonObject();
                    JsonObject userIdentifiersObject = responseData.get("userIdentifiers").getAsJsonObject();
                    for (Map.Entry<String, JsonElement> entry : userIdentifiersObject.entrySet()) {
                        User user = null;
                        if(!entry.getValue().isJsonNull()){
                            JsonObject userIdentifierObject = userIdentifiersObject.getAsJsonObject(entry.getKey());
                            user = new User();
                            user.setIdentifierType(UserIdentifierType.EMAIL);
                            user.setIdentifier(entry.getKey());
                            user.setGUID(userIdentifierObject.get("GUID").getAsString());
                            user.setRMS(userIdentifierObject.get("RMS").getAsString());
                        }
                        users.add(entry.getKey(), user);
                    }
                    connection.disconnect();
                    return users;
                } else {
                    throw new ExchangeClientException(exchangeResponse.get("code").getAsInt(), exchangeResponse.get("exceptionMessage").getAsString());
                }

            }else if (responseCode == HttpCodes.HTTP_400_BAD_REQUEST || responseCode == HttpCodes.HTTP_401_UNAUTHORIZED) {
                loadErrorReponseJsonObject(connection);
                connection.disconnect();
                throw new ExchangeClientException(_responseErrorCode, _responseErrorMessage);
            } else {
                String errorMessage = connection.getResponseMessage();
                connection.disconnect();
                throw new ExchangeClientException(responseCode, errorMessage);
            }
        }
        catch (ExchangeClientException dRxE) {
            throw dRxE;
        } catch (Exception e) {
            throw new ExchangeClientException(500, e.toString(), e);
        }
    }

    @Override
    public String sendReceipt(DigitalReceiptGenerator reciept) throws ExchangeClientException {

        try {
            HttpURLConnection connection = createConnection(_exchangeProtocol, _exchangeHostname, "/receipt", CONTENT_TYPE_JSON,
                    "POST", _receiptVersion);
            OutputStream os = connection.getOutputStream();
            os.write(reciept.encodeJson().getBytes());
            os.flush();
            connection.connect();
            int responseCode = connection.getResponseCode();
            if (responseCode == HttpCodes.HTTP_201_CREATED) {
                JsonObject exchangeResponse = getReponseJsonObject(connection);
                if (exchangeResponse.get("success").getAsBoolean()) {
                    JsonObject responseData = exchangeResponse.get("responseData").getAsJsonObject();
                    return responseData.get("receiptId").getAsString();
                } else {
                    throw new ExchangeClientException(exchangeResponse.get("code").getAsInt(), exchangeResponse.get("exceptionMessage").getAsString());
                }

            } else if (responseCode == HttpCodes.HTTP_400_BAD_REQUEST || responseCode == HttpCodes.HTTP_401_UNAUTHORIZED) {
                loadErrorReponseJsonObject(connection);
                connection.disconnect();
                throw new ExchangeClientException(_responseErrorCode, _responseErrorMessage);
            } else {
                String errorMessage = connection.getResponseMessage();
                connection.disconnect();
                throw new ExchangeClientException(responseCode, errorMessage);
            }
        }
        catch (ExchangeClientException dRxE) {
            throw dRxE;
        } catch (Exception e) {
            throw new ExchangeClientException(500, e.toString(), e);
        }
    }

    @Override
    public DigitalReceipt lookupReceipt(String receiptId) throws ExchangeClientException {
        try {
            HttpURLConnection connection = createConnection(_exchangeProtocol, _exchangeHostname, "/receipt/" + receiptId, CONTENT_TYPE_JSON, "GET", _receiptVersion);
            connection.connect();
            int responseCode = connection.getResponseCode();
            if (responseCode == HttpCodes.HTTP_200_OK) {
                String response = getReponseString(connection);
                connection.disconnect();
                return new DigitalReceipt(response);
            } else if (responseCode == HttpCodes.HTTP_400_BAD_REQUEST || responseCode == HttpCodes.HTTP_401_UNAUTHORIZED) {
                loadErrorReponseJsonObject(connection);
                connection.disconnect();
                throw new ExchangeClientException(_responseErrorCode, _responseErrorMessage);
            } else {
                String errorMessage = connection.getResponseMessage();
                connection.disconnect();
                throw new ExchangeClientException(responseCode, errorMessage);
            }
        } catch (Exception e) {
            throw new ExchangeClientException(500, e.toString(), e);
        }
    }

    public boolean downloadReceiptPDF(String receiptId) throws ExchangeClientException {
        try {
            HttpURLConnection connection = createConnection(_exchangeProtocol, _exchangeHostname, "/receipt/" + receiptId, CONTENT_TYPE_PDF, "GET", _receiptVersion);
            connection.connect();
            int responseCode = connection.getResponseCode();
            if (responseCode == HttpCodes.HTTP_200_OK) {
                InputStream pdfInputStream = connection.getInputStream();
                String saveFilePath = _downloadDirectory + File.separator + receiptId + ".pdf";

                // opens an output stream to save into file
                FileOutputStream pdfOutputStream = new FileOutputStream(saveFilePath);

                int bytesRead;
                byte[] buffer = new byte[BUFFER_SIZE];
                while ((bytesRead = pdfInputStream.read(buffer)) != -1) {
                    pdfOutputStream.write(buffer, 0, bytesRead);
                }

                pdfOutputStream.close();
                pdfInputStream.close();

                return true;
            } else if (responseCode == HttpCodes.HTTP_400_BAD_REQUEST || responseCode == HttpCodes.HTTP_401_UNAUTHORIZED) {
                loadErrorReponseJsonObject(connection);
                connection.disconnect();
                throw new ExchangeClientException(_responseErrorCode, _responseErrorMessage);
            } else {
                String errorMessage = connection.getResponseMessage();
                connection.disconnect();
                throw new ExchangeClientException(responseCode, errorMessage);
            }
        }
        catch (ExchangeClientException dRxE) {
            throw dRxE;
        }  catch (Exception e) {
            throw new ExchangeClientException(500, e.toString(), e);
        }
    }
    
    public NewUserRegistrationResult registerNewUser(NewUser newUser) throws ExchangeClientException {
        List<NewUser> _newUserCollection = new ArrayList<>();
        _newUserCollection.add(newUser);
        Map<String, NewUserRegistrationResult> _newUserRegistrationResponse = this.registerNewUser(_newUserCollection);
        return _newUserRegistrationResponse.get(newUser.getEmail());
    }
    
    public Map<String, NewUserRegistrationResult> registerNewUser(List<NewUser> newUsers) throws ExchangeClientException {
        Map<String, NewUserRegistrationResult> _newUserRegistrationResponse = new HashMap<>();
        try {
            Gson gson = new GsonBuilder()
                .setDateFormat("yyyy-MM-dd'T'HH:mm:ssZ")
                .registerTypeAdapter(newUsers.getClass(), new NewUsersSerializer())
                .create();
            String newUserRegistrationJson = gson.toJson(newUsers);
            
            HttpURLConnection connection = createConnection(_exchangeProtocol, _exchangeHostname, "/user", CONTENT_TYPE_JSON, "POST", _userVersion);
            OutputStream os = connection.getOutputStream();
            os.write(newUserRegistrationJson.getBytes());
            os.flush();
            connection.connect();
            int responseCode = connection.getResponseCode();
            if (responseCode == HttpCodes.HTTP_201_CREATED || responseCode == HttpCodes.HTTP_200_OK) {
                JsonObject exchangeResponse = getReponseJsonObject(connection);
                if (exchangeResponse.get("success").getAsBoolean()) {
                    JsonObject responseData = exchangeResponse.get("responseData").getAsJsonObject();
                    JsonObject usersObject = responseData.get("users").getAsJsonObject();
                    for (Map.Entry<String, JsonElement> entry : usersObject.entrySet()) {
                        NewUserRegistrationResult newUserRegistrationResult = new NewUserRegistrationResult();
                        if(!entry.getValue().isJsonNull()){
                            JsonObject userRegistrationObject = usersObject.getAsJsonObject(entry.getKey());
                            if(userRegistrationObject.get("success").getAsBoolean()){
                                newUserRegistrationResult.setUserGUID(userRegistrationObject.get("GUID").getAsString());
                            }else{
                                newUserRegistrationResult.setException(userRegistrationObject.get("code").getAsInt(),userRegistrationObject.get("exception").getAsString());
                            }
                        }
                        _newUserRegistrationResponse.put(entry.getKey(), newUserRegistrationResult);
                    }
                } else {
                    throw new ExchangeClientException(exchangeResponse.get("code").getAsInt(), exchangeResponse.get("exceptionMessage").getAsString());
                }

            } else if (responseCode == HttpCodes.HTTP_400_BAD_REQUEST || responseCode == HttpCodes.HTTP_401_UNAUTHORIZED) {
                loadErrorReponseJsonObject(connection);
                connection.disconnect();
                throw new ExchangeClientException(_responseErrorCode, _responseErrorMessage);
            } else if (responseCode == HttpCodes.HTTP_404_NOTFOUND) {
                connection.disconnect();
                throw new ExchangeClientException(_responseErrorCode, "Exchange could not be found, ensure internet connection or valid URL");
            }else {
                String errorMessage = connection.getResponseMessage();
                connection.disconnect();
                throw new ExchangeClientException(responseCode, errorMessage);
            }
        }
        catch (ExchangeClientException dRxE) {
            throw dRxE;
        } catch (Exception e) {
            throw new ExchangeClientException(500, e.toString(), e);
        }
        
        return _newUserRegistrationResponse;
    }

    public Merchant lookupMerchant(String Id) throws ExchangeClientException {

        try {
            String merchantLocationHostname = "https://cdn.dreceiptx.net/merchant/location/";
            if(!_environmentType.equals(EnvironmentType.PROD)){
                merchantLocationHostname = "https://cdn.dreceiptx.net/uat/merchant/location/";
            }
            URL merchantRequest = new URL(merchantLocationHostname + Id + "/info.json");
            HttpURLConnection connection = (HttpURLConnection) merchantRequest.openConnection();
            
            int responseCode = connection.getResponseCode();
            if (responseCode == HttpCodes.HTTP_200_OK) {
                BufferedReader merchantRequestConnectionReader = new BufferedReader(new InputStreamReader(
                    connection.getInputStream()));
                StringBuilder response = new StringBuilder();
                String line;
                while ((line = merchantRequestConnectionReader.readLine()) != null) {
                    response.append(line);
                }
                merchantRequestConnectionReader.close();
                Gson gson = new Gson();
                Merchant merchant = gson.fromJson(response.toString(), Merchant.class);
                merchant.setMerchantLocationHostname(merchantLocationHostname);
                return merchant;
            }
            else{
                throw new ExchangeClientException(404, "Unknown merchant Id, please supply a valid ");
            }
        }
        catch (ExchangeClientException dRxE) {
            throw dRxE;
        }  catch (Exception e) {
            throw new ExchangeClientException(500, e.toString(), e);
        }
    }

    public static String createHashedIdentifier(UserIdentifierType identiferType, String identifer) {
        String salt = "yBhjjJKkjh4rTWZ3Pvyh4XfjTYy2r7m2";
        String key = identiferType.getValue() + ":" + identifer;
        try {
            Mac sha256_HMAC = Mac.getInstance("HmacSHA256");
            SecretKeySpec secret_key = new SecretKeySpec(salt.getBytes(), "HmacSHA256");
            sha256_HMAC.init(secret_key);
            return Base64.encodeBase64String(sha256_HMAC.doFinal(key.getBytes()));

        } catch (Exception ex) {
            //TODO: Handle Error
        }

        return null;
    }

    private String createAuthKey(String timestamp) throws NoSuchAlgorithmException, InvalidKeyException, UnsupportedEncodingException {
        String key = _configManager.getConfigValue("api.key") + ":" + String.valueOf(timestamp) + ":" + _configManager.getConfigValue("api.requesterId");
        Mac sha256_HMAC = Mac.getInstance("HmacSHA256");
        SecretKeySpec secret_key = new SecretKeySpec(_configManager.getConfigValue("api.secret").getBytes(), "HmacSHA256");
        sha256_HMAC.init(secret_key);
        String hash = Base64.encodeBase64String(sha256_HMAC.doFinal(key.getBytes()));
        System.out.println(hash);
        return "DRX " + hash;
    }


    private HttpURLConnection createConnection(String protocol, String hostname, String uri, String contentType, String requestMethod,
                                               String requestVersion) throws IOException, InvalidKeyException, NoSuchAlgorithmException {
        URL url = new URL(protocol + "://" + hostname + uri);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setRequestMethod(requestMethod);
        connection.setDoOutput(true);
        connection.setDoInput(true);
        if(contentType != null) {
            connection.setRequestProperty("Content-Type", contentType);
        }
        String timestamp = String.valueOf(System.currentTimeMillis());
        connection.setRequestProperty("dRx-RequestTimeStamp", timestamp);
        connection.setRequestProperty("dRx-Requester", _requesterId);
        connection.setRequestProperty("Authorization", createAuthKey(timestamp));
        connection.setRequestProperty("dRx-Version", requestVersion);
        connection.setConnectTimeout(5000);
        connection.setReadTimeout(20000);
        return connection;
    }

    private String getReponseString(HttpURLConnection connection) throws IOException, ExchangeClientException {
        try (BufferedReader br = new BufferedReader(new InputStreamReader(connection.getInputStream()))) {
            StringBuilder sb = new StringBuilder();
            String line;
            while ((line = br.readLine()) != null) {
                sb.append(line + "\n");
            }
            return sb.toString();
        } catch (Exception e) {
            throw new ExchangeClientException(500, e.toString(), e);
        }
    }
    private JsonObject getReponseJsonObject(HttpURLConnection connection) throws IOException, ExchangeClientException {
        JsonParser parser = new JsonParser();
        JsonObject jsonResponse = parser.parse(getReponseString(connection)).getAsJsonObject();
        JsonElement element = jsonResponse.get("exchangeResponse");
        JsonObject exchangeResponse = element.getAsJsonObject();
        return exchangeResponse;
    }

    private void loadErrorReponseJsonObject(HttpURLConnection connection) throws ExchangeClientException {
        try (BufferedReader br = new BufferedReader(new InputStreamReader(connection.getErrorStream()))) {
            StringBuilder sb = new StringBuilder();
            String line;
            while ((line = br.readLine()) != null) {
                sb.append(line + "\n");
            }
            JsonParser parser = new JsonParser();
            JsonObject jsonResponse = parser.parse(sb.toString()).getAsJsonObject();
            JsonObject exchangeResponse = jsonResponse.get("exchangeResponse").getAsJsonObject();
            _responseErrorCode = exchangeResponse.get("code").getAsInt();
            _responseErrorMessage = exchangeResponse.get("exceptionMessage").getAsString();
        } catch (Exception e) {
            throw new ExchangeClientException(500, e.toString(), e);
        }
    }
    
    private String validateConfigOption(String configParameter) throws ExchangeClientException {
        if(_configManager.exists(configParameter)){
            return _configManager.getConfigValue(configParameter);
        }else{
            throw new ExchangeClientException(101, "Required config parameter "+configParameter+" not supplied");
        }
    }

    private static class HttpCodes {
        private static int HTTP_200_OK = 200;
        private static int HTTP_201_CREATED = 201;
        private static int HTTP_400_BAD_REQUEST = 400;
        private static int HTTP_401_UNAUTHORIZED = 401;
        private static int HTTP_404_NOTFOUND = 404;
        private static int HTTP_501_INTERNAL_SERVER_ERROR = 501;
    }
    
    private enum EnvironmentType {
        DEV("DEV", "Development or System Test"),
        UAT("UAT", "User acceptance testing or model office"),
        PROD("PROD", "Production/Live environment");

        private String _value;

        private String _displayName;

        EnvironmentType(String value, String name) {
            _value = value;
            _displayName = name;
        }

        public String getValue() {
            return _value;
        }

        public String getDisplayName() {
            return _displayName;
        }

        @Override
        public String toString() {
            return _displayName;
        }
    }
}
