/*
 * Copyright 2016 Digital Receipt Exchange Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.dreceiptx.client;

import net.dreceiptx.receipt.DigitalReceipt;
import net.dreceiptx.receipt.merchant.Merchant;
import net.dreceiptx.users.NewUser;
import net.dreceiptx.users.NewUserRegistrationResult;
import net.dreceiptx.users.User;
import net.dreceiptx.users.UserIdentifierType;
import net.dreceiptx.users.Users;
import net.dreceiptx.client.exception.ExchangeClientException;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public interface ExchangeClient {

    /**
     * @param identifierType UserIdentifierType
     * @param identifier String
     * @return The matching users will be returned if it is found otherwise null
     * @throws ExchangeClientException will be thrown if problem calling the service.
     * @throws UnsupportedEncodingException will be thrown if problem calling the service.
     */
    User searchUser(UserIdentifierType identifierType, String identifier) throws ExchangeClientException, UnsupportedEncodingException;

    /**
     * @param identifierType UserIdentifierType
     * @param userIdentifiers  list of user identifiers
     * @return The matching users will be returned if it is found otherwise null
     * @throws ExchangeClientException will be thrown if problem calling the service.
     */
    Users searchUsers(UserIdentifierType identifierType, ArrayList<String> userIdentifiers) throws ExchangeClientException;
    
    /**
     * @param receiptPostRequest ReceiptPostRequest
     * @return transactionId
     * @throws ExchangeClientException will be thrown if problem calling the service.
     */
    String sendProductionReceipt(ReceiptPostRequest receiptPostRequest) throws ExchangeClientException;

    /**
     * @param receiptPostRequest ReceiptPostRequest
     * @return transactionId
     * @throws ExchangeClientException will be thrown if problem calling the service.
     */
    String sendDryRunReceipt(ReceiptPostRequest receiptPostRequest) throws ExchangeClientException;

    /**
     * @param newUser NewUser
     * @return newUserRegistrationResult
     * @throws ExchangeClientException will be thrown if problem calling the service.
     */
    public NewUserRegistrationResult registerNewUser(NewUser newUser) throws ExchangeClientException;
    
    /**
     * @param newUsers list of users to be registered
     * @return map of userid to NewUserRegistrationResult
     * @throws ExchangeClientException will be thrown if problem calling the service.
     */
    public Map<String, NewUserRegistrationResult> registerNewUser(List<NewUser> newUsers) throws ExchangeClientException;
    
    /**
     * @param receiptId String
     * @return DigitalReceipt
     * @throws ExchangeClientException will be thrown if problem calling the service.
     */
    DigitalReceipt lookupReceipt(String receiptId) throws ExchangeClientException;

    /**
     * @param receiptId String
     * @return boolean
     * @throws ExchangeClientException will be thrown if problem calling the service.
     */
    boolean downloadReceiptPDF(String receiptId) throws ExchangeClientException;


    /**
     * @param Id String
     * @return Merchant
     * @throws ExchangeClientException will be thrown if problem calling the service.
     */
    Merchant lookupMerchant(String Id) throws ExchangeClientException;

}
