package net.dreceiptx.client;

import net.dreceiptx.client.exception.ExchangeClientException;

/**
 * The EnvVarExchangeCredentials is a simply mechanism to create ExchangeCredentials
 * from Environment Variables.
 */
public final class EnvVarExchangeCredentials implements ExchangeCredentials{
    private String _requesterId;
    private String _apiKey;
    private String _apiSecret;

    /**
     * Creates instance using the credentials from environment variables. The following
     * environment varaibles MUST be supplied otherwise the ExchangeClientException will
     * be thrown:
     * <ul>
     *     <li>DRX_REQUESTER_ID</li>
     *     <li>DRX_API_KEY</li>
     *     <li>DRX_API_SECRET</li>
     * </ul>
     * @throws ExchangeClientException will be thrown if any of the required environment variables
     * are not defined
     */
    public EnvVarExchangeCredentials() throws ExchangeClientException {
        _requesterId = getEnvVarOrThrow("DRX_REQUESTER_ID");
        _apiKey = getEnvVarOrThrow("DRX_API_KEY");
        _apiSecret = getEnvVarOrThrow("DRX_API_SECRET");
    }

    private String getEnvVarOrThrow(String envVar) throws ExchangeClientException {
        String value = System.getenv(envVar);
        if(value == null || value.isEmpty()) {
            throw new ExchangeClientException(envVar + " can not be null or empty");
        }
        return value;
    }

    @Override
    public String getRequesterId() {
        return _requesterId;
    }

    @Override
    public String getAPIKey() {
        return _apiKey;
    }

    @Override
    public String getAPISecret() {
        return _apiSecret;
    }
}
