/*
 * Copyright 2016 Digital Receipt Exchange Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.dreceiptx.receipt;

import net.dreceiptx.SDK;
import net.dreceiptx.client.ReceiptPostRequest;
import net.dreceiptx.config.ConfigKeys;
import net.dreceiptx.config.ConfigManager;
import net.dreceiptx.config.Location;
import net.dreceiptx.receipt.allowanceCharge.ReceiptAllowanceCharge;
import net.dreceiptx.receipt.common.Currency;
import net.dreceiptx.receipt.document.ReceiptContact;
import net.dreceiptx.receipt.document.ReceiptContactType;
import net.dreceiptx.receipt.document.StandardBusinessDocumentHeader;
import net.dreceiptx.receipt.invoice.Invoice;
import net.dreceiptx.receipt.lineitem.LineItem;
import net.dreceiptx.receipt.lineitem.StandardLineItem;
import net.dreceiptx.receipt.serialization.json.DigitalReceiptSerializer;
import net.dreceiptx.receipt.serialization.json.InvoiceSerializer;
import net.dreceiptx.receipt.serialization.json.LineItemSerializer;
import net.dreceiptx.receipt.serialization.json.PaymentReceiptsSerializer;
import net.dreceiptx.receipt.settlement.PaymentMethodType;
import net.dreceiptx.receipt.settlement.PaymentReceipt;
import net.dreceiptx.receipt.tax.Tax;
import net.dreceiptx.receipt.tax.TaxCategory;
import net.dreceiptx.receipt.tax.TaxCode;
import net.dreceiptx.users.UserIdentifierType;
import net.dreceiptx.client.exception.ExchangeClientException;
import net.dreceiptx.receipt.validation.ReceiptValidation;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.annotations.SerializedName;
import com.google.gson.reflect.TypeToken;
import net.dreceiptx.receipt.common.*;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;

public class DigitalReceiptBuilder {
    @SerializedName("standardBusinessDocumentHeader")
    private StandardBusinessDocumentHeader _standardBusinessDocumentHeader;
    @SerializedName("invoice")
    private Invoice _invoice;
    @SerializedName("paymentReceipts")
    private List<PaymentReceipt> _paymentReceipts;

    private final static AtomicInteger _paymentReceiptId = new AtomicInteger(1);
    private String _dRxGLN;
    private String _merchantGLN;
    private String _userGUID;
    private Location _location;
    private TaxCategory _defaultTaxCategory = TaxCategory.APPLICABLE;
    private TaxCode _defaultTaxCode = TaxCode.GoodsAndServicesTax;
    private Currency _defaultCurrency = Currency.USDollar;
    private Language _defaultLanguage = Language.English;
    private TimeZone _defaultTimeZone = TimeZone.getTimeZone("GMT");
    private Country _defaultCountry;
    private boolean isDryRunReceipt = false;

    public DigitalReceiptBuilder(Location location) {
        _location = location;
        _defaultCountry = location.getCountry();
        _defaultLanguage = location.getLanguage();
        _defaultTimeZone = location.getTimeZone();
        _defaultCurrency = location.getCurrency();
        _defaultTaxCode = location.getGoodsAndServicesTaxCode();

        _standardBusinessDocumentHeader = new StandardBusinessDocumentHeader();
        _standardBusinessDocumentHeader.setdRxGLN(location.getRegion().getGLN());
        _standardBusinessDocumentHeader.getDocumentIdentification().setTypeVersion(SDK.LATEST_RECEIPT_VERSION);
        _standardBusinessDocumentHeader.getDocumentIdentification().setCreationDateAndTime(Calendar.getInstance().getTime());

        _paymentReceipts = new ArrayList<>();
        _invoice = new Invoice(location);
    }

    public DigitalReceiptBuilder(Location location, String merchantGLN) {
        this(location);
        _standardBusinessDocumentHeader.setMerchantGLN(merchantGLN);
    }

    public DigitalReceiptBuilder(ConfigManager configManager) throws ExchangeClientException {
        _defaultCountry = Country.codeOf(validateConfigOption(configManager, ConfigKeys.DefaultCountry));
        _defaultLanguage = (configManager.exists(ConfigKeys.DefaultLanguage) ? Language.codeOf(configManager.getConfigValue(ConfigKeys.DefaultLanguage)) : _defaultLanguage);
        _defaultTimeZone = TimeZone.getTimeZone(validateConfigOption(configManager, ConfigKeys.DefaultTimeZone));
        _defaultCurrency = Currency.codeOf(validateConfigOption(configManager, ConfigKeys.DefaultCurrency));
        _defaultTaxCategory = Enum.valueOf(TaxCategory.class, validateConfigOption(configManager, ConfigKeys.DefaultTaxCategory));
        _defaultTaxCode = Enum.valueOf(TaxCode.class, validateConfigOption(configManager, ConfigKeys.DefaultTaxCode));

        _standardBusinessDocumentHeader = new StandardBusinessDocumentHeader();
        if (configManager.exists(ConfigKeys.dRxGLN)) {
            _standardBusinessDocumentHeader.setdRxGLN(configManager.getConfigValue(ConfigKeys.dRxGLN));
        }
        if (configManager.exists(ConfigKeys.MerchantGLN)) {
            _standardBusinessDocumentHeader.setMerchantGLN(configManager.getConfigValue(ConfigKeys.MerchantGLN));
        }
        _standardBusinessDocumentHeader.getDocumentIdentification().setTypeVersion(
                configManager.exists(ConfigKeys.ReceiptVersion) ? configManager.getConfigValue(ConfigKeys.ReceiptVersion) : SDK.LATEST_RECEIPT_VERSION
        );
        _standardBusinessDocumentHeader.getDocumentIdentification().setCreationDateAndTime(Calendar.getInstance().getTime());
        _paymentReceipts = new ArrayList<>();
        _invoice = new Invoice(configManager);
    }


    public String getdRxGLN() {
        return _standardBusinessDocumentHeader.getdRxGLN();
    }

    public Location getLocation() {
        return _location;
    }

    public TaxCategory getDefaultTaxCategory() {
        return _defaultTaxCategory;
    }

    public TaxCode getDefaultTaxCode() {
        return _defaultTaxCode;
    }

    public Currency getDefaultCurrency() {
        return _defaultCurrency;
    }

    public Language getDefaultLanguage() {
        return _defaultLanguage;
    }

    public TimeZone getDefaultTimeZone() {
        return _defaultTimeZone;
    }

    public Country getDefaultCountry() {
        return _defaultCountry;
    }

    public void setDefaultTaxCategory(TaxCategory taxCategory) {
        this._defaultTaxCategory = taxCategory;
    }

    public void setDefaultTaxCode(TaxCode taxCode) {
        this._defaultTaxCode = taxCode;
    }

    public void setDefaultCurrency(Currency currency) {
        this._defaultCurrency = currency;
    }

    public void setDefaultLanguage(Language language) {
        this._defaultLanguage = language;
    }

    public void setDefaultTimeZone(TimeZone timeZone) {
        this._defaultTimeZone = timeZone;
    }

    public void setDefaultCountry(Country country) {
        this._defaultCountry = country;
    }

    public boolean isDryRunReceipt() {
        return isDryRunReceipt;
    }

    public DigitalReceiptBuilder setDryRunReceipt(boolean dryRunReceipt) {
        isDryRunReceipt = dryRunReceipt;
        return this;
    }

    public DigitalReceiptBuilder setMerchantGLN(String merchantGLN) {
        _standardBusinessDocumentHeader.setMerchantGLN(merchantGLN);
        return this;
    }

    public String getMerchantGLN() {
        return _standardBusinessDocumentHeader.getMerchantGLN();
    }

    public DigitalReceiptBuilder setUserGUID(UserIdentifierType userIdentifierType, String userIdentifierValue) throws ExchangeClientException {
        _userGUID = userIdentifierType.getValue() + ":" + userIdentifierValue;
        _standardBusinessDocumentHeader.setUserIdentifier(_userGUID);
        return this;
    }

    public DigitalReceiptBuilder setMerchantReference(String merchantReference) {
        _standardBusinessDocumentHeader.getDocumentIdentification().setInstanceIdentifier(merchantReference);
        if (_invoice.getInvoiceIdentification() == null) {
            _invoice.setInvoiceIdentification(merchantReference);
        }
        return this;
    }

    public DigitalReceiptBuilder setReceiptCurrency(Currency currency) {
        _defaultCurrency = currency;
        return this;
    }

    public DigitalReceiptBuilder setReceiptLanguage(Language languageCode) {
        _defaultLanguage = languageCode;
        return this;
    }

    public DigitalReceiptBuilder setReceiptDateTime(Date invoiceDate) {
        _invoice.setCreationDateTime(invoiceDate);
        return this;
    }

    public DigitalReceiptBuilder setPurchaseOrderNumber(String purchaseOrder) {
        _invoice.setPurchaseOrder(purchaseOrder);
        return this;
    }

    public DigitalReceiptBuilder setCustomerReferenceNumber(String customerReference) {
        _invoice.setCustomerReference(customerReference);
        return this;
    }

    public DigitalReceiptBuilder setBillingInformation(String name) {
        _invoice.setBillingInformation(name);
        return this;
    }

    public DigitalReceiptBuilder setSalesOrderReference(String salesOrderReference) {
        _invoice.setSalesOrderReference(salesOrderReference);
        return this;
    }

    public DigitalReceiptBuilder addClientRecipientContact(String name, String email, String phone) {
        addRMSContact(ReceiptContactType.RECIPIENT_CONTACT, name, email, phone);
        return this;
    }

    public DigitalReceiptBuilder addClientRecipientContact(String name, String email) {
        addRMSContact(ReceiptContactType.RECIPIENT_CONTACT, name, email, null);
        return this;
    }

    public DigitalReceiptBuilder addClientRecipientContact(String name) {
        addRMSContact(ReceiptContactType.RECIPIENT_CONTACT, name, null, null);
        return this;
    }

    public DigitalReceiptBuilder addClientPurchasingContact(String name, String email, String phone) {
        addRMSContact(ReceiptContactType.PURCHASING_CONTACT, name, email, phone);
        return this;
    }

    public DigitalReceiptBuilder addClientPurchasingContact(String name, String email) {
        addRMSContact(ReceiptContactType.PURCHASING_CONTACT, name, email, null);
        return this;
    }

    public DigitalReceiptBuilder addClientPurchasingContact(String name) {
        addRMSContact(ReceiptContactType.PURCHASING_CONTACT, name, null, null);
        return this;
    }

    private DigitalReceiptBuilder addRMSContact(ReceiptContactType type, String name, String email, String phone) {
        ReceiptContact rmsContact = new ReceiptContact(type, name);
        if (email != null) {
            rmsContact.addEmailAddress(email);
        }
        if (phone != null) {
            rmsContact.addTelephoneNumber(phone);
        }
        _standardBusinessDocumentHeader.addRMSContact(rmsContact);
        return this;
    }

    public DigitalReceiptBuilder addMerchantCustomerRelationsContact(String name, String email, String phone) {
        addMerchantContact(ReceiptContactType.CUSTOMER_RELATIONS, name, email, phone);
        return this;
    }

    public DigitalReceiptBuilder addMerchantCustomerRelationsContact(String name, String email) {
        addMerchantContact(ReceiptContactType.CUSTOMER_RELATIONS, name, email, null);
        return this;
    }

    public DigitalReceiptBuilder addMerchantCustomerRelationsContact(String name) {
        addMerchantContact(ReceiptContactType.CUSTOMER_RELATIONS, name, null, null);
        return this;
    }

    public DigitalReceiptBuilder addMerchantDeliveryContact(String name, String email, String phone) {
        addMerchantContact(ReceiptContactType.DELIVERY_CONTACT, name, email, phone);
        return this;
    }

    public DigitalReceiptBuilder addMerchantDeliveryContact(String name, String email) {
        addMerchantContact(ReceiptContactType.DELIVERY_CONTACT, name, email, null);
        return this;
    }

    public DigitalReceiptBuilder addMerchantDeliveryContact(String name) {
        addMerchantContact(ReceiptContactType.DELIVERY_CONTACT, name, null, null);
        return this;
    }

    public DigitalReceiptBuilder addMerchantSalesAssistantContact(String name, String email, String phone) {
        addMerchantContact(ReceiptContactType.SALES_ADMINISTRATION, name, email, phone);
        return this;
    }

    public DigitalReceiptBuilder addMerchantSalesAssistantContact(String name, String email) {
        addMerchantContact(ReceiptContactType.SALES_ADMINISTRATION, name, email, null);
        return this;
    }

    public DigitalReceiptBuilder addMerchantSalesAssistantContact(String name) {
        addMerchantContact(ReceiptContactType.SALES_ADMINISTRATION, name, null, null);
        return this;
    }

    private DigitalReceiptBuilder addMerchantContact(ReceiptContactType type, String name, String email, String phone) {
        ReceiptContact merchantContact = new ReceiptContact(type, name);
        if (email != null) {
            merchantContact.addEmailAddress(email);
        }
        if (phone != null) {
            merchantContact.addTelephoneNumber(phone);
        }
        _standardBusinessDocumentHeader.addMerchantContact(merchantContact);
        return this;
    }

    public DigitalReceiptBuilder setReceiptNumber(String receiptNumber) {
        _invoice.setInvoiceIdentification(receiptNumber);

        if (_standardBusinessDocumentHeader.getDocumentIdentification().getInstanceIdentifier() == null) {
            _standardBusinessDocumentHeader.getDocumentIdentification().setInstanceIdentifier(receiptNumber);
        }
        return this;
    }

    public DigitalReceiptBuilder addLineItem(LineItem lineItem, Tax tax) {
        lineItem.addTax(tax);
        for (Tax lineItemTax : lineItem.getTaxes()) {
            configureTax(lineItemTax);
        }
        _invoice.addLineItem(lineItem);
        return this;
    }

    public DigitalReceiptBuilder addLineItem(LineItem lineItem) {
        for (Tax lineItemTax : lineItem.getTaxes()) {
            configureTax(lineItemTax);
        }
        _invoice.addLineItem(lineItem);
        return this;
    }

    public DigitalReceiptBuilder addLineItem(String brand, String name, int quantity, double price) {
        LineItem lineItem = new StandardLineItem(brand, name, "", quantity, price);
        _invoice.addLineItem(lineItem);
        return this;
    }

    public DigitalReceiptBuilder addLineItem(String brand, String name, int quantity, double price, Tax tax) {
        LineItem lineItem = new StandardLineItem(brand, name, "", quantity, price);
        lineItem.addTax(configureTax(tax));
        _invoice.addLineItem(lineItem);
        return this;
    }

    public DigitalReceiptBuilder addLineItem(String brand, String name, String description, int quantity, double price) {
        LineItem lineItem = new StandardLineItem(brand, name, description, quantity, price);
        _invoice.addLineItem(lineItem);
        return this;
    }

    public DigitalReceiptBuilder addLineItem(String brand, String name, String description, int quantity, double price, Tax tax) {
        LineItem lineItem = new StandardLineItem(brand, name, description, quantity, price);
        lineItem.addTax(configureTax(tax));
        _invoice.addLineItem(lineItem);
        return this;
    }

    public DigitalReceiptBuilder addLineItem(String brand, String name, String description, int quantity, double price, double taxRate, boolean taxInclusive) {
        LineItem lineItem;

        if (taxInclusive) {
            BigDecimal netPrice = BigDecimal.valueOf(price).multiply((BigDecimal.valueOf(1).subtract(BigDecimal.valueOf(taxRate))));
            BigDecimal total = BigDecimal.valueOf(quantity).multiply(netPrice);
            Tax tax = new Tax(_defaultTaxCategory, _defaultTaxCode, total.doubleValue(), taxRate);
            lineItem = new StandardLineItem(brand, name, description, quantity, price);
            lineItem.addTax(tax);
        } else {
            BigDecimal netPrice = BigDecimal.valueOf(price);
            BigDecimal total = BigDecimal.valueOf(quantity).multiply(netPrice);
            Tax tax = new Tax(_defaultTaxCategory, _defaultTaxCode, total.doubleValue(), taxRate);
            lineItem = new StandardLineItem(brand, name, description, quantity, price);
            lineItem.addTax(tax);
        }

        _invoice.addLineItem(lineItem);
        return this;
    }

    public DigitalReceiptBuilder addPaymentReceipt(PaymentMethodType paymentMethodCode, Double paymentAmount) {
        PaymentReceipt paymentReceipt = new PaymentReceipt(paymentMethodCode, paymentAmount);
        paymentReceipt.setSettlementCurrency(_defaultCurrency);
        paymentReceipt.setId(_paymentReceiptId.getAndIncrement());
        _paymentReceipts.add(paymentReceipt);
        return this;
    }

    public DigitalReceiptBuilder addPaymentReceipt(PaymentReceipt paymentReceipt) {
        paymentReceipt.setId(_paymentReceiptId.getAndIncrement());
        paymentReceipt.setSettlementCurrency(_defaultCurrency);
        _paymentReceipts.add(paymentReceipt);
        paymentReceipt.getId();
        return this;
    }

    public DigitalReceiptBuilder addGeneralDiscount(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.GeneralDiscount(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addGeneralDiscount(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.GeneralDiscount(amount, description, configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder addTip(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.Tip(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addTip(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.Tip(amount, description, configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder addPackagingFee(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.PackagingFee(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addPackagingFee(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.PackagingFee(amount, description, configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder addDeliveryFee(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.DeliveryFee(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addDeliveryFee(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.DeliveryFee(amount, description, configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder addFreightFee(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.FreightFee(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addFreightFee(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.FreightFee(amount, description, configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder addProcessingFee(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.ProcessingFee(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addProcessingFee(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.ProcessingFee(amount, description, configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder addBookingFee(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.BookingFee(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addBookingFee(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.BookingFee(amount, description, configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder addAdminFee(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.AdminFee(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addAdminFee(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.AdminFee(amount, description, this.configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder addAmendmentFee(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.AmendmentFee(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addAmendmentFee(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.AmendmentFee(amount, description, this.configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder addServiceFee(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.ServiceFee(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addServiceFee(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.ServiceFee(amount, description, this.configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder addReturnOrCancellationFee(double amount, String description) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.ReturnOrCancellationFee(amount, description));
        return this;
    }

    public DigitalReceiptBuilder addReturnOrCancellationFee(double amount, String description, Tax tax) {
        _invoice.addAllowanceOrCharge(ReceiptAllowanceCharge.ReturnOrCancellationFee(amount, description, this.configureTax(tax)));
        return this;
    }

    public DigitalReceiptBuilder setDeliveryInformation(DeliveryInformation deliveryInformation) {
        _invoice.setDestinationInformation(deliveryInformation.getLocationInformation());
        _invoice.getAllowanceOrCharges().addAll(deliveryInformation.getDeliveryFees());
        _invoice.setDespatchInformation(deliveryInformation.getDespatchInformation());
        return this;
    }

    public DigitalReceiptBuilder setDeliveryAddress(Address address) {
        _invoice.getDestinationInformation().setAddress(address);
        return this;
    }

    public DigitalReceiptBuilder setDeliveryAddress(Address address, Contact contact) {
        _invoice.getDestinationInformation().setAddress(address);
        _invoice.getDestinationInformation().addContact(contact);
        return this;
    }

    public DigitalReceiptBuilder setDestinationCoordinates(GeographicalCoordinates geographicalCoordinates) {
        _invoice.getDestinationInformation().setGeographicalCoordinates(geographicalCoordinates);
        return this;
    }

    public DigitalReceiptBuilder setDeliveryDate(Date deliverDate) {
        _invoice.getDespatchInformation().setDeliveryDate(deliverDate);
        return this;
    }

    public DigitalReceiptBuilder setOriginAddress(Address address) {
        _invoice.getOriginInformation().setAddress(address);
        return this;
    }

    public DigitalReceiptBuilder setOriginAddress(Address address, Contact contact) {
        _invoice.getOriginInformation().setAddress(address);
        _invoice.getOriginInformation().addContact(contact);
        return this;
    }

    public DigitalReceiptBuilder setOriginCoordinates(GeographicalCoordinates geographicalCoordinates) {
        _invoice.getOriginInformation().setGeographicalCoordinates(geographicalCoordinates);
        return this;
    }

    public DigitalReceiptBuilder validate() {
        ReceiptValidation receiptValidation = new ReceiptValidation();
        _standardBusinessDocumentHeader.validate(receiptValidation);
        _invoice.validate(receiptValidation);
        return this;
    }

    private Tax configureTax(Tax tax) {
        if (tax.getTaxCategory() == null) {
            tax.setTaxCategory(_defaultTaxCategory);
        }

        if (tax.getTaxCode() == null) {
            tax.setTaxCode(_defaultTaxCode);
        }
        return tax;
    }

    public BigDecimal getReceiptTotal(){
        return _invoice.getTotal();
    }

    public BigDecimal getReceiptGSTTotal(){
        return _invoice.getTaxesTotal(TaxCode.GoodsAndServicesTax);
    }

    public BigDecimal getReceiptTaxTotal(){
        return _invoice.getTaxesTotal();
    }

    public ReceiptPostRequest buildReceiptPostRequest() {
        DigitalReceiptSerializer digitalReceiptSerializer = new DigitalReceiptSerializer();
        digitalReceiptSerializer.setInvoice(_invoice);
        digitalReceiptSerializer.setStandardBusinessDocumentHeader(_standardBusinessDocumentHeader);
        digitalReceiptSerializer.setPaymentReceipts(_paymentReceipts);
        ReceiptPostRequest receiptPostRequest = new ReceiptPostRequest();
        receiptPostRequest.setDigitalReceiptSerializer(digitalReceiptSerializer);
        receiptPostRequest.setDryRun(isDryRunReceipt);
        receiptPostRequest.setCurrency(_defaultCurrency);
        receiptPostRequest.setLocation(_location);
        receiptPostRequest.setUserGUID(_userGUID);
        receiptPostRequest.setdRxGLN(_dRxGLN);
        return receiptPostRequest;
    }

    public String buildJson() {
        return buildReceiptPostRequest().getJsonPayloadContent();
    }

    private String validateConfigOption(ConfigManager configManager, String configParameter) throws ExchangeClientException {
        if (configManager.exists(configParameter)) {
            return configManager.getConfigValue(configParameter);
        } else {
            throw new ExchangeClientException(101, "Required config parameter " + configParameter + " not supplied");
        }
    }
}