/*
 * Copyright 2016 Digital Receipt Exchange Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.dreceiptx.config;

import com.google.gson.annotations.SerializedName;
import net.dreceiptx.receipt.common.Country;
import net.dreceiptx.receipt.common.Currency;
import net.dreceiptx.receipt.common.Language;
import net.dreceiptx.receipt.tax.TaxCode;

import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;

public enum Location {
    @SerializedName("AUS")
    Australia(
            "AUS",
            Country.Australia,
            Currency.AustralianDollar,
            Region.Australasia,
            TaxCode.GoodsAndServicesTax,
            Language.English,
            TimeZone.getTimeZone("Australia/Sydney")),
    @SerializedName("CAN")
    Canada(
            "CAN",
            Country.Canada,
            Currency.CanadianDollar,
            Region.Canada,
            TaxCode.GoodsAndServicesTax,
            Language.English,
            TimeZone.getTimeZone("Canada/Eastern")),
    @SerializedName("FRA")
    France(
            "FRA",
            Country.France,
            Currency.Euro,
            Region.EuroWest,
            TaxCode.ValueAddedTax,
            Language.French,
            TimeZone.getTimeZone("Europe/Paris")),
    @SerializedName("IRE")
    Ireland(
            "IRE",
            Country.Ireland,
            Currency.Euro,
            Region.EuroWest,
            TaxCode.ValueAddedTax,
            Language.English,
            TimeZone.getTimeZone("Europe/Dublin")),
    @SerializedName("NZL")
    NewZealand(
            "NZL",
            Country.NewZealand,
            Currency.NewZealandDollar,
            Region.Australasia,
            TaxCode.GoodsAndServicesTax,
            Language.English,
            TimeZone.getTimeZone("NZ")),
    @SerializedName("GBR")
    UnitedKingdom(
            "GBR",
            Country.UnitedKingdom,
            Currency.BritishSterling,
            Region.EuroWest,
            TaxCode.ValueAddedTax,
            Language.English,
            TimeZone.getTimeZone("Europe/London")),
    @SerializedName("USEastCoast")
    USEastCoast(
            "USEastCoast",
            Country.UnitedStatesOfAmerica,
            Currency.USDollar,
            Region.USEast,
            TaxCode.ValueAddedTax,
            Language.English,
            TimeZone.getTimeZone("America/New_York")),
    @SerializedName("USWestCoast")
    USWestCoast(
            "USWestCoast",
            Country.UnitedStatesOfAmerica,
            Currency.USDollar,
            Region.USWest,
            TaxCode.ValueAddedTax,
            Language.English,
            TimeZone.getTimeZone("America/Los_Angeles"));

    private String _code;

    private Country _country;

    private Currency _currency;

    private TaxCode _salesTaxCode;

    private Region _region;

    private Language _language;

    private TimeZone _timeZone;

    private static Map<String, Location> enums = new HashMap<String, Location>();

    static {
        for (Location country : Location.values()) {
            enums.put(country._code, country);
        }
    }

    Location(String code, Country country, Currency name, Region region, TaxCode salesTaxCode, Language language, TimeZone timeZone) {
        _code = code;
        _country = country;
        _currency = name;
        _region = region;
        _salesTaxCode = salesTaxCode;
        _language = language;
        _timeZone = timeZone;
    }

    public String getCode() {
        return _code;
    }

    public Country getCountry() {
        return _country;
    }

    public Currency getCurrency() {
        return _currency;
    }

    public Region getRegion() {
        return _region;
    }

    public TaxCode getGoodsAndServicesTaxCode() {
        return _salesTaxCode;
    }

    public Language getLanguage() {
        return _language;
    }

    public TimeZone getTimeZone() {
        return _timeZone;
    }

    @Override
    public String toString() {
        return _code;
    }

    private Location(final String code) { _code = code; }

    public static Location codeOf(String code) {
        return enums.get(code);
    }
}
