/*
	Apache License
	Version 2.0, January 2004
	http://www.apache.org/licenses/

	TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION

	1. Definitions.

	"License" shall mean the terms and conditions for use, reproduction, and
	distribution as defined by Sections 1 through 9 of this document.

	"Licensor" shall mean the copyright owner or entity authorized by the copyright
	owner that is granting the License.

	"Legal Entity" shall mean the union of the acting entity and all other entities
	that control, are controlled by, or are under common control with that entity.
	For the purposes of this definition, "control" means (i) the power, direct or
	indirect, to cause the direction or management of such entity, whether by
	contract or otherwise, or (ii) ownership of fifty percent (50%) or more of the
	outstanding shares, or (iii) beneficial ownership of such entity.

	"You" (or "Your") shall mean an individual or Legal Entity exercising
	permissions granted by this License.

	"Source" form shall mean the preferred form for making modifications, including
	but not limited to software source code, documentation source, and configuration
	files.

	"Object" form shall mean any form resulting from mechanical transformation or
	translation of a Source form, including but not limited to compiled object code,
	generated documentation, and conversions to other media types.

	"Work" shall mean the work of authorship, whether in Source or Object form, made
	available under the License, as indicated by a copyright notice that is included
	in or attached to the work (an example is provided in the Appendix below).

	"Derivative Works" shall mean any work, whether in Source or Object form, that
	is based on (or derived from) the Work and for which the editorial revisions,
	annotations, elaborations, or other modifications represent, as a whole, an
	original work of authorship. For the purposes of this License, Derivative Works
	shall not include works that remain separable from, or merely link (or bind by
	name) to the interfaces of, the Work and Derivative Works thereof.

	"Contribution" shall mean any work of authorship, including the original version
	of the Work and any modifications or additions to that Work or Derivative Works
	thereof, that is intentionally submitted to Licensor for inclusion in the Work
	by the copyright owner or by an individual or Legal Entity authorized to submit
	on behalf of the copyright owner. For the purposes of this definition,
	"submitted" means any form of electronic, verbal, or written communication sent
	to the Licensor or its representatives, including but not limited to
	communication on electronic mailing lists, source code control systems, and
	issue tracking systems that are managed by, or on behalf of, the Licensor for
	the purpose of discussing and improving the Work, but excluding communication
	that is conspicuously marked or otherwise designated in writing by the copyright
	owner as "Not a Contribution."

	"Contributor" shall mean Licensor and any individual or Legal Entity on behalf
	of whom a Contribution has been received by Licensor and subsequently
	incorporated within the Work.

	2. Grant of Copyright License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable copyright license to reproduce, prepare Derivative Works of,
	publicly display, publicly perform, sublicense, and distribute the Work and such
	Derivative Works in Source or Object form.

	3. Grant of Patent License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable (except as stated in this section) patent license to make, have
	made, use, offer to sell, sell, import, and otherwise transfer the Work, where
	such license applies only to those patent claims licensable by such Contributor
	that are necessarily infringed by their Contribution(s) alone or by combination
	of their Contribution(s) with the Work to which such Contribution(s) was
	submitted. If You institute patent litigation against any entity (including a
	cross-claim or counterclaim in a lawsuit) alleging that the Work or a
	Contribution incorporated within the Work constitutes direct or contributory
	patent infringement, then any patent licenses granted to You under this License
	for that Work shall terminate as of the date such litigation is filed.

	4. Redistribution.

	You may reproduce and distribute copies of the Work or Derivative Works thereof
	in any medium, with or without modifications, and in Source or Object form,
	provided that You meet the following conditions:

	You must give any other recipients of the Work or Derivative Works a copy of
	this License; and
	You must cause any modified files to carry prominent notices stating that You
	changed the files; and
	You must retain, in the Source form of any Derivative Works that You distribute,
	all copyright, patent, trademark, and attribution notices from the Source form
	of the Work, excluding those notices that do not pertain to any part of the
	Derivative Works; and
	If the Work includes a "NOTICE" text file as part of its distribution, then any
	Derivative Works that You distribute must include a readable copy of the
	attribution notices contained within such NOTICE file, excluding those notices
	that do not pertain to any part of the Derivative Works, in at least one of the
	following places: within a NOTICE text file distributed as part of the
	Derivative Works; within the Source form or documentation, if provided along
	with the Derivative Works; or, within a display generated by the Derivative
	Works, if and wherever such third-party notices normally appear. The contents of
	the NOTICE file are for informational purposes only and do not modify the
	License. You may add Your own attribution notices within Derivative Works that
	You distribute, alongside or as an addendum to the NOTICE text from the Work,
	provided that such additional attribution notices cannot be construed as
	modifying the License.
	You may add Your own copyright statement to Your modifications and may provide
	additional or different license terms and conditions for use, reproduction, or
	distribution of Your modifications, or for any such Derivative Works as a whole,
	provided Your use, reproduction, and distribution of the Work otherwise complies
	with the conditions stated in this License.

	5. Submission of Contributions.

	Unless You explicitly state otherwise, any Contribution intentionally submitted
	for inclusion in the Work by You to the Licensor shall be under the terms and
	conditions of this License, without any additional terms or conditions.
	Notwithstanding the above, nothing herein shall supersede or modify the terms of
	any separate license agreement you may have executed with Licensor regarding
	such Contributions.

	6. Trademarks.

	This License does not grant permission to use the trade names, trademarks,
	service marks, or product names of the Licensor, except as required for
	reasonable and customary use in describing the origin of the Work and
	reproducing the content of the NOTICE file.

	7. Disclaimer of Warranty.

	Unless required by applicable law or agreed to in writing, Licensor provides the
	Work (and each Contributor provides its Contributions) on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied,
	including, without limitation, any warranties or conditions of TITLE,
	NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A PARTICULAR PURPOSE. You are
	solely responsible for determining the appropriateness of using or
	redistributing the Work and assume any risks associated with Your exercise of
	permissions under this License.

	8. Limitation of Liability.

	In no event and under no legal theory, whether in tort (including negligence),
	contract, or otherwise, unless required by applicable law (such as deliberate
	and grossly negligent acts) or agreed to in writing, shall any Contributor be
	liable to You for damages, including any direct, indirect, special, incidental,
	or consequential damages of any character arising as a result of this License or
	out of the use or inability to use the Work (including but not limited to
	damages for loss of goodwill, work stoppage, computer failure or malfunction, or
	any and all other commercial damages or losses), even if such Contributor has
	been advised of the possibility of such damages.

	9. Accepting Warranty or Additional Liability.

	While redistributing the Work or Derivative Works thereof, You may choose to
	offer, and charge a fee for, acceptance of support, warranty, indemnity, or
	other liability obligations and/or rights consistent with this License. However,
	in accepting such obligations, You may act only on Your own behalf and on Your
	sole responsibility, not on behalf of any other Contributor, and only if You
	agree to indemnify, defend, and hold each Contributor harmless for any liability
	incurred by, or claims asserted against, such Contributor by reason of your
	accepting any such warranty or additional liability.

	END OF TERMS AND CONDITIONS

	APPENDIX: How to apply the Apache License to your work

	To apply the Apache License to your work, attach the following boilerplate
	notice, with the fields enclosed by brackets "{}" replaced with your own
	identifying information. (Don't include the brackets!) The text should be
	enclosed in the appropriate comment syntax for the file format. We also
	recommend that a file or class name and description of purpose be included on
	the same "printed page" as the copyright notice for easier identification within
	third-party archives.

	   Copyright 2020 t-io

	   Licensed under the Apache License, Version 2.0 (the "License");
	   you may not use this file except in compliance with the License.
	   You may obtain a copy of the License at

	     http://www.apache.org/licenses/LICENSE-2.0

	   Unless required by applicable law or agreed to in writing, software
	   distributed under the License is distributed on an "AS IS" BASIS,
	   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	   See the License for the specific language governing permissions and
	   limitations under the License.
*/
package org.tio.http.common;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tio.core.ChannelContext;
import org.tio.core.Node;
import org.tio.core.Tio;
import org.tio.http.common.HttpConst.RequestBodyFormat;
import org.tio.utils.SysConst;
import org.tio.utils.hutool.StrUtil;

import java.nio.charset.Charset;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @author tanyaowu
 */
public class HttpRequest extends HttpPacket {
	private static final long serialVersionUID = -3849253977016967211L;
	private static final Logger log = LoggerFactory.getLogger(HttpRequest.class);
	private static final AtomicLong ID_GEN = new AtomicLong();
	private final long id = ID_GEN.incrementAndGet();
	public RequestLine requestLine = null;
	public ChannelContext channelContext;
	public HttpConfig httpConfig;
	protected Map<String, String> headers = new HashMap<>();
	private boolean needForward = false;
	private boolean isForward = false;
	/**
	 * 请求参数
	 */
	private Map<String, Object[]> params = new HashMap<>();
	private Map<String, Cookie> cookieMap = null;
	private int contentLength;
	private String connection;
	private String bodyString;
	private RequestBodyFormat bodyFormat;
	private Charset charset = HttpConst.CHARSET;
	private Node remote = null;
	private String domain = null;
	private String host = null;
	/**
	 * 该HttpRequest对象的创建时间
	 */
	private long createTime = System.currentTimeMillis();
	private boolean closed = false;
	private Integer forwardCount = null;

	/**
	 * @author tanyaowu
	 * 2017年2月22日 下午4:14:40
	 */
	public HttpRequest(Node remote) {
		this.remote = remote;
	}

	public HttpRequest() {
	}

	/**
	 * @return the id
	 */
	public long getId() {
		return id;
	}

	/**
	 * 关闭连接
	 */
	public void close() {
		close(null);
	}

	/**
	 * 关闭连接
	 *
	 * @param remark
	 */
	public void close(String remark) {
		closed = true;
		Tio.remove(channelContext, remark);
	}

	public void addParam(String key, Object value) {
		if (value == null) {
			return;
		}
		Object[] existValue = params.get(key);
		if (existValue != null) {
			Object[] newExistValue = new Object[existValue.length + 1];
			if (value instanceof String) {
				newExistValue = new String[existValue.length + 1];
			} else if (value instanceof UploadFile) {
				newExistValue = new UploadFile[existValue.length + 1];
			}
			System.arraycopy(existValue, 0, newExistValue, 0, existValue.length);
			newExistValue[newExistValue.length - 1] = value;
			params.put(key, newExistValue);
		} else {
			Object[] newExistValue = null;
			if (value instanceof String) {
				newExistValue = new String[]{(String) value};
			} else if (value instanceof UploadFile) {
				newExistValue = new UploadFile[]{(UploadFile) value};
			}
			params.put(key, newExistValue);
		}
	}

	/**
	 * @param newPath
	 * @return
	 * @throws Exception
	 */
	public HttpResponse forward(String newPath) {
		if (StrUtil.contains(newPath, '?')) {
			requestLine.path = StrUtil.subBefore(newPath, "?", false);
			requestLine.queryString = StrUtil.subAfter(newPath, "?", false);
		} else {
			requestLine.path = newPath;
			requestLine.queryString = null;
		}

		if (forwardCount == null) {
			forwardCount = 1;
		} else {
			forwardCount++;
		}
		if (forwardCount > httpConfig.maxForwardCount) {
			log.error("forwardCount[{}] is too large, newPath:{}", forwardCount, newPath);
			this.close();
			return null;
		}
		this.needForward = true;
		return HttpResponse.NULL_RESPONSE;
	}

	/**
	 * @return the bodyFormat
	 */
	public RequestBodyFormat getBodyFormat() {
		return bodyFormat;
	}

	/**
	 * @param bodyFormat the bodyFormat to set
	 */
	public void setBodyFormat(RequestBodyFormat bodyFormat) {
		this.bodyFormat = bodyFormat;
	}

	/**
	 * 获取请求头中的User-Agent字段
	 *
	 * @return
	 * @author: tanyaowu
	 */
	public String getUserAgent() {
		return this.headers.get(HttpConst.RequestHeaderKey.User_Agent);
	}

	/**
	 * 获取请求头中的host字段，形如：www.t-io.org:8080, www.t-io.org等值
	 *
	 * @return
	 * @author: tanyaowu
	 */
	public String getHost() {
		if (host != null) {
			return host;
		}
		host = this.headers.get(HttpConst.RequestHeaderKey.Host);
		return host;
	}

	/**
	 * 获取真实的客户端ip
	 *
	 * @return
	 * @author tanyaowu
	 */
	public String getClientIp() {
		return remote.getIp();
	}

	public void addHeader(String key, String value) {
		headers.put(key, value);
	}

	public void addHeaders(Map<String, String> headers) {
		if (headers != null) {
			Set<Entry<String, String>> set = headers.entrySet();
			for (Entry<String, String> entry : set) {
				this.addHeader(entry.getKey(), entry.getValue());
			}
		}
	}

	/**
	 * 根据host字段，获取去除端口的纯域名部分的值，形如：www.t-io.org, t-io.org等值
	 *
	 * @return
	 * @author tanyaowu
	 */
	public String getDomain() {
		if (domain != null) {
			return domain;
		}
		if (StrUtil.isBlank(getHost())) {
			return null;
		}
		domain = StrUtil.subBefore(getHost(), ":", false);
		return domain;
	}

	/**
	 * @return the bodyString
	 */
	public String getBodyString() {
		return bodyString;
	}

	/**
	 * @param bodyString the bodyString to set
	 */
	public void setBodyString(String bodyString) {
		this.bodyString = bodyString;
	}

	/**
	 * @return the channelContext
	 */
	public ChannelContext getChannelContext() {
		return channelContext;
	}

	/**
	 * @param channelContext the channelContext to set
	 */
	public void setChannelContext(ChannelContext channelContext) {
		this.channelContext = channelContext;
	}

	/**
	 * @return the charset
	 */
	public Charset getCharset() {
		return charset;
	}

	/**
	 * @param charset the charset to set
	 */
	public void setCharset(Charset charset) {
		this.charset = charset;
	}

	/**
	 * @return the bodyLength
	 */
	public int getContentLength() {
		return contentLength;
	}

	/**
	 * @param contentLength the bodyLength to set
	 */
	public void setContentLength(int contentLength) {
		this.contentLength = contentLength;
	}

	/**
	 * @return the cookieMap
	 */
	public Map<String, Cookie> getCookieMap() {
		if (cookieMap != null) {
			return cookieMap;
		}
		if (headers == null || headers.isEmpty()) {
			return Collections.emptyMap();
		}
		String cookieLine = headers.get(HttpConst.RequestHeaderKey.Cookie);
		if (cookieLine == null) {
			return Collections.emptyMap();
		}
		this.cookieMap = new HashMap<>();
		Map<String, String> cookieLineMap = Cookie.getEqualMap(cookieLine);
		Set<Entry<String, String>> set = cookieLineMap.entrySet();
		for (Entry<String, String> cookieMapEntry : set) {
			HashMap<String, String> cookieOneMap = new HashMap<>();
			cookieOneMap.put(cookieMapEntry.getKey(), cookieMapEntry.getValue());
			Cookie cookie = Cookie.buildCookie(cookieOneMap, httpConfig);
			this.cookieMap.put(cookie.getName(), cookie);
		}
		return cookieMap;
	}

	/**
	 * @return the cookies
	 */
	public Collection<Cookie> getCookies() {
		Map<String, Cookie> cookieMaps = getCookieMap();
		if (cookieMaps == null) {
			return Collections.emptyList();
		}
		return cookieMaps.values();
	}

	public Cookie getCookie(String name) {
		Map<String, Cookie> cookieMaps = getCookieMap();
		if (cookieMaps == null) {
			return null;
		}
		return cookieMaps.get(name);
	}

	/**
	 * @return the httpConfig
	 */
	public HttpConfig getHttpConfig() {
		return httpConfig;
	}

	/**
	 * @param httpConfig the httpConfig to set
	 */
	public void setHttpConfig(HttpConfig httpConfig) {
		this.httpConfig = httpConfig;
	}

	public String getHeader(String key) {
		return headers.get(key);
	}

	/**
	 * @return the headers
	 */
	public Map<String, String> getHeaders() {
		return headers;
	}

	/**
	 * 设置好header后，会把cookie等头部信息也设置好
	 *
	 * @param headers the headers to set
	 */
	public void setHeaders(Map<String, String> headers) {
		this.headers = headers;
	}

	public void removeHeader(String key, String value) {
		headers.remove(key);
	}

	/**
	 * @return the isSupportGzip
	 */
	public boolean getIsSupportGzip() {
		String header = getHeader(HttpConst.RequestHeaderKey.Accept_Encoding);
		return header != null && header.contains(HttpConst.RequestHeaderValue.Accept.gzip);
	}

	/**
	 * @return the params
	 */
	public Map<String, Object[]> getParams() {
		return params;
	}

	/**
	 * @param params the params to set
	 */
	public void setParams(Map<String, Object[]> params) {
		this.params = params;
	}

	/**
	 * 把类型为数组的参数值转换成Object，相当于是取了数组的第一个值，便于业务开发（因为大部分参数值其实只有一个）
	 *
	 * @return
	 */
	public Map<String, Object> getParam() {
		Map<String, Object> paramMap = new HashMap<>();
		if (getParams() != null) {
			for (Entry<String, Object[]> entry : this.params.entrySet()) {
				Object[] param = entry.getValue();
				if (param != null && param.length > 0) {
					String key = entry.getKey();
					paramMap.put(key, param[0]);
				}
			}
		}
		return paramMap;
	}

	public Object getObject(String name) {
		if (StrUtil.isBlank(name)) {
			return null;
		}
		Object[] values = params.get(name);
		if (values != null && values.length > 0) {
			return values[0];
		}
		return null;
	}

	/**
	 * @param name
	 * @return
	 * @author: tanyaowu
	 */
	public String getParam(String name) {
		return (String) getObject(name);
	}

	/**
	 * 同getParam(String name)
	 *
	 * @param name
	 * @return
	 * @author tanyaowu
	 */
	public String getString(String name) {
		return getParam(name);
	}

	/**
	 * @param name
	 * @return
	 * @author tanyaowu
	 */
	public UploadFile getUploadFile(String name) {
		Object[] values = params.get(name);
		if (values != null && values.length > 0) {
			Object obj = values[0];
			return (UploadFile) obj;
		}
		return null;
	}

	/**
	 * @param name name
	 * @return Integer
	 */
	public Integer getInt(String name) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return null;
		}
		return Integer.parseInt(value);
	}

	/**
	 * 获取 int 参数
	 *
	 * @param name         name
	 * @param defaultValue 默认值
	 * @return int
	 */
	public int getInt(String name, int defaultValue) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return defaultValue;
		}
		try {
			return Integer.parseInt(value);
		} catch (NumberFormatException e) {
			return defaultValue;
		}
	}

	public Short getShort(String name) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return null;
		}
		return Short.parseShort(value);
	}

	public short getShort(String name, short defaultValue) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return defaultValue;
		}
		try {
			return Short.parseShort(value);
		} catch (NumberFormatException e) {
			return defaultValue;
		}
	}

	public Byte getByte(String name) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return null;
		}
		return Byte.parseByte(value);
	}

	public byte getByte(String name, byte defaultValue) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return defaultValue;
		}
		try {
			return Byte.parseByte(value);
		} catch (NumberFormatException e) {
			return defaultValue;
		}
	}

	public Long getLong(String name) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return null;
		}
		return Long.parseLong(value);
	}

	public long getLong(String name, long defaultValue) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return defaultValue;
		}
		try {
			return Long.parseLong(value);
		} catch (NumberFormatException e) {
			return defaultValue;
		}
	}

	public double getDouble(String name, double defaultValue) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return defaultValue;
		}
		try {
			return Double.parseDouble(value);
		} catch (NumberFormatException e) {
			return defaultValue;
		}
	}

	public Double getDouble(String name) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return null;
		}
		return Double.parseDouble(value);
	}

	public Float getFloat(String name) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return null;
		}
		return Float.parseFloat(value);
	}

	public float getFloat(String name, float defaultValue) {
		String value = getParam(name);
		if (StrUtil.isBlank(value)) {
			return defaultValue;
		}
		try {
			return Float.parseFloat(value);
		} catch (NumberFormatException e) {
			return defaultValue;
		}
	}

	/**
	 * @param name name
	 * @return Object array
	 */
	public Object[] getParamArray(String name) {
		return params.get(name);
	}

	public Node getRemote() {
		return remote;
	}

	public void setRemote(Node remote) {
		this.remote = remote;
	}

	/**
	 * @return the firstLine
	 */
	public RequestLine getRequestLine() {
		return requestLine;
	}

	/**
	 * @param requestLine the requestLine to set
	 */
	public void setRequestLine(RequestLine requestLine) {
		this.requestLine = requestLine;
	}

	/**
	 * @return
	 */
	@Override
	public String logstr() {
		String str = "\r\n请求ID_" + getId() + SysConst.CRLF + getHeaderString();
		if (null != getBodyString()) {
			str += getBodyString();
		}
		return str;
	}

	public long getCreateTime() {
		return createTime;
	}

	public void setCreateTime(long createTime) {
		this.createTime = createTime;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append(this.requestLine.toString()).append(SysConst.CRLF);
		if (this.getHeaderString() != null) {
			sb.append(this.getHeaderString()).append(SysConst.CRLF);
		}
		if (this.getBodyString() != null) {
			sb.append(this.getBodyString());
		}
		return sb.toString();
	}

	public boolean isClosed() {
		return closed;
	}

	public void setClosed(boolean closed) {
		this.closed = closed;
	}

	/**
	 * @return the connection
	 */
	public String getConnection() {
		return connection;
	}

	/**
	 * @param connection the connection to set
	 */
	public void setConnection(String connection) {
		this.connection = connection;
	}

	public String getReferer() {
		return getHeader(HttpConst.RequestHeaderKey.Referer);
	}

	public boolean isNeedForward() {
		return needForward;
	}

	public void setNeedForward(boolean needForward) {
		this.needForward = needForward;
	}

	public boolean isForward() {
		return isForward;
	}

	public void setForward(boolean isForward) {
		this.isForward = isForward;
	}

}
