/*
	Apache License
	Version 2.0, January 2004
	http://www.apache.org/licenses/

	TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION

	1. Definitions.

	"License" shall mean the terms and conditions for use, reproduction, and
	distribution as defined by Sections 1 through 9 of this document.

	"Licensor" shall mean the copyright owner or entity authorized by the copyright
	owner that is granting the License.

	"Legal Entity" shall mean the union of the acting entity and all other entities
	that control, are controlled by, or are under common control with that entity.
	For the purposes of this definition, "control" means (i) the power, direct or
	indirect, to cause the direction or management of such entity, whether by
	contract or otherwise, or (ii) ownership of fifty percent (50%) or more of the
	outstanding shares, or (iii) beneficial ownership of such entity.

	"You" (or "Your") shall mean an individual or Legal Entity exercising
	permissions granted by this License.

	"Source" form shall mean the preferred form for making modifications, including
	but not limited to software source code, documentation source, and configuration
	files.

	"Object" form shall mean any form resulting from mechanical transformation or
	translation of a Source form, including but not limited to compiled object code,
	generated documentation, and conversions to other media types.

	"Work" shall mean the work of authorship, whether in Source or Object form, made
	available under the License, as indicated by a copyright notice that is included
	in or attached to the work (an example is provided in the Appendix below).

	"Derivative Works" shall mean any work, whether in Source or Object form, that
	is based on (or derived from) the Work and for which the editorial revisions,
	annotations, elaborations, or other modifications represent, as a whole, an
	original work of authorship. For the purposes of this License, Derivative Works
	shall not include works that remain separable from, or merely link (or bind by
	name) to the interfaces of, the Work and Derivative Works thereof.

	"Contribution" shall mean any work of authorship, including the original version
	of the Work and any modifications or additions to that Work or Derivative Works
	thereof, that is intentionally submitted to Licensor for inclusion in the Work
	by the copyright owner or by an individual or Legal Entity authorized to submit
	on behalf of the copyright owner. For the purposes of this definition,
	"submitted" means any form of electronic, verbal, or written communication sent
	to the Licensor or its representatives, including but not limited to
	communication on electronic mailing lists, source code control systems, and
	issue tracking systems that are managed by, or on behalf of, the Licensor for
	the purpose of discussing and improving the Work, but excluding communication
	that is conspicuously marked or otherwise designated in writing by the copyright
	owner as "Not a Contribution."

	"Contributor" shall mean Licensor and any individual or Legal Entity on behalf
	of whom a Contribution has been received by Licensor and subsequently
	incorporated within the Work.

	2. Grant of Copyright License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable copyright license to reproduce, prepare Derivative Works of,
	publicly display, publicly perform, sublicense, and distribute the Work and such
	Derivative Works in Source or Object form.

	3. Grant of Patent License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable (except as stated in this section) patent license to make, have
	made, use, offer to sell, sell, import, and otherwise transfer the Work, where
	such license applies only to those patent claims licensable by such Contributor
	that are necessarily infringed by their Contribution(s) alone or by combination
	of their Contribution(s) with the Work to which such Contribution(s) was
	submitted. If You institute patent litigation against any entity (including a
	cross-claim or counterclaim in a lawsuit) alleging that the Work or a
	Contribution incorporated within the Work constitutes direct or contributory
	patent infringement, then any patent licenses granted to You under this License
	for that Work shall terminate as of the date such litigation is filed.

	4. Redistribution.

	You may reproduce and distribute copies of the Work or Derivative Works thereof
	in any medium, with or without modifications, and in Source or Object form,
	provided that You meet the following conditions:

	You must give any other recipients of the Work or Derivative Works a copy of
	this License; and
	You must cause any modified files to carry prominent notices stating that You
	changed the files; and
	You must retain, in the Source form of any Derivative Works that You distribute,
	all copyright, patent, trademark, and attribution notices from the Source form
	of the Work, excluding those notices that do not pertain to any part of the
	Derivative Works; and
	If the Work includes a "NOTICE" text file as part of its distribution, then any
	Derivative Works that You distribute must include a readable copy of the
	attribution notices contained within such NOTICE file, excluding those notices
	that do not pertain to any part of the Derivative Works, in at least one of the
	following places: within a NOTICE text file distributed as part of the
	Derivative Works; within the Source form or documentation, if provided along
	with the Derivative Works; or, within a display generated by the Derivative
	Works, if and wherever such third-party notices normally appear. The contents of
	the NOTICE file are for informational purposes only and do not modify the
	License. You may add Your own attribution notices within Derivative Works that
	You distribute, alongside or as an addendum to the NOTICE text from the Work,
	provided that such additional attribution notices cannot be construed as
	modifying the License.
	You may add Your own copyright statement to Your modifications and may provide
	additional or different license terms and conditions for use, reproduction, or
	distribution of Your modifications, or for any such Derivative Works as a whole,
	provided Your use, reproduction, and distribution of the Work otherwise complies
	with the conditions stated in this License.

	5. Submission of Contributions.

	Unless You explicitly state otherwise, any Contribution intentionally submitted
	for inclusion in the Work by You to the Licensor shall be under the terms and
	conditions of this License, without any additional terms or conditions.
	Notwithstanding the above, nothing herein shall supersede or modify the terms of
	any separate license agreement you may have executed with Licensor regarding
	such Contributions.

	6. Trademarks.

	This License does not grant permission to use the trade names, trademarks,
	service marks, or product names of the Licensor, except as required for
	reasonable and customary use in describing the origin of the Work and
	reproducing the content of the NOTICE file.

	7. Disclaimer of Warranty.

	Unless required by applicable law or agreed to in writing, Licensor provides the
	Work (and each Contributor provides its Contributions) on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied,
	including, without limitation, any warranties or conditions of TITLE,
	NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A PARTICULAR PURPOSE. You are
	solely responsible for determining the appropriateness of using or
	redistributing the Work and assume any risks associated with Your exercise of
	permissions under this License.

	8. Limitation of Liability.

	In no event and under no legal theory, whether in tort (including negligence),
	contract, or otherwise, unless required by applicable law (such as deliberate
	and grossly negligent acts) or agreed to in writing, shall any Contributor be
	liable to You for damages, including any direct, indirect, special, incidental,
	or consequential damages of any character arising as a result of this License or
	out of the use or inability to use the Work (including but not limited to
	damages for loss of goodwill, work stoppage, computer failure or malfunction, or
	any and all other commercial damages or losses), even if such Contributor has
	been advised of the possibility of such damages.

	9. Accepting Warranty or Additional Liability.

	While redistributing the Work or Derivative Works thereof, You may choose to
	offer, and charge a fee for, acceptance of support, warranty, indemnity, or
	other liability obligations and/or rights consistent with this License. However,
	in accepting such obligations, You may act only on Your own behalf and on Your
	sole responsibility, not on behalf of any other Contributor, and only if You
	agree to indemnify, defend, and hold each Contributor harmless for any liability
	incurred by, or claims asserted against, such Contributor by reason of your
	accepting any such warranty or additional liability.

	END OF TERMS AND CONDITIONS

	APPENDIX: How to apply the Apache License to your work

	To apply the Apache License to your work, attach the following boilerplate
	notice, with the fields enclosed by brackets "{}" replaced with your own
	identifying information. (Don't include the brackets!) The text should be
	enclosed in the appropriate comment syntax for the file format. We also
	recommend that a file or class name and description of purpose be included on
	the same "printed page" as the copyright notice for easier identification within
	third-party archives.

	   Copyright 2018 JFinal

	   Licensed under the Apache License, Version 2.0 (the "License");
	   you may not use this file except in compliance with the License.
	   You may obtain a copy of the License at

	     http://www.apache.org/licenses/LICENSE-2.0

	   Unless required by applicable law or agreed to in writing, software
	   distributed under the License is distributed on an "AS IS" BASIS,
	   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	   See the License for the specific language governing permissions and
	   limitations under the License.
*/
package org.tio.http.common;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tio.core.ChannelContext;
import org.tio.core.Node;
import org.tio.core.Tio;
import org.tio.core.exception.TioDecodeException;
import org.tio.http.common.HttpConst.RequestBodyFormat;
import org.tio.http.common.utils.HttpParseUtils;
import org.tio.http.common.utils.IpUtils;
import org.tio.utils.SysConst;
import org.tio.utils.hutool.StrUtil;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * http server中使用
 *
 * @author tanyaowu
 */
public class HttpRequestDecoder {
	private static final Logger log = LoggerFactory.getLogger(HttpRequestDecoder.class);

	/**
	 * 头部，最多有多少字节
	 */
	public static final int MAX_LENGTH_OF_HEADER = 20480;
	/**
	 * 头部，每行最大的字节数
	 */
	public static final int MAX_LENGTH_OF_HEADERLINE = 2048;
	/**
	 * 请求行的最大长度
	 */
	public static final int MAX_LENGTH_OF_REQUESTLINE = 2048;

	/**
	 * @author tanyaowu
	 * 2017年2月22日 下午4:06:42
	 */
	public HttpRequestDecoder() {

	}

	/**
	 * @param buffer
	 * @param limit
	 * @param position
	 * @param readableLength
	 * @param channelContext
	 * @param httpConfig
	 * @return
	 * @throws TioDecodeException
	 * @author tanyaowu
	 */
	public static HttpRequest decode(ByteBuffer buffer, int limit, int position, int readableLength, ChannelContext channelContext, HttpConfig httpConfig)
		throws TioDecodeException {
		//		int initPosition = position;
		//		int count = 0;
		//		Step step = Step.firstline;
		//		StringBuilder currLine = new StringBuilder();
		Map<String, String> headers = new HashMap<>();
		int contentLength = 0;
		byte[] bodyBytes = null;
		//		StringBuilder headerSb = null;//new StringBuilder(512);
		RequestLine firstLine = null;
		//		boolean appendRequestHeaderString = httpConfig.isAppendRequestHeaderString();

		//		if (httpConfig != null) {
		//
		//		}
		//		if (appendRequestHeaderString) {
		//			headerSb = new StringBuilder(512);
		//		}

		// request line start
		firstLine = parseRequestLine(buffer, channelContext);
		if (firstLine == null) {
			return null;
		}
		// request line end

		//		HttpRequestHandler httpRequestHandler = (HttpRequestHandler)channelContext.tioConfig.getAttribute(TioConfigKey.HTTP_REQ_HANDLER);
		//		if (httpRequestHandler != null) {
		//			httpRequest.setHttpConfig(httpRequestHandler.getHttpConfig(httpRequest));
		//		}

		// request header start
		boolean headerCompleted = parseHeaderLine(buffer, headers, 0, httpConfig);
		if (!headerCompleted) {
			return null;
		}
		String contentLengthStr = headers.get(HttpConst.RequestHeaderKey.Content_Length);

		if (StrUtil.isBlank(contentLengthStr)) {
			contentLength = 0;
		} else {
			contentLength = Integer.parseInt(contentLengthStr);
			if (contentLength > httpConfig.getMaxLengthOfPostBody()) {
				throw new TioDecodeException("post body length is too big[" + contentLength + "], max length is " + httpConfig.getMaxLengthOfPostBody() + " byte");
			}
		}

		int headerLength = (buffer.position() - position);
		int allNeedLength = headerLength + contentLength; //这个packet所需要的字节长度(含头部和体部)

		int notReceivedLength = allNeedLength - readableLength; //尚未接收到的数据长度
		if (notReceivedLength > 0) {
			if (notReceivedLength > channelContext.getReadBufferSize()) {
				channelContext.setReadBufferSize(notReceivedLength);
			}

			channelContext.setPacketNeededLength(allNeedLength);
			return null;
		}
		// request header end

		// ----------------------------------------------- request body start

		//		httpRequest.setHttpConfig((HttpConfig) channelContext.tioConfig.getAttribute(TioConfigKey.HTTP_SERVER_CONFIG));

		String realIp = IpUtils.getRealIp(channelContext, httpConfig, headers);
		if (Tio.IpBlacklist.isInBlacklist(channelContext.tioConfig, realIp)) {
			throw new TioDecodeException("[" + realIp + "] in black list");
		}
		if (httpConfig.checkHost) {
			if (!headers.containsKey(HttpConst.RequestHeaderKey.Host)) {
				throw new TioDecodeException("there is no host header");
			}
		}

		Node realNode = null;
		if (Objects.equals(realIp, channelContext.getClientNode().getIp())) {
			realNode = channelContext.getClientNode();
		} else {
			realNode = new Node(realIp, channelContext.getClientNode().getPort()); //realNode
			channelContext.setProxyClientNode(realNode);
		}

		HttpRequest httpRequest = new HttpRequest(realNode);
		httpRequest.setRequestLine(firstLine);
		httpRequest.setChannelContext(channelContext);
		httpRequest.setHttpConfig(httpConfig);
		httpRequest.setHeaders(headers);
		httpRequest.setContentLength(contentLength);
		//		if (appendRequestHeaderString) {
		//			httpRequest.setHeaderString(headerSb.toString());
		//		} else {
		//			httpRequest.setHeaderString("");
		//		}

		String connection = headers.get(HttpConst.RequestHeaderKey.Connection);
		if (connection != null) {
			httpRequest.setConnection(connection.toLowerCase());
		}

		if (StrUtil.isNotBlank(firstLine.queryString)) {
			decodeParams(httpRequest.getParams(), firstLine.queryString, httpRequest.getCharset(), channelContext);
		}

		if (contentLength > 0) {
			bodyBytes = new byte[contentLength];
			buffer.get(bodyBytes);
			httpRequest.setBody(bodyBytes);
			//解析消息体
			parseBody(httpRequest, firstLine, bodyBytes, channelContext, httpConfig);
		} else {
			//			if (StrUtil.isNotBlank(firstLine.getQuery())) {
			//				decodeParams(httpRequest.getParams(), firstLine.getQuery(), httpRequest.getCharset(), channelContext);
			//			}
		}
		// ----------------------------------------------- request body end

		//解析User_Agent(浏览器操作系统等信息)
		//		String User_Agent = headers.get(HttpConst.RequestHeaderKey.User_Agent);
		//		if (StrUtil.isNotBlank(User_Agent)) {
		//			//			long start = System.currentTimeMillis();
		//			UserAgentAnalyzer userAgentAnalyzer = UserAgentAnalyzerFactory.getUserAgentAnalyzer();
		//			UserAgent userAgent = userAgentAnalyzer.parse(User_Agent);
		//			httpRequest.setUserAgent(userAgent);
		//		}

		//		StringBuilder logstr = new StringBuilder();
		//		logstr.append("\r\n------------------ websocket header start ------------------------\r\n");
		//		logstr.append(firstLine.getInitStr()).append(SysConst.CRLF);
		//		Set<Entry<String, String>> entrySet = headers.entrySet();
		//		for (Entry<String, String> entry : entrySet) {
		//			logstr.append(StrUtil.leftPad(entry.getKey(), 30)).append(" : ").append(entry.getValue()).append(SysConst.CRLF);
		//		}
		//		logstr.append("------------------ websocket header start ------------------------\r\n");
		//		log.error(logstr.toString());

		return httpRequest;
	}

	/**
	 * @param params
	 * @param queryString
	 * @param charset
	 * @param channelContext
	 * @throws TioDecodeException
	 * @author tanyaowu
	 */
	public static void decodeParams(Map<String, Object[]> params, String queryString, String charset, ChannelContext channelContext) throws TioDecodeException {
		if (StrUtil.isBlank(queryString)) {
			return;
		}

		String[] keyvalues = queryString.split(SysConst.STR_AMP);
		for (String keyvalue : keyvalues) {
			String[] keyvalueArr = keyvalue.split(SysConst.STR_EQ);
			String value1 = null;
			if (keyvalueArr.length == 2) {
				value1 = keyvalueArr[1];
			} else if (keyvalueArr.length > 2) {
				throw new TioDecodeException("含有多个" + SysConst.STR_EQ);
			}

			String key = keyvalueArr[0];
			String value;
			if (StrUtil.isBlank(value1)) {
				value = null;
			} else {
				try {
					value = URLDecoder.decode(value1, charset);
				} catch (UnsupportedEncodingException e) {
					throw new TioDecodeException(e);
				}
			}

			Object[] existValue = params.get(key);
			if (existValue != null) {
				String[] newExistValue = new String[existValue.length + 1];
				System.arraycopy(existValue, 0, newExistValue, 0, existValue.length);
				newExistValue[newExistValue.length - 1] = value;
				params.put(key, newExistValue);
			} else {
				String[] newExistValue = new String[]{value};
				params.put(key, newExistValue);
			}
		}
		return;
	}

	/**
	 * 解析消息体
	 *
	 * @param httpRequest
	 * @param firstLine
	 * @param bodyBytes
	 * @param channelContext
	 * @param httpConfig
	 * @throws TioDecodeException
	 * @author tanyaowu
	 */
	private static void parseBody(HttpRequest httpRequest, RequestLine firstLine, byte[] bodyBytes, ChannelContext channelContext, HttpConfig httpConfig)
		throws TioDecodeException {
		parseBodyFormat(httpRequest, httpRequest.getHeaders());
		RequestBodyFormat bodyFormat = httpRequest.getBodyFormat();

		httpRequest.setBody(bodyBytes);

		//		if (bodyFormat == RequestBodyFormat.MULTIPART) {
		//			if (log.isInfoEnabled()) {
		//				String bodyString = null;
		//				if (bodyBytes != null && bodyBytes.length > 0) {
		//					if (log.isDebugEnabled()) {
		//						try {
		//							bodyString = new String(bodyBytes, httpRequest.getCharset());
		//							log.debug("{} multipart body value\r\n{}", channelContext, bodyString);
		//						} catch (UnsupportedEncodingException e) {
		//							log.error(channelContext.toString(), e);
		//						}
		//					}
		//				}
		//			}
		//
		//			//【multipart/form-data; boundary=----WebKitFormBoundaryuwYcfA2AIgxqIxA0】
		//			String initboundary = HttpParseUtils.getPerprotyEqualValue(httpRequest.getHeaders(), HttpConst.RequestHeaderKey.Content_Type, "boundary");
		//			log.debug("{}, initboundary:{}", channelContext, initboundary);
		//			HttpMultiBodyDecoder.decode(httpRequest, firstLine, bodyBytes, initboundary, channelContext, httpConfig);
		//		} else {
		//			String bodyString = null;
		//			if (bodyBytes != null && bodyBytes.length > 0) {
		//				try {
		//					bodyString = new String(bodyBytes, httpRequest.getCharset());
		//					httpRequest.setBodyString(bodyString);
		//					if (log.isInfoEnabled()) {
		//						log.info("{} body value\r\n{}", channelContext, bodyString);
		//					}
		//				} catch (UnsupportedEncodingException e) {
		//					log.error(channelContext.toString(), e);
		//				}
		//			}
		//
		//			if (bodyFormat == RequestBodyFormat.URLENCODED) {
		//				parseUrlencoded(httpRequest, firstLine, bodyBytes, bodyString, channelContext);
		//			}
		//		}

		switch (bodyFormat) {
			case MULTIPART:
				if (log.isInfoEnabled()) {
					String bodyString = null;
					if (bodyBytes != null && bodyBytes.length > 0) {
						if (log.isDebugEnabled()) {
							try {
								bodyString = new String(bodyBytes, httpRequest.getCharset());
								log.debug("{} multipart body value\r\n{}", channelContext, bodyString);
							} catch (UnsupportedEncodingException e) {
								log.error(channelContext.toString(), e);
							}
						}
					}
				}

				//【multipart/form-data; boundary=----WebKitFormBoundaryuwYcfA2AIgxqIxA0】
				String contentType = httpRequest.getHeader(HttpConst.RequestHeaderKey.Content_Type);
				String initboundary = HttpParseUtils.getSubAttribute(contentType, "boundary");//.getPerprotyEqualValue(httpRequest.getHeaders(), HttpConst.RequestHeaderKey.Content_Type, "boundary");
				if (log.isDebugEnabled()) {
					log.debug("{}, initboundary:{}", channelContext, initboundary);
				}
				HttpMultiBodyDecoder.decode(httpRequest, firstLine, bodyBytes, initboundary, channelContext, httpConfig);
				break;

			default:
				String bodyString = null;
				if (bodyBytes != null && bodyBytes.length > 0) {
					try {
						bodyString = new String(bodyBytes, httpRequest.getCharset());
						httpRequest.setBodyString(bodyString);
						if (log.isInfoEnabled()) {
							log.info("{} body value\r\n{}", channelContext, bodyString);
						}
					} catch (UnsupportedEncodingException e) {
						log.error(channelContext.toString(), e);
					}
				}

				if (bodyFormat == RequestBodyFormat.URLENCODED) {
					parseUrlencoded(httpRequest, firstLine, bodyBytes, bodyString, channelContext);
				}
				break;
		}
	}

	/**
	 * Content-Type : application/x-www-form-urlencoded; charset=UTF-8
	 * Content-Type : application/x-www-form-urlencoded; charset=UTF-8
	 *
	 * @param httpRequest
	 * @param headers
	 * @author tanyaowu
	 */
	public static void parseBodyFormat(HttpRequest httpRequest, Map<String, String> headers) {
		String contentType = headers.get(HttpConst.RequestHeaderKey.Content_Type);
		String Content_Type = null;
		if (contentType != null) {
			Content_Type = contentType.toLowerCase();
		}
		if (Content_Type.startsWith(HttpConst.RequestHeaderValue.Content_Type.text_plain)) {
			httpRequest.setBodyFormat(RequestBodyFormat.TEXT);
		} else if (Content_Type.startsWith(HttpConst.RequestHeaderValue.Content_Type.multipart_form_data)) {
			httpRequest.setBodyFormat(RequestBodyFormat.MULTIPART);
		} else {
			httpRequest.setBodyFormat(RequestBodyFormat.URLENCODED);
		}
		if (StrUtil.isNotBlank(Content_Type)) {
			String charset = HttpParseUtils.getSubAttribute(Content_Type, "charset");
			if (StrUtil.isNotBlank(charset)) {
				httpRequest.setCharset(charset);
			} else {
				httpRequest.setCharset(SysConst.DEFAULT_ENCODING);
			}
		}
	}

	/**
	 * 解析请求头的每一行
	 *
	 * @param buffer
	 * @param headers
	 * @param hasReceivedHeaderLength
	 * @param httpConfig
	 * @return 头部是否解析完成，true: 解析完成, false: 没有解析完成
	 * @throws TioDecodeException
	 * @author tanyaowu
	 */
	public static boolean parseHeaderLine(ByteBuffer buffer, Map<String, String> headers, int hasReceivedHeaderLength, HttpConfig httpConfig) throws TioDecodeException {
		byte[] allbs = buffer.array();
		int initPosition = buffer.position();
		int lastPosition = initPosition;
		int remaining = buffer.remaining();
		if (remaining == 0) {
			return false;
		} else if (remaining > 1) {
			byte b1 = buffer.get();
			byte b2 = buffer.get();
			if (SysConst.CR == b1 && SysConst.LF == b2) {
				return true;
			} else if (SysConst.LF == b1) {
				return true;
			}
		} else {
			if (SysConst.LF == buffer.get()) {
				return true;
			}
		}

		String name = null;
		String value = null;
		boolean hasValue = false;

		boolean needIteration = false;
		while (buffer.hasRemaining()) {
			byte b = buffer.get();
			if (name == null) {
				if (b == SysConst.COL) {
					int len = buffer.position() - lastPosition - 1;
					name = StrCache.get(allbs, lastPosition, len);
					//					name = new String(allbs, lastPosition, len);
					lastPosition = buffer.position();
				} else if (b == SysConst.LF) {
					byte lastByte = buffer.get(buffer.position() - 2);
					int len = buffer.position() - lastPosition - 1;
					if (lastByte == SysConst.CR) {
						len = buffer.position() - lastPosition - 2;
					}
					name = StrCache.get(allbs, lastPosition, len);
					//					name = new String(allbs, lastPosition, len);
					lastPosition = buffer.position();
					headers.put(StrCache.getLowercase(name), "");

					needIteration = true;
					break;
				}
				continue;
			} else if (value == null) {
				if (b == SysConst.LF) {
					byte lastByte = buffer.get(buffer.position() - 2);
					int len = buffer.position() - lastPosition - 1;
					if (lastByte == SysConst.CR) {
						len = buffer.position() - lastPosition - 2;
					}
					value = new String(allbs, lastPosition, len);
					lastPosition = buffer.position();

					headers.put(StrCache.getLowercase(name), StrUtil.trimEnd(value));
					needIteration = true;
					break;
				} else {
					if (!hasValue && b == SysConst.SPACE) {
						lastPosition = buffer.position();
					} else {
						hasValue = true;
					}
				}
			}
		}

		int lineLength = buffer.position() - initPosition; //这一行(header line)的字节数
		//		log.error("lineLength:{}, headerLength:{}, headers:\r\n{}", lineLength, hasReceivedHeaderLength, Json.toFormatedJson(headers));
		if (lineLength > MAX_LENGTH_OF_HEADERLINE) {
			//			log.error("header line is too long, max length of header line is " + MAX_LENGTH_OF_HEADERLINE);
			throw new TioDecodeException("header line is too long, max length of header line is " + MAX_LENGTH_OF_HEADERLINE);
		}

		if (needIteration) {
			int headerLength = lineLength + hasReceivedHeaderLength; //header占用的字节数
			if (headerLength > MAX_LENGTH_OF_HEADER) {
				throw new TioDecodeException("header is too long, max length of header is " + MAX_LENGTH_OF_HEADER);
			}
			return parseHeaderLine(buffer, headers, headerLength, httpConfig);
		}

		return false;
	}

	/**
	 * 解析请求头的每一行
	 *
	 * @param line
	 * @param headers
	 * @return 头部是否解析完成，true: 解析完成, false: 没有解析完成
	 * @author tanyaowu
	 */
	@SuppressWarnings("unused")
	private static boolean parseHeaderLine2(ByteBuffer buffer, Map<String, String> headers, int headerLength, HttpConfig httpConfig) throws TioDecodeException {
		int initPosition = buffer.position();
		int lastPosition = initPosition;
		int remaining = buffer.remaining();
		if (remaining == 0) {
			return false;
		} else if (remaining > 1) {
			byte b1 = buffer.get();
			byte b2 = buffer.get();
			if (SysConst.CR == b1 && SysConst.LF == b2) {
				return true;
			} else if (SysConst.LF == b1) {
				return true;
			}
		} else {
			if (SysConst.LF == buffer.get()) {
				return true;
			}
		}

		String name = null;
		String value = null;
		boolean hasValue = false;

		boolean needIteration = false;
		while (buffer.hasRemaining()) {
			byte b = buffer.get();
			if (name == null) {
				if (b == SysConst.COL) {
					int nowPosition = buffer.position();
					byte[] bs = new byte[nowPosition - lastPosition - 1];
					buffer.position(lastPosition);
					buffer.get(bs);
					name = new String(bs);
					lastPosition = nowPosition;
					buffer.position(nowPosition);
				} else if (b == SysConst.LF) {
					int nowPosition = buffer.position();
					byte[] bs = null;
					byte lastByte = buffer.get(nowPosition - 2);

					if (lastByte == SysConst.CR) {
						bs = new byte[nowPosition - lastPosition - 2];
					} else {
						bs = new byte[nowPosition - lastPosition - 1];
					}

					buffer.position(lastPosition);
					buffer.get(bs);
					name = new String(bs);
					lastPosition = nowPosition;
					buffer.position(nowPosition);

					headers.put(name.toLowerCase(), null);
					needIteration = true;
					break;
				}
				continue;
			} else if (value == null) {
				if (b == SysConst.LF) {
					int nowPosition = buffer.position();
					byte[] bs = null;
					byte lastByte = buffer.get(nowPosition - 2);

					if (lastByte == SysConst.CR) {
						bs = new byte[nowPosition - lastPosition - 2];
					} else {
						bs = new byte[nowPosition - lastPosition - 1];
					}

					buffer.position(lastPosition);
					buffer.get(bs);
					value = new String(bs);
					lastPosition = nowPosition;
					buffer.position(nowPosition);

					headers.put(name.toLowerCase(), StrUtil.trimEnd(value));
					needIteration = true;
					break;
					//					return true;
				} else {
					if (!hasValue && b == SysConst.SPACE) {
						lastPosition = buffer.position();
					} else {
						hasValue = true;
					}
				}
			}
		}

		if (needIteration) {
			int myHeaderLength = buffer.position() - initPosition;
			if (myHeaderLength > MAX_LENGTH_OF_HEADER) {
				throw new TioDecodeException("header is too long");
			}
			return parseHeaderLine(buffer, headers, myHeaderLength + headerLength, httpConfig);
		}

		if (remaining > MAX_LENGTH_OF_HEADERLINE) {
			throw new TioDecodeException("header line is too long");
		}
		return false;
	}

	/**
	 * parse request line(the first line)
	 *
	 * @param line           GET /tio?value=tanyaowu HTTP/1.1
	 * @param channelContext
	 * @return
	 * @author tanyaowu
	 * 2017年2月23日 下午1:37:51
	 */
	public static RequestLine parseRequestLine(ByteBuffer buffer, ChannelContext channelContext) throws TioDecodeException {
		//		if (!buffer.hasArray()) {
		//			return parseRequestLine2(buffer, channelContext);
		//		}

		byte[] allbs = buffer.array();

		int initPosition = buffer.position();

		//		int remaining = buffer.remaining();
		String methodStr = null;
		String pathStr = null;
		String queryStr = null;
		String protocol = null;
		String version = null;
		int lastPosition = initPosition;//buffer.position();
		while (buffer.hasRemaining()) {
			byte b = buffer.get();
			if (methodStr == null) {
				if (b == SysConst.SPACE) {
					int len = buffer.position() - lastPosition - 1;
					methodStr = StrCache.get(allbs, lastPosition, len);
					//					methodStr = new String(allbs, lastPosition, len);
					lastPosition = buffer.position();
				}
				continue;
			} else if (pathStr == null) {
				if (b == SysConst.SPACE || b == SysConst.ASTERISK) {
					int len = buffer.position() - lastPosition - 1;
					pathStr = StrCache.get(allbs, lastPosition, len);
					//					pathStr = new String(allbs, lastPosition, len);
					lastPosition = buffer.position();

					if (b == SysConst.SPACE) {
						queryStr = SysConst.BLANK;
					}
				}
				continue;
			} else if (queryStr == null) {
				if (b == SysConst.SPACE) {
					int len = buffer.position() - lastPosition - 1;
					queryStr = new String(allbs, lastPosition, len);
					lastPosition = buffer.position();
				}
				continue;
			} else if (protocol == null) {
				if (b == SysConst.BACKSLASH) {
					int len = buffer.position() - lastPosition - 1;
					protocol = StrCache.get(allbs, lastPosition, len);
					//					protocol = new String(allbs, lastPosition, len);
					lastPosition = buffer.position();
				}
				continue;
			} else if (version == null) {
				if (b == SysConst.LF) {
					byte lastByte = buffer.get(buffer.position() - 2);
					int len = buffer.position() - lastPosition - 1;
					if (lastByte == SysConst.CR) {
						len = buffer.position() - lastPosition - 2;
					}

					version = StrCache.get(allbs, lastPosition, len);
					//					version = new String(allbs, lastPosition, len);

					lastPosition = buffer.position();

					RequestLine requestLine = new RequestLine();
					Method method = Method.from(methodStr);
					requestLine.setMethod(method);
					requestLine.setPath(pathStr);
					requestLine.setInitPath(pathStr);
					requestLine.setQueryString(queryStr);
					requestLine.setProtocol(protocol);
					requestLine.setVersion(version);

					//					requestLine.setLine(line);

					return requestLine;
				}
				continue;
			}
		}

		if ((buffer.position() - initPosition) > MAX_LENGTH_OF_REQUESTLINE) {
			throw new TioDecodeException("request line is too long");
		}
		return null;
	}

	@SuppressWarnings("unused")
	private static RequestLine parseRequestLine2(ByteBuffer buffer, ChannelContext channelContext) throws TioDecodeException {
		int initPosition = buffer.position();
		//		int remaining = buffer.remaining();
		String methodStr = null;
		String pathStr = null;
		String queryStr = null;
		String protocol = null;
		String version = null;
		int lastPosition = initPosition;//buffer.position();
		while (buffer.hasRemaining()) {
			byte b = buffer.get();
			if (methodStr == null) {
				if (b == SysConst.SPACE) {
					int nowPosition = buffer.position();
					byte[] bs = new byte[nowPosition - lastPosition - 1];
					buffer.position(lastPosition);
					buffer.get(bs);
					methodStr = new String(bs);
					lastPosition = nowPosition;
					buffer.position(nowPosition);
				}
				continue;
			} else if (pathStr == null) {
				if (b == SysConst.SPACE || b == SysConst.ASTERISK) {
					int nowPosition = buffer.position();
					byte[] bs = new byte[nowPosition - lastPosition - 1];
					buffer.position(lastPosition);
					buffer.get(bs);
					pathStr = new String(bs);
					lastPosition = nowPosition;
					buffer.position(nowPosition);

					if (b == SysConst.SPACE) {
						queryStr = "";
					}
				}
				continue;
			} else if (queryStr == null) {
				if (b == SysConst.SPACE) {
					int nowPosition = buffer.position();
					byte[] bs = new byte[nowPosition - lastPosition - 1];
					buffer.position(lastPosition);
					buffer.get(bs);
					queryStr = new String(bs);
					lastPosition = nowPosition;
					buffer.position(nowPosition);
				}
				continue;
			} else if (protocol == null) {
				if (b == '/') {
					int nowPosition = buffer.position();
					byte[] bs = new byte[nowPosition - lastPosition - 1];
					buffer.position(lastPosition);
					buffer.get(bs);
					protocol = new String(bs);
					lastPosition = nowPosition;
					buffer.position(nowPosition);
				}
				continue;
			} else if (version == null) {
				if (b == SysConst.LF) {
					int nowPosition = buffer.position();
					byte[] bs = null;
					byte lastByte = buffer.get(nowPosition - 2);

					if (lastByte == SysConst.CR) {
						bs = new byte[nowPosition - lastPosition - 2];
					} else {
						bs = new byte[nowPosition - lastPosition - 1];
					}

					buffer.position(lastPosition);
					buffer.get(bs);
					version = new String(bs);
					lastPosition = nowPosition;
					buffer.position(nowPosition);

					RequestLine requestLine = new RequestLine();
					Method method = Method.from(methodStr);
					requestLine.setMethod(method);
					requestLine.setPath(pathStr);
					requestLine.setInitPath(pathStr);
					requestLine.setQueryString(queryStr);
					requestLine.setProtocol(protocol);
					requestLine.setVersion(version);

					//					requestLine.setLine(line);

					return requestLine;
				}
				continue;
			}
		}

		if ((buffer.position() - initPosition) > MAX_LENGTH_OF_REQUESTLINE) {
			throw new TioDecodeException("request line is too long");
		}
		return null;
	}

	/**
	 * 解析URLENCODED格式的消息体
	 * 形如： 【Content-Type : application/x-www-form-urlencoded; charset=UTF-8】
	 *
	 * @throws TioDecodeException
	 * @author tanyaowu
	 */
	private static void parseUrlencoded(HttpRequest httpRequest, RequestLine firstLine, byte[] bodyBytes, String bodyString, ChannelContext channelContext)
		throws TioDecodeException {
		decodeParams(httpRequest.getParams(), bodyString, httpRequest.getCharset(), channelContext);
	}

	//	/**
	//	 * 解析查询
	//	 * @param httpRequest
	//	 * @param requestLine
	//	 * @param channelContext
	//	 */
	//	private static void parseQueryString(HttpRequest httpRequest, RequestLine requestLine, ChannelContext channelContext) {
	//		String queryString = requestLine.getQueryString();
	//		decodeParams(httpRequest.getParams(), queryString, httpRequest.getCharset(), channelContext);
	//	}

	public static enum Step {
		firstline, header, body
	}

}
