package net.dreamlu.iot.mqtt.spring.client;

import lombok.RequiredArgsConstructor;
import net.dreamlu.iot.mqtt.codec.MqttPublishMessage;
import net.dreamlu.iot.mqtt.core.client.IMqttClientMessageListener;
import net.dreamlu.iot.mqtt.core.common.TopicFilterType;
import org.springframework.util.ReflectionUtils;
import org.tio.core.ChannelContext;

import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.Map;

/**
 * MqttClientSubscribe 注解的默认监听处理器
 *
 * @author L.cm
 */
@RequiredArgsConstructor
class MqttClientSubscribeMessageListener implements IMqttClientMessageListener {
	private final Object bean;
	private final Method method;
	private final String[] topicTemplates;
	private final String[] topicFilters;

	@Override
	public void onMessage(ChannelContext context, String topic, MqttPublishMessage message, byte[] payload) {
		// 处理参数
		Object[] methodParameters = getMethodParameters(method, topicTemplates, topicFilters, topic, message, payload);
		// 反射调用
		ReflectionUtils.invokeMethod(method, bean, methodParameters);
	}

	/**
	 * 获取反射参数
	 *
	 * @param method         Method
	 * @param topicTemplates topicTemplates
	 * @param topicFilters   topicFilters
	 * @param topic          topic
	 * @param message        message
	 * @param payload        payload
	 * @return Object array
	 */
	private static Object[] getMethodParameters(Method method, String[] topicTemplates, String[] topicFilters,
												String topic, MqttPublishMessage message, byte[] payload) {
		int paramCount = method.getParameterCount();
		Class<?>[] parameterTypes = method.getParameterTypes();
		Object[] parameters = new Object[paramCount];
		for (int i = 0; i < parameterTypes.length; i++) {
			Class<?> parameterType = parameterTypes[i];
			if (parameterType == String.class) {
				parameters[i] = topic;
			} else if (parameterType == Map.class) {
				parameters[i] = getTopicVars(topicTemplates, topicFilters, topic);
			} else if (parameterType == MqttPublishMessage.class) {
				parameters[i] = message;
			} else if (parameterType == byte[].class) {
				parameters[i] = payload;
			} else if (parameterType == ByteBuffer.class) {
				parameters[i] = ByteBuffer.wrap(payload);
			}
		}
		return parameters;
	}

	/**
	 * 获取 topic 变量
	 *
	 * @param topicTemplates topicTemplates
	 * @param topicFilters   topicFilters
	 * @param topic          topic
	 * @return 变量集合
	 */
	private static Map<String, String> getTopicVars(String[] topicTemplates, String[] topicFilters, String topic) {
		for (int j = 0; j < topicFilters.length; j++) {
			String topicFilter = topicFilters[j];
			TopicFilterType topicFilterType = TopicFilterType.getType(topicFilter);
			if (topicFilterType.match(topicFilter, topic)) {
				String topicTemplate = topicTemplates[j];
				return topicFilterType.extractTopicVars(topicTemplate, topic);
			}
		}
		return Collections.emptyMap();
	}

}
