package net.dongliu.xhttp.json;

import net.dongliu.xhttp.AsyncResponseHandler;
import net.dongliu.xhttp.ContentType;
import net.dongliu.xhttp.MimeType;
import net.dongliu.xhttp.ResponseHandler;
import net.dongliu.xhttp.body.AbstractBody;
import net.dongliu.xhttp.body.Body;

import java.io.*;
import java.lang.reflect.Type;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Flow;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Objects.requireNonNull;

/**
 * Json processor, used to decode/encode json values, blocking and non-blocking.
 *
 * @author Liu Dong
 */
public abstract class JsonProcessor {

    /**
     * Serialize value to json, and write to writer. The output stream is leaved unclosed.
     */
    protected abstract void marshal(Object value, OutputStream out, Charset charset) throws IOException;

    /**
     * Deserialize json from Reader, to specific type. The input stream is closed when finished or error occurred.
     *
     * @param charset the charset, cannot be null
     */
    protected abstract <T> T unmarshal(InputStream in, Charset charset, Type type) throws IOException;

    /**
     * Create a new json http body
     *
     * @param value the value of body, to be marshall by this processor
     * @param <T>   the value type
     * @return json http body
     */
    public <T> Body<T> jsonBody(T value) {
        return new JsonBody<>(value);
    }

    /**
     * Create a new json http body
     *
     * @param value the value of body, to be marshall by this processor
     * @param <T>   the value type
     * @return json http body
     */
    public <T> Body<T> jsonBody(T value, Charset charset) {
        return new JsonBody<>(value, charset);
    }

    /**
     * Return a ResponseHandler, which deserialize json with charset and type.
     */
    public <T> ResponseHandler<T> responseHandler(Class<T> type) {
        return responseHandler((Type) type);
    }

    /**
     * Return a ResponseHandler, which deserialize json with charset and type.
     */
    public <T> ResponseHandler<T> responseHandler(TypeToken<T> type) {
        return responseHandler(type.getType());
    }


    /**
     * Return a ResponseHandler, which deserialize json with charset and type.
     */
    private <T> ResponseHandler<T> responseHandler(Type type) {
        return (charset, info) -> unmarshal(info.body(), charset.get(), type);
    }

    /**
     * Return a AsyncResponseHandler, which deserialize json from ByteBuffers async, with charset and type.
     */
    public <T> AsyncResponseHandler<T> asyncResponseHandler(TypeToken<T> typeToken) {
        return asyncResponseHandler(typeToken.getType());
    }

    /**
     * Return a AsyncResponseHandler, which deserialize json from ByteBuffers async, with charset and type.
     */
    public <T> AsyncResponseHandler<T> asyncResponseHandler(Class<T> cls) {
        return asyncResponseHandler((Type) cls);
    }

    /**
     * Return a AsyncResponseHandler, which deserialize json from ByteBuffers async, with charset and type.
     * Default impl read all data of response body unblocking, and then unmarshal using blocking api.
     * Override this method if implementation can decode json with non-blocking API.
     */
    protected <T> AsyncResponseHandler<T> asyncResponseHandler(Type type) {
        requireNonNull(type);
        return (charset, info) -> {
            var bodySubscriber = new JsonBodySubscriber<T>(charset.get(), type);
            info.body().subscribe(bodySubscriber);
            return bodySubscriber.getBody().toCompletableFuture();
        };
    }


    /**
     * Json http body
     *
     * @param <T> the value type
     */
    private class JsonBody<T> extends AbstractBody<T> implements Body<T> {

        private JsonBody(T value) {
            this(value, UTF_8);
        }

        private JsonBody(T value, Charset charset) {
            super(value, ContentType.of(MimeType.JSON, requireNonNull(charset)));
        }

        @Override
        public HttpRequest.BodyPublisher asBodyPublisher() {
            return asPublisher(body(), contentType().charset().orElseThrow());
        }

        /**
         * Marshal value to json, for async http using. Default using block api to get all data once, and then publish.
         *
         * @param value   the value
         * @param charset the charset
         * @return http body publisher
         */
        HttpRequest.BodyPublisher asPublisher(Object value, Charset charset) {
            try (ByteArrayOutputStream bos = new ByteArrayOutputStream()) {
                marshal(value, bos, charset);
                return HttpRequest.BodyPublishers.ofByteArray(bos.toByteArray());
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }

    private class JsonBodySubscriber<T> implements HttpResponse.BodySubscriber<T> {
        private final CompletableFuture<T> future;
        private final Charset charset;
        private final Type type;
        private Flow.Subscription subscription;
        private final List<ByteBuffer> buffers;

        public JsonBodySubscriber(Charset charset, Type type) {
            this.charset = charset;
            this.type = type;
            future = new CompletableFuture<>();
            buffers = new ArrayList<>();
        }

        @Override
        public CompletionStage<T> getBody() {
            return future;
        }

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            this.subscription = subscription;
            subscription.request(1);
        }

        @Override
        public void onNext(List<ByteBuffer> item) {
            buffers.addAll(item);
            subscription.request(1);
        }

        @Override
        public void onError(Throwable throwable) {
            future.completeExceptionally(throwable);
            subscription.cancel();
        }

        @Override
        public void onComplete() {
            try (var in = new ByteBuffersInputStream(buffers)) {
                future.complete(unmarshal(in, charset, type));
            } catch (Throwable t) {
                future.completeExceptionally(t);
            }
        }
    }
}
