package net.dongliu.jlink;

import net.dongliu.commons.Joiner;
import net.dongliu.commons.Strings;
import net.dongliu.commons.collection.Lists;
import net.dongliu.jlink.model.Launcher;
import net.dongliu.jlink.util.ModuleInfo;
import net.dongliu.jlink.util.MojoLog;
import net.dongliu.jlink.util.ProcessResult;
import net.dongliu.jlink.util.ProcessUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.*;
import org.apache.maven.project.MavenProject;
import org.apache.maven.toolchain.Toolchain;
import org.apache.maven.toolchain.ToolchainManager;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.*;
import java.util.stream.Collectors;

import static java.nio.charset.StandardCharsets.ISO_8859_1;
import static java.util.stream.Collectors.toList;

/**
 * @author dongliu
 */
@Mojo(name = "link", defaultPhase = LifecyclePhase.PACKAGE, requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME)
public class JLinkMojo extends AbstractMojo {
    @Parameter(defaultValue = "${project}", readonly = true)
    private MavenProject project;
    @Parameter(defaultValue = "${project.build.directory}/${project.build.finalName}.${project.packaging}")
    private File projectArtifact;
    @Parameter(property = "modulesDirectory", defaultValue = "${project.build.directory}/modules")
    private File modulesDirectory;
    @Parameter
    private List<String> jvmOptions;
    @Parameter
    private List<String> excludeResources;
    @Parameter
    private List<String> excludeFiles;
    @Parameter
    private boolean excludeDesktop;
    @Component
    private ToolchainManager toolchainManager;
    @Parameter(defaultValue = "${session}", readonly = true)
    private MavenSession mavenSession;
    @Parameter
    private String baseJdk;
    @Parameter(defaultValue = "[]")
    private List<File> modulePath;
    @Parameter(defaultValue = "${project.build.directory}/jlink")
    private File output;
    @Parameter(defaultValue = "${project.build.directory}/jlink_working")
    private File workingDirectory;
    @Parameter
    private List<String> addModules;
    @Parameter
    private List<Launcher> launchers;
    @Parameter
    private int compress;
    @Parameter(defaultValue = "false")
    private boolean stripDebug;
    @Parameter(property = "ignoreSigningInformation", defaultValue = "false")
    private boolean ignoreSigningInformation;

    @Parameter(defaultValue = "false")
    private boolean noHeaderFiles;

    @Parameter(defaultValue = "false")
    private boolean noManPages;

    @Override
    public void execute() throws MojoExecutionException {
        String packaging = project.getModel().getPackaging();
        if (!packaging.equalsIgnoreCase("jar")) {
            getLog().error("require packaging type to be jar or jmod, '" + packaging + " not supported'");
            return;
        }
        Path javaHome = getJavaHome();
        Path jmodsPath = javaHome.resolve("jmods");

        Set<Path> effectiveModulePath = this.modulePath.stream()
                .map(File::toPath)
                .collect(Collectors.toSet());

        effectiveModulePath.add(jmodsPath);
        effectiveModulePath.add(modulesDirectory.toPath());

        copyModules();

        try {
            excludeDesktop(jmodsPath);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }

        DescribeModule describeModule = new DescribeModule(projectArtifact.toPath());
        ModuleInfo projectModuleInfo = describeModule.describe();
        if (addModules == null) {
            addModules = new ArrayList<>();
        }
        if (projectModuleInfo != null) {
            final String name = projectModuleInfo.name();
            if (!addModules.contains(name)) {
                getLog().info("add project module: " + name);
                addModules.add(name);
            }
        }
        if (addModules.isEmpty()) {
            throw new MojoExecutionException("add-modules empty");
        }

        getLog().info("creating jlink image at " + output.toString());
        new JLink(
                effectiveModulePath,
                addModules,
                Lists.nullToEmpty(launchers),
                output.toPath(),
                compress,
                stripDebug,
                ignoreSigningInformation,
                Lists.nullToEmpty(excludeResources),
                Lists.nullToEmpty(excludeFiles),
                new MojoLog(getLog()),
                noHeaderFiles,
                noManPages
        ).run();

        try {
            tryAddJvmOptions();
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void excludeDesktop(Path jmodsPath) throws MojoExecutionException, IOException {
        if (!excludeDesktop) {
            return;
        }
        Path desktopPath = jmodsPath.resolve("java.desktop.jmod");
        String jmodPath = Paths.get(System.getProperty("java.home"), "bin", "jmod").toString();
        ProcessResult result = ProcessUtils.execute(jmodPath, "list", desktopPath.toString());
        if (result.exitCode() != 0) {
            throw new MojoExecutionException("list desktop jmod failed, exit with: " + result.exitCode());
        }
        Set<String> set = new HashSet<>();
        List<String> desktopExcludeResources = new ArrayList<>();
        Path excludeResourcesPath = workingDirectory.toPath().resolve("desktop_exclude_resources.list");
        getLog().info("generate desktop excluded resources to " + excludeResourcesPath);
        for (String line : result.stdout().split("\n")) {
            line = line.trim();
            if (line.isEmpty() || !line.startsWith("classes/") || line.contains("/beans/")) {
                continue;
            }
            line = line.substring("classes/".length());
            String packageName = Strings.subStringBeforeLast(line, "/");
            if (!set.contains(packageName)) {
                String fileName = Strings.subStringAfterLast(line, "/");
                String fileNameEscaped = fileName.replace(".", "\\.").replace("$", "\\$");
                desktopExcludeResources.add("regex:/java.desktop/" + packageName + "/(?!" + fileNameEscaped + ")[_a-zA-Z0-9\\.\\$]+");
                set.add(packageName);
            }
        }
        Files.write(excludeResourcesPath, String.join("\n", desktopExcludeResources).getBytes());
        if (excludeResources == null) {
            excludeResources = new ArrayList<>();
        }
        excludeResources.add("@" + excludeResourcesPath);
        if (excludeFiles == null) {
            excludeFiles = new ArrayList<>();
        }
        excludeFiles.add("glob:/java.desktop/lib/*");
    }

    private void tryAddJvmOptions() throws IOException {
        if (jvmOptions != null && !jvmOptions.isEmpty()) {
            String jvmOptionStr = Joiner.of(" ").join(jvmOptions);
            List<Path> paths = Files.list(output.toPath().resolve("bin")).collect(toList());
            for (Path path : paths) {
                String content = new String(Files.readAllBytes(path), ISO_8859_1);
                if (content.contains("JLINK_VM_OPTIONS=")) {
                    getLog().info("add jvm options to launcher: " + path.getFileName());
                    content = content.replace("JLINK_VM_OPTIONS=", "JLINK_VM_OPTIONS=" + jvmOptionStr);
                    Files.write(path, content.getBytes(ISO_8859_1));
                }
            }
        }
    }

    /**
     * Returns the directory with the jmod files to be used for creating the image.
     * If {@code baseJdk} has been given, the jmod files from the JDK identified that way
     * will be used; otherwise the jmod files from the JDK running the current build
     * will be used.
     */
    private Path getJavaHome() throws MojoExecutionException {
        if (baseJdk != null) {
            List<Toolchain> toolChains = toolchainManager.getToolchains(mavenSession, "jdk", getToolChainRequirements(baseJdk));
            if (toolChains.isEmpty()) {
                throw new MojoExecutionException("Found no tool chain of type 'jdk' and matching requirements '" + baseJdk + "'");
            } else if (toolChains.size() > 1) {
                throw new MojoExecutionException("Found more than one tool chain of type 'jdk' and matching requirements '" + baseJdk + "'");
            } else {
                Toolchain toolchain = toolChains.get(0);

                String javac = toolchain.findTool("javac");

                // #63; when building on Linux / OS X but creating a Windows runtime image
                // the tool lookup must be for javac.exe explicitly (as the toolchain mechanism
                // itself won't append the suffix if not running this build on Windows
                if (javac == null) {
                    javac = toolchain.findTool("javac.exe");
                }
                if (javac == null) {
                    throw new MojoExecutionException("Couldn't locate toolchain directory");
                }
                return new File(javac)
                        .toPath()
                        .getParent()
                        .getParent();
            }
        } else {
            return Paths.get(System.getProperty("java.home"));
        }
    }

    private Map<String, String> getToolChainRequirements(String baseJdk) throws MojoExecutionException {
        Map<String, String> toolChainRequirements = new HashMap<>();
        String[] requirements = baseJdk.split(",");

        for (String requirement : requirements) {
            String[] keyAndValue = requirement.split("=");
            if (keyAndValue.length != 2) {
                throw new MojoExecutionException(
                        "Toolchain requirements must be given in the form 'key1=value1,key2=value2,...'." +
                                "Given value '" + baseJdk + "' doesn't match this pattern.");
            }

            toolChainRequirements.put(keyAndValue[0].trim(), keyAndValue[1].trim());
        }

        return toolChainRequirements;
    }

    public void copyModules() {
        Path outputPath = modulesDirectory.toPath();

        createDirectories();

        Set<Artifact> dependencies = project.getArtifacts();
        Path modulesBasePath = project.getBasedir().toPath().resolve("src/main/modules");
        for (Artifact artifact : dependencies) {
            String scope = artifact.getScope();
            if (!scope.equals("compile") && !scope.equals("runtime")) {
                continue;
            }

            String groupId = artifact.getGroupId();
            String artifactId = artifact.getArtifactId();
            Path inputFile = artifact.getFile().toPath();
            Path moduleInfoPath = modulesBasePath.resolve(groupId).resolve(artifactId).resolve("module-info.java");
            if (!Files.exists(moduleInfoPath)) {
                getLog().debug("copy module " + artifact.getArtifactId());
                Path outputJar = modulesDirectory.toPath().resolve(inputFile.getFileName());

                try {
                    Files.copy(inputFile, outputJar, StandardCopyOption.REPLACE_EXISTING);
                } catch (IOException e) {
                    throw new UncheckedIOException("Couldn't copy JAR file", e);
                }
                continue;
            }

            getLog().info("add module info to artifact: " + artifact.getArtifactId());

            String moduleInfoSource;
            try {
                moduleInfoSource = new String(Files.readAllBytes(moduleInfoPath), StandardCharsets.UTF_8);
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }

            AddModuleInfo addModuleInfo = new AddModuleInfo(
                    moduleInfoSource,
                    inputFile,
                    outputPath,
                    true
            );

            addModuleInfo.run();
        }

        Path projectJarPath = projectArtifact.toPath();
        Path outputJar = modulesDirectory.toPath().resolve(projectJarPath.getFileName());
        try {
            Files.copy(projectJarPath, outputJar, StandardCopyOption.REPLACE_EXISTING);
        } catch (IOException e) {
            throw new UncheckedIOException("Couldn't copy JAR file", e);
        }
    }

    private void createDirectories() {
        if (!workingDirectory.exists()) {
            workingDirectory.mkdirs();
        }
        if (!modulesDirectory.exists()) {
            modulesDirectory.mkdirs();
        }
    }
}
