package net.dongliu.cute.http.internal;

import net.dongliu.cute.http.exception.RequestsException;
import net.dongliu.cute.http.NameValue;
import net.dongliu.cute.http.Param;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Util methods for encode / decode url.
 */
public class UrlUtils {

    /**
     * Encode multi queries
     */
    public static String encodeQueries(Collection<? extends NameValue> queries, Charset charset) {
        StringBuilder sb = new StringBuilder();
        for (var query : queries) {
            sb.append(URLEncoder.encode(query.name(), charset));
            sb.append('=');
            sb.append(URLEncoder.encode(query.value(), charset));
            sb.append('&');
        }
        if (sb.length() > 0) {
            sb.deleteCharAt(sb.length() - 1);
        }
        return sb.toString();
    }

    /**
     * Decode key-value query parameter
     */
    public static Param decodeQuery(String s, Charset charset) {
        int idx = s.indexOf("=");
        if (idx < 0) {
            return Param.of("", URLDecoder.decode(s, charset));
        }
        return Param.of(URLDecoder.decode(s.substring(0, idx), charset), URLDecoder.decode(s.substring(idx + 1), charset));
    }

    /**
     * Parse query params
     */
    public static List<Param> decodeQueries(String queryStr, Charset charset) {
        String[] queries = queryStr.split("&");
        List<Param> list = new ArrayList<>(queries.length);
        for (String query : queries) {
            list.add(decodeQuery(query, charset));
        }
        return list;
    }

    /**
     * Encode key-value form parameter
     */
    public static String encodeForm(Param query, Charset charset) {
        try {
            return URLEncoder.encode(query.name(), charset.name()) + "=" + URLEncoder.encode(query.value(),
                    charset.name());
        } catch (UnsupportedEncodingException e) {
            // should not happen
            throw new RequestsException(e);
        }
    }

    /**
     * Encode multi form parameters
     */
    public static String encodeForms(Collection<? extends Param> queries, Charset charset) {
        StringBuilder sb = new StringBuilder();
        try {
            for (Param query : queries) {
                sb.append(URLEncoder.encode(query.name(), charset.name()));
                sb.append('=');
                sb.append(URLEncoder.encode(query.value(), charset.name()));
                sb.append('&');
            }
        } catch (UnsupportedEncodingException e) {
            // should not happen
            throw new RequestsException(e);
        }
        if (sb.length() > 0) {
            sb.deleteCharAt(sb.length() - 1);
        }
        return sb.toString();
    }

    /**
     * Decode key-value query parameter
     */
    public static Param decodeForm(String s, Charset charset) {
        int idx = s.indexOf("=");
        try {
            if (idx < 0) {
                return Param.of("", URLDecoder.decode(s, charset.name()));
            }
            return Param.of(URLDecoder.decode(s.substring(0, idx), charset.name()),
                    URLDecoder.decode(s.substring(idx + 1), charset.name()));
        } catch (UnsupportedEncodingException e) {
            // should not happen
            throw new RequestsException(e);
        }
    }

    /**
     * Parse query params
     */
    public static List<Param> decodeForms(String queryStr, Charset charset) {
        String[] queries = queryStr.split("&");
        List<Param> list = new ArrayList<>(queries.length);
        for (String query : queries) {
            list.add(decodeForm(query, charset));
        }

        return list;
    }

    public static URL joinUrl(URL url, Collection<? extends NameValue> params, Charset charset) {
        if (params.isEmpty()) {
            return url;
        }

        StringBuilder sb = new StringBuilder();
        sb.append(url.getProtocol()).append(':');
        if (url.getAuthority() != null && !url.getAuthority().isEmpty()) {
            sb.append("//").append(url.getAuthority());
        }
        if (url.getPath() != null) {
            sb.append(url.getPath());
        }

        String query = url.getQuery();
        String newQuery = encodeQueries(params, charset);
        if (query == null || query.isEmpty()) {
            sb.append('?').append(newQuery);
        } else {
            sb.append('?').append(query).append('&').append(newQuery);
        }

        if (url.getRef() != null) {
            sb.append('#').append(url.getRef());
        }

        URL fullURL;
        try {
            fullURL = new URL(sb.toString());
        } catch (MalformedURLException e) {
            throw new RequestsException(e);

        }
        return fullURL;
    }
}
