package net.dongliu.cute.http;

import java.util.Objects;
import java.util.Optional;

import static java.util.Objects.requireNonNull;

/**
 * HTTP mime type.
 */
public class MimeType {
    private final String type;
    private final String subType;

    // text types
    public static final MimeType TEXT_PLAIN = MimeType.of("text", "plain");
    public static final MimeType HTML = MimeType.of("text", "html");
    public static final MimeType CSS = MimeType.of("text", "css");

    // application textual types
    public static final MimeType JSON = MimeType.of("application", "json");
    public static final MimeType XHTML = MimeType.of("application", "xhtml+xml");
    public static final MimeType XML = MimeType.of("application", "xml");

    // application binary types
    public static final MimeType OCTET_STREAM = MimeType.of("application", "octet-stream");
    // multi part
    public static final MimeType MULTI_PART = MimeType.of("multipart", "form-data");
    // www-form-encoded
    public static final MimeType WWW_FORM_ENCODED = MimeType.of("application", "x-www-form-urlencoded");

    private MimeType(String type, String subType) {
        this.type = type;
        this.subType = subType;
    }

    /**
     * Create new mime type instance, by type and subType.
     *
     * @param type    the mime type
     * @param subType the mime sub type
     * @return a MimeType instance
     */
    public static MimeType of(String type, String subType) {
        return new MimeType(requireNonNull(type), requireNonNull(subType));
    }

    /**
     * Parse mime type string.
     *
     * @param mimeType the mime-type string
     * @return If parse failed, return empty Optional.
     */
    public static Optional<MimeType> safeParse(String mimeType) {
        Objects.requireNonNull(mimeType);
        if (mimeType.isEmpty()) {
            return Optional.empty();
        }
        int idx = mimeType.indexOf("/");
        if (idx == -1) {
            return Optional.empty();
        }
        String type = mimeType.substring(0, idx);
        String subType = mimeType.substring(idx + 1);
        if (type.isEmpty() || subType.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(new MimeType(type, subType));
    }

    /**
     * The mime type
     */
    public String type() {
        return type;
    }

    /**
     * The mime subType
     */
    public String subType() {
        return subType;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        MimeType mimeType = (MimeType) o;
        return Objects.equals(type, mimeType.type) &&
                Objects.equals(subType, mimeType.subType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(type, subType);
    }

    @Override
    public String toString() {
        return type + "/" + subType;
    }
}
