package net.dongliu.cute.http.internal;

import net.dongliu.cute.http.body.Body;
import net.dongliu.cute.http.HeaderNames;
import net.dongliu.cute.http.Request;
import net.dongliu.cute.http.exception.RequestsException;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpRequest;
import java.net.http.HttpRequest.BodyPublishers;
import java.util.Base64;

import static java.nio.charset.StandardCharsets.UTF_8;
import static net.dongliu.cute.http.HeaderNames.ACCEPT_ENCODING;

/**
 * Utils method to make http request instance.
 */
public class RequestUtils {

    private static final String COMPRESS_ENCODINGS = "gzip, deflate";

    /**
     * Construct a HttpRequest from Request instance.
     *
     * @param request the request
     * @return HttpRequest
     */
    public static HttpRequest toHttpRequest(Request request) {
        var body = request.body();

        var httpRequestBuilder = HttpRequest.newBuilder();
        var bodyPublisher = body
                .map(Body::asBodyPublisher)
                .orElseGet(BodyPublishers::noBody);

        // HttpClient set bodyPublisher to null when method is GET.
        // For now (JDK 10.0.0), HttpClient may hang when send GET Request with a empty body.
        switch (request.method()) {
            case GET:
                httpRequestBuilder.GET();
                break;
            case DELETE:
                httpRequestBuilder.DELETE();
                break;
            default:
                httpRequestBuilder.method(request.method().name(), bodyPublisher);
        }

        var url = UrlUtils.joinUrl(request.url(), request.params(), request.paramCharset());
        URI uri;
        try {
            uri = url.toURI();
        } catch (URISyntaxException e) {
            throw new RequestsException(e);
        }
        httpRequestBuilder.uri(uri);
        httpRequestBuilder.expectContinue(false);
        body.ifPresent(b -> {
            var contentType = b.contentType();
            httpRequestBuilder.setHeader(HeaderNames.CONTENT_TYPE, contentType.toString());
        });

        httpRequestBuilder.timeout(request.timeout());

        // headers
        request.userAgent().ifPresent(ua -> httpRequestBuilder.setHeader(HeaderNames.USER_AGENT, ua));

        request.referer().ifPresent(referer -> httpRequestBuilder.setHeader(HeaderNames.REFERER, referer));

        request.basicAuth().ifPresent(ba -> {
            var authStr = ba.getUserName() + ":" + new String(ba.getPassword());
            String auth = "Basic " + Base64.getEncoder().encodeToString(authStr.getBytes(UTF_8));
            httpRequestBuilder.setHeader(HeaderNames.AUTHORIZATION, auth);
        });

        if (request.acceptCompressed()) {
            httpRequestBuilder.setHeader(ACCEPT_ENCODING, COMPRESS_ENCODINGS);
        }

        // set cookies
        if (!request.cookies().isEmpty()) {
            // When HTTP/1.1 is used, a single Cookie: header should be sent. When HTTP/2 is used, then multiple Cookie headers should be sent.
            // we just set multi cookie header, hope JDK HttpClient can send this headers to our custom CookieManager.
            for (var cookie : request.cookies()) {
                httpRequestBuilder.header(HeaderNames.COOKIE, cookie.name() + "=" + cookie.value());
            }
        }

        // set user custom headers, can override headers above
        for (var header : request.headers()) {
            httpRequestBuilder.setHeader(header.name(), String.valueOf(header.value()));
        }

        httpRequestBuilder.timeout(request.timeout());

        return httpRequestBuilder.build();
    }
}
