package net.dongliu.cute.http;

import net.dongliu.cute.http.exception.RequestInterruptedException;
import net.dongliu.cute.http.exception.RequestsException;
import net.dongliu.cute.http.internal.RequestUtils;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.http.HttpResponse;

/**
 * For carrying http client for chained call.
 */
public class RequestContext extends AbstractRequestBuilder<RequestContext> {
    private final Client client;

    RequestContext(Client client, Method method, URL url) {
        super(method, url, client.jsonMarshaller().orElse(null));
        this.client = client;
    }

    /**
     * Build http request, and send out sync.
     *
     * @throws RequestInterruptedException if send is interrupted
     * @throws UncheckedIOException        if io error occurred
     */
    public ResponseContext send() {
        Request request = build();

        // send
        var httpRequest = RequestUtils.toHttpRequest(request);
        HttpResponse<InputStream> response;
        try {
            response = client.httpClient.send(httpRequest, HttpResponse.BodyHandlers.ofInputStream());
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        } catch (InterruptedException e) {
            throw new RequestInterruptedException(e);
        }

        int status = response.statusCode();
        var httpHeaders = response.headers();
        Headers headers = Headers.ofHttpHeaders(httpHeaders);
        URL url;
        try {
            url = response.uri().toURL();
        } catch (MalformedURLException e) {
            throw new RequestsException(e);
        }
        var responseInfo = new ResponseInfo(status, headers);
        var resp = new RawResponse<>(method, url, responseInfo, response.body());
        return new ResponseContext(resp, client.jsonMarshaller().orElse(null));
    }


    /**
     * Build http request, and send out sync.
     */
    public AsyncResponseContext sendAsync() {
        Request request = build();
        var httpRequest = RequestUtils.toHttpRequest(request);

        var responseFuture = client.httpClient.sendAsync(httpRequest, HttpResponse.BodyHandlers.ofPublisher())
                .thenApply(r -> {
                    var headers = Headers.ofHttpHeaders(r.headers());
                    URL url;
                    try {
                        url = r.uri().toURL();
                    } catch (MalformedURLException e) {
                        throw new RequestsException(e);
                    }
                    var responseInfo = new ResponseInfo(r.statusCode(), headers);
                    return new RawResponse<>(method, url, responseInfo, r.body());
                });

        return new AsyncResponseContext(responseFuture, client.jsonMarshaller().orElse(null));
    }

    @Override
    protected RequestContext self() {
        return this;
    }
}
