package net.dongliu.cute.http;


import java.net.URL;
import java.util.function.Function;

import static java.util.Objects.requireNonNull;

/**
 * Http Response with body be transformed
 *
 * @author Liu Dong
 */
public class HTTPResponse<T> {
    private final URL url;
    private final int statusCode;
    private final HTTPHeaders headers;
    private final T body;

    HTTPResponse(URL url, int statusCode, HTTPHeaders headers, T body) {
        this.url = requireNonNull(url);
        this.statusCode = statusCode;
        this.headers = requireNonNull(headers);
        // we do not check null here, cause for type Void we use null.
        this.body = body;
    }

    /**
     * Get actual url (redirected)
     */
    public URL url() {
        return url;
    }

    /**
     * Return http response status code.
     *
     * @return the status code
     */
    public int statusCode() {
        return statusCode;
    }

    /**
     * Get all cookies returned by this response.
     *
     * @return a immutable list of cookies
     */
    public HTTPCookies cookies() {
        return headers.cookies();
    }

    /**
     * Get all response headers
     */
    public HTTPHeaders headers() {
        return headers;
    }

    /**
     * Return the response body.
     *
     * @return The response body
     */
    public T body() {
        return body;
    }

    /**
     * Return a new Response, with body is the result apply transformer on the original response body.
     *
     * @param transformer the function to convert response body
     * @param <R>         the response body type
     */
    public <R> HTTPResponse<R> transform(Function<T, R> transformer) {
        R body = transformer.apply(this.body);
        return new HTTPResponse<>(url, statusCode, headers, body);
    }

}
