package net.dongliu.cute.http;

import net.dongliu.cute.http.exception.RequestsException;
import net.dongliu.cute.http.internal.RequestUtils;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.http.HttpResponse;

/**
 * For carrying http client for chained call.
 */
public class HTTPRequestContext extends AbstractHTTPRequestBuilder<HTTPRequestContext> {
    private final HTTPClient client;

    HTTPRequestContext(HTTPClient client, HTTPMethod method, URL url) {
        super(method, url, client.jsonMarshaller().orElse(null));
        this.client = client;
    }

    /**
     * Build http request, and send out sync.
     */
    public HTTPResponseContext send() {
        HTTPRequest request = build();

        // send
        var httpRequest = RequestUtils.toHttpRequest(request);
        HttpResponse<InputStream> response;
        try {
            response = client.httpClient.send(httpRequest, HttpResponse.BodyHandlers.ofInputStream());
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        } catch (InterruptedException e) {
            // this may be the best we can do to handle InterruptedException except declare throwing in method signature
            // mark the interrupt bit
            Thread.currentThread().interrupt();
            throw new RequestsException("HTTP Request has been interrupted");
        }

        int status = response.statusCode();
        var httpHeaders = response.headers();
        HTTPHeaders headers = HTTPHeaders.ofHttpHeaders(httpHeaders);
        URL url;
        try {
            url = response.uri().toURL();
        } catch (MalformedURLException e) {
            throw new RequestsException(e);
        }
        var responseInfo = new ResponseInfo(status, headers);
        var resp = new RawHTTPResponse<>(method, url, responseInfo, response.body());
        return new HTTPResponseContext(resp, client.jsonMarshaller().orElse(null));
    }


    /**
     * Build http request, and send out sync.
     */
    public AsyncHTTPResponseContext sendAsync() {
        HTTPRequest request = build();
        var httpRequest = RequestUtils.toHttpRequest(request);

        var responseFuture = client.httpClient.sendAsync(httpRequest, HttpResponse.BodyHandlers.ofPublisher())
                .thenApply(r -> {
                    var headers = HTTPHeaders.ofHttpHeaders(r.headers());
                    URL url;
                    try {
                        url = r.uri().toURL();
                    } catch (MalformedURLException e) {
                        throw new RequestsException(e);
                    }
                    var responseInfo = new ResponseInfo(r.statusCode(), headers);
                    return new RawHTTPResponse<>(method, url, responseInfo, r.body());
                });

        return new AsyncHTTPResponseContext(responseFuture, client.jsonMarshaller().orElse(null));
    }

    @Override
    protected HTTPRequestContext self() {
        return this;
    }
}
