package net.dongliu.cute.http;

import net.dongliu.cute.http.body.Body;
import org.checkerframework.checker.nullness.qual.Nullable;

import java.net.PasswordAuthentication;
import java.net.URL;
import java.nio.charset.Charset;
import java.time.Duration;
import java.util.List;
import java.util.Optional;

/**
 * Http request
 *
 * @author Liu Dong
 */
public class HTTPRequest {
    private final HTTPMethod method;
    private final List<HTTPHeader> headers;
    private final List<HTTPCookie> cookies;
    private final List<Param> params;

    private final Charset paramCharset;
    @Nullable
    private final Body<?> body;
    private final boolean acceptCompress;
    @Nullable
    private final PasswordAuthentication basicAuth;
    private final URL url;

    @Nullable
    private final String userAgent;
    @Nullable
    private final String referer;
    private final Duration timeout;

    HTTPRequest(AbstractHTTPRequestBuilder<?> builder) {
        method = builder.method;
        url = builder.url;
        headers = builder.headers;
        cookies = builder.cookies;
        userAgent = builder.userAgent;
        referer = builder.referer;
        paramCharset = builder.paramCharset;
        body = builder.body;
        acceptCompress = builder.acceptCompress;
        basicAuth = builder.basicAuth;
        params = builder.params;
        timeout = builder.timeout;
    }

    public HTTPMethod method() {
        return method;
    }

    public List<HTTPHeader> headers() {
        return headers;
    }

    /**
     * The cookie set by user.
     */
    public List<HTTPCookie> cookies() {
        return cookies;
    }

    /**
     * The user set user-agent (in Client or in Request). If not any custom user-agent was set, return empty optional.
     */
    public Optional<String> userAgent() {
        return Optional.ofNullable(userAgent);
    }

    /**
     * The referer set by user. If not any custom user-agent was set, return empty optional.
     */
    public Optional<String> referer() {
        return Optional.ofNullable(referer);
    }

    /**
     * The charset used to encode params to url query string.
     */
    public Charset paramCharset() {
        return paramCharset;
    }

    public Optional<Body<?>> body() {
        return Optional.ofNullable(body);
    }

    public boolean acceptCompress() {
        return acceptCompress;
    }

    /**
     * Basic auth for this request.
     *
     * @return if not exist, return empty Optional
     */
    public Optional<PasswordAuthentication> basicAuth() {
        return Optional.ofNullable(basicAuth);
    }

    public URL url() {
        return url;
    }

    public Duration timeout() {
        return timeout;
    }

    /**
     * Parameter to append to url.
     */
    public List<Param> params() {
        return params;
    }

}
