package net.dongliu.cute.http;

import net.dongliu.cute.http.internal.Asserts;
import net.dongliu.cute.http.internal.HonoredCookieManager;
import net.dongliu.cute.http.json.JsonMarshaller;
import org.checkerframework.checker.nullness.qual.Nullable;

import java.net.Authenticator;
import java.net.CookieHandler;
import java.net.Proxy;
import java.net.ProxySelector;
import java.security.KeyStore;
import java.time.Duration;
import java.util.concurrent.Executor;

import static java.util.Objects.requireNonNull;

/**
 * Builder to build Http Client.
 */
public class HTTPClientBuilder {
    @Nullable String name;
    @Nullable ProxySelector proxySelector = null;
    boolean useHttp2 = true;
    boolean verifyCert = true;
    boolean followRedirect = true;
    @Nullable Authenticator authenticator = null;
    CookieHandler cookieHandler = new HonoredCookieManager();
    @Nullable Executor executor = null;
    @Nullable KeyStore keyStore = null;
    // connect timeout, include ssl handshake etc.
    Duration connectTimeout = Duration.ofSeconds(5);

    // The response timeout. this filed can be override by per request setting
    Duration timeout = Duration.ofSeconds(10);
    @Nullable String userAgent = null;
    boolean acceptCompress = true;

    @Nullable JsonMarshaller jsonMarshaller;

    /**
     * Build a new http Client.
     *
     * @return the http client
     */
    public HTTPClient build() {
        return new HTTPClient(this);
    }

    /**
     * Set name for this http client
     */
    public HTTPClientBuilder name(String name) {
        this.name = requireNonNull(name);
        return this;
    }

    /**
     * Set a proxy selector for this client. If not set, will use no proxy.
     * Note: seems JDK HttpClient Only support http proxy.
     */
    public HTTPClientBuilder proxy(ProxySelector proxySelector) {
        this.proxySelector = requireNonNull(proxySelector);
        return this;
    }

    /**
     * Set a proxy for this client. If not set, will use no proxy.
     * Note: seems JDK HttpClient Only support http proxy.
     */
    public HTTPClientBuilder proxy(Proxy proxy) {
        requireNonNull(proxy);
        this.proxySelector = requireNonNull(ProxySelectors.staticSelector(proxy));
        return this;
    }

    /**
     * If use http2. default true
     */
    public HTTPClientBuilder useHttp2(boolean useHttp2) {
        this.useHttp2 = useHttp2;
        return this;
    }

    /**
     * If check https certificate, default true.
     */
    public HTTPClientBuilder verifyCert(boolean verify) {
        this.verifyCert = verify;
        return this;
    }

    /**
     * If follow redirects. Default true
     */
    public HTTPClientBuilder followRedirect(boolean followRedirect) {
        this.followRedirect = followRedirect;
        return this;
    }

    /**
     * Set timeout for connecting phase. The timeout include ssl handshake, etc. Default is 5 secs.
     */
    public HTTPClientBuilder connectTimeout(Duration timeout) {
        this.connectTimeout = Asserts.checkTimeout(timeout);
        return this;
    }

    /**
     * The timeout for one request to return response, do not include connect time. Default is 10 secs.
     * Note: The can be override by request setting.
     */
    public HTTPClientBuilder timeout(Duration timeout) {
        this.timeout = Asserts.checkTimeout(timeout);
        return this;
    }

    /**
     * The client user-agent.
     * Note: The can be override by request setting.
     */
    public HTTPClientBuilder userAgent(String userAgent) {
        this.userAgent = requireNonNull(userAgent);
        return this;
    }

    /**
     * Auto set Accept-Encoding header for request. Default is true.
     * Note: This setting can be override by per request setting.
     */
    public HTTPClientBuilder acceptCompress(boolean acceptCompress) {
        this.acceptCompress = acceptCompress;
        return this;
    }

    /**
     * Provide Authenticator for http basic or digest Authentication. If not set, will use system default Authenticator.
     *
     * @param authenticator the authenticator
     * @return self
     */
    public HTTPClientBuilder authenticator(Authenticator authenticator) {
        this.authenticator = requireNonNull(authenticator);
        return this;
    }

    /**
     * Provide CookieHandler for handle cookies.If not set, will use default in-memory store CookieManager.
     *
     * @param cookieHandler the cookieManager
     * @return self
     */
    public HTTPClientBuilder cookieHandler(CookieHandler cookieHandler) {
        this.cookieHandler = requireNonNull(cookieHandler);
        return this;
    }

    /**
     * Provide KeyStore for create ssl connection, trust the server certificate.
     *
     * @param keyStore the keyStore
     * @return self
     */
    public HTTPClientBuilder keyStore(KeyStore keyStore) {
        this.keyStore = requireNonNull(keyStore);
        return this;
    }

    /**
     * Set a custom executor, for asynchronous and dependent tasks.
     *
     * @param executor the executor
     * @return self
     */
    public HTTPClientBuilder executor(Executor executor) {
        this.executor = requireNonNull(executor);
        return this;
    }

    /**
     * Set json marshaller for this http client.
     * If not set, will try to find available JsonMarshaller implementation by spi; If no spi implementation was found,
     * a {@link net.dongliu.cute.http.exception.JsonMarshallerNotFoundException} would be thrown when handle with json.
     *
     * @param jsonMarshaller the JsonMarshaller
     */
    public HTTPClientBuilder jsonMarshaller(JsonMarshaller jsonMarshaller) {
        this.jsonMarshaller = requireNonNull(jsonMarshaller);
        return this;
    }
}
