package net.dongliu.cute.http;

import net.dongliu.commons.io.InputStreams;
import net.dongliu.commons.io.Readers;
import org.checkerframework.checker.nullness.qual.Nullable;

import java.io.*;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.function.Function;
import java.util.zip.GZIPInputStream;
import java.util.zip.Inflater;
import java.util.zip.InflaterInputStream;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Objects.requireNonNull;

/**
 * Raw blocking http response. The http headers already received, the http body not consumed, can be get as InputStream.
 * It you do not consume http response body, with {@link #toStringResponse()}, {@link #toBinaryResponse()},
 * {@link #writeTo(Path)} etc.., you need to close this raw response manually.
 *
 * This class is not thread-safe.
 *
 * @author Liu Dong
 */
public class HTTPResponseContext implements AutoCloseable {
    private final HTTPMethod method;
    private final URL url;
    private final ResponseInfo info;
    // user specified charset
    @Nullable
    private Charset charset = null;
    private boolean autoDecompress = true;

    HTTPResponseContext(HTTPMethod method, URL url, ResponseInfo info) {
        this.method = method;
        this.url = url;
        this.info = requireNonNull(info);
    }

    /**
     * Set response body read charset.
     * If not set, would get charset from response headers. If not found, would use UTF-8.
     *
     * @param charset the charset used to decode response body. Cannot be null
     */
    public HTTPResponseContext charset(Charset charset) {
        this.charset = requireNonNull(charset);
        return this;
    }

    /**
     * If decompress http response body automatically. Default True.
     */
    public HTTPResponseContext autoDecompress(boolean autoDecompress) {
        this.autoDecompress = autoDecompress;
        return this;
    }

    /**
     * Handle response body with handler, return a new response with content as handler result.
     * ResponseHandler should consume all InputStream data, or connection may close and cannot reuse.
     * The response is closed whether this call succeed or failed with exception.
     */
    public <T> HTTPResponse<T> handle(ResponseHandler<T> responseHandler) {
        ResponseInfo info = this.info;
        if (autoDecompress) {
            InputStream body = wrapCompressedInput(method, info.statusCode(), info.headers(), info.body());
            info = new ResponseInfo(info.statusCode(), info.headers(), body);
        }

        try (var body = info.body()) {
            T result = responseHandler.handle(info.statusCode(), info.headers(), body, this::getCharset);
            return new HTTPResponse<>(url, info.statusCode(), info.headers(), result);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    /**
     * Wrap response input stream if it is compressed
     */
    private InputStream wrapCompressedInput(HTTPMethod method, int status, HTTPHeaders headers, InputStream input) {
        // if has no body, some server still set content-encoding header,
        // GZIPInputStream wrap empty input stream will cause exception. we should check this
        if (noBody(method, status)) {
            return input;
        }

        var contentEncoding = headers.getHeader(HeaderNames.CONTENT_ENCODING).orElse("");

        //we should remove the content-encoding header here?
        switch (contentEncoding) {
            case "gzip":
                try {
                    return new GZIPInputStream(input);
                } catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            case "deflate":
                // Note: deflate implements may or may not wrap in zlib due to rfc confusing.
                // here deal with deflate without zlib header
                return new InflaterInputStream(input, new Inflater(true));
            case "identity":
            default:
                return input;
        }
    }

    private boolean noBody(HTTPMethod method, int status) {
        return method.equals(HTTPMethod.HEAD)
                || (status >= 100 && status < 200)
                || status == StatusCodes.NOT_MODIFIED || status == StatusCodes.NO_CONTENT;
    }

    /**
     * Convert to response, with body as text. The origin raw response will be closed
     */
    public HTTPResponse<String> toStringResponse() {
        return handle((statusCode, headers, body, charset) -> {
            try (Reader reader = new InputStreamReader(body, charset.get())) {
                return Readers.readAll(reader);
            }
        });
    }

    /**
     * Convert to response, with body as byte array
     */
    public HTTPResponse<byte[]> toBinaryResponse() {
        return handle((statusCode, headers, body, charset) -> body.readAllBytes());
    }

    /**
     * Write response body to file
     */
    public HTTPResponse<Path> writeTo(Path path) {
        return handle((statusCode, headers, body, charset) -> {
            try (var out = Files.newOutputStream(path)) {
                body.transferTo(out);
            }
            return null;
        });
    }

    /**
     * Write response body to OutputStream. OutputStream will not be closed.
     */
    public HTTPResponse<Void> writeTo(OutputStream out) {
        return handle((statusCode, headers, body, charset) -> {
            body.transferTo(out);
            return null;
        });
    }

    /**
     * Write response body to Writer, charset can be set using {@link #charset(Charset)},
     * or will use charset detected from response header if not set.
     * Writer will not be closed.
     */
    public HTTPResponse<Void> writeTo(Writer writer) {
        return handle((statusCode, headers, body, charset) -> {
            try (Reader reader = new InputStreamReader(body, charset.get())) {
                Readers.transferTo(reader, writer);
                return null;
            }
        });
    }

    /**
     * Consume and discard this response body.
     */
    public HTTPResponse<Void> discard() {
        return handle((statusCode, headers, body, charset) -> {
            InputStreams.discardAll(body);
            return null;
        });
    }

    /**
     * Handle response content as reader. The reader passed to handler will be closed when handler finished.
     *
     * @param handler the handler
     * @param <T> the response body type to convert to
     */
    public <T> HTTPResponse<T> handleAsReader(Function<? super Reader, T> handler) {
        return handle((statusCode, headers, body, charset) -> {
            try (Reader reader = new InputStreamReader(body, charset.get())) {
                return handler.apply(reader);
            }
        });
    }

    /**
     * Handle response content as InputStream. The InputStream passed to handler will be closed when handler finished.
     *
     * @param handler the handler
     * @param <T> the response body type to convert to
     */
    public <T> HTTPResponse<T> handleAsInput(Function<? super InputStream, T> handler) {
        return handle((statusCode, headers, body, charset) -> handler.apply(body));
    }

    private Charset getCharset() {
        if (charset != null) {
            return charset;
        }
        return info.headers().contentType()
                .flatMap(ContentType::charset)
                .orElse(UTF_8);
    }

    @Override
    public void close() {
        discard();
    }
}
