package net.dongliu.cute.http;

import net.dongliu.commons.Lazy;
import net.dongliu.cute.http.exception.RequestsException;
import net.dongliu.cute.http.internal.SSLContextFactories;
import org.checkerframework.checker.nullness.qual.Nullable;

import java.net.*;
import java.net.http.HttpClient;
import java.security.KeyStore;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;

import static java.net.http.HttpClient.Redirect.ALWAYS;
import static java.net.http.HttpClient.Redirect.NEVER;
import static java.net.http.HttpClient.Version.HTTP_1_1;
import static java.net.http.HttpClient.Version.HTTP_2;
import static java.util.Objects.requireNonNull;
import static java.util.concurrent.Executors.newCachedThreadPool;
import static net.dongliu.commons.Objects2.elvis;
import static net.dongliu.commons.concurrent.ThreadFactories.newDaemonThreadFactory;

/**
 * The http client.
 */
public class HTTPClient {
    private final String name;
    @Nullable
    private final ProxySelector proxySelector;
    private final boolean useHttp2;
    private final boolean verifyCert;
    private final boolean followRedirect;
    @Nullable
    private final Authenticator authenticator;
    private final CookieHandler cookieHandler;
    private final Executor executor;
    @Nullable
    private final KeyStore keyStore;
    private final Duration connectTimeout;

    private final Duration timeout;
    private final String userAgent;
    private final boolean acceptCompress;

    final HttpClient httpClient;

    private static final AtomicLong seq = new AtomicLong(0);

    HTTPClient(HTTPClientBuilder builder) {
        this.name = elvis(builder.name, () -> "HttpClient-" + seq.getAndIncrement());
        this.proxySelector = builder.proxySelector;
        this.useHttp2 = builder.useHttp2;
        this.verifyCert = builder.verifyCert;
        this.followRedirect = builder.followRedirect;
        this.timeout = builder.timeout;
        this.userAgent = elvis(builder.userAgent, () -> "HttpClient, Java " + System.getProperty("java.version"));
        this.acceptCompress = builder.acceptCompress;
        this.authenticator = builder.authenticator;
        this.cookieHandler = builder.cookieHandler;
        this.keyStore = builder.keyStore;
        this.connectTimeout = builder.connectTimeout;

        this.executor = elvis(builder.executor, () -> newCachedThreadPool(newDaemonThreadFactory(name)));

        var httpClientBuilder = HttpClient.newBuilder();
        if (!verifyCert) {
            httpClientBuilder.sslContext(SSLContextFactories.getTrustAllSSLContext());
        } else if (keyStore != null) {
            httpClientBuilder.sslContext(SSLContextFactories.getCustomTrustSSLContext(keyStore));
        }

        httpClientBuilder.executor(executor);
        httpClientBuilder.version(useHttp2 ? HTTP_2 : HTTP_1_1)
                .followRedirects(followRedirect ? ALWAYS : NEVER);
        if (proxySelector != null) {
            httpClientBuilder.proxy(proxySelector);
        }
        if (authenticator != null) {
            httpClientBuilder.authenticator(authenticator);
        }
        httpClientBuilder.cookieHandler(cookieHandler);
        httpClientBuilder.connectTimeout(connectTimeout);

        this.httpClient = httpClientBuilder.build();
    }

    /**
     * Get a new HttpClient Builder
     */
    public static HTTPClientBuilder builder() {
        return new HTTPClientBuilder();
    }


    private static final Lazy<HTTPClient> defaultClient = Lazy.of(() -> new HTTPClientBuilder().build());

    /**
     * Return the default Http Client with default settings. The client is shared across one jvm process.
     */
    public static HTTPClient defaultClient() {
        return defaultClient.get();
    }

    /**
     * Start a GET request
     */
    public HTTPRequestContext get(URL url) {
        return newRequest(HTTPMethod.GET, url);
    }

    /**
     * Start a POST request
     */
    public HTTPRequestContext post(URL url) {
        return newRequest(HTTPMethod.POST, url);
    }

    /**
     * Start a PUT request
     */
    public HTTPRequestContext put(URL url) {
        return newRequest(HTTPMethod.PUT, url);
    }

    /**
     * Start a DELETE request
     */
    public HTTPRequestContext delete(URL url) {
        return newRequest(HTTPMethod.DELETE, url);
    }

    /**
     * Start a HEAD request
     */
    public HTTPRequestContext head(URL url) {
        return newRequest(HTTPMethod.HEAD, url);
    }

    /**
     * Create new request with method and url
     */
    public HTTPRequestContext newRequest(HTTPMethod method, URL url) {
        return new HTTPRequestContext(this, method, url)
                .timeout(timeout)
                .acceptCompress(acceptCompress)
                .userAgent(userAgent);
    }

    /**
     * Start a GET request
     */
    public HTTPRequestContext get(String url) {
        return newRequest(HTTPMethod.GET, url);
    }

    /**
     * Start a POST request
     */
    public HTTPRequestContext post(String url) {
        return newRequest(HTTPMethod.POST, url);
    }

    /**
     * Start a PUT request
     */
    public HTTPRequestContext put(String url) {
        return newRequest(HTTPMethod.PUT, url);
    }

    /**
     * Start a DELETE request
     */
    public HTTPRequestContext delete(String url) {
        return newRequest(HTTPMethod.DELETE, url);
    }

    /**
     * Start a HEAD request
     */
    public HTTPRequestContext head(String url) {
        return newRequest(HTTPMethod.HEAD, url);
    }

    /**
     * Create new request with method and url
     */
    public HTTPRequestContext newRequest(HTTPMethod method, String urlStr) {
        URL url;
        try {
            url = new URL(requireNonNull(urlStr));
        } catch (MalformedURLException e) {
            throw new RequestsException("Resolve url error, url: " + urlStr, e);
        }
        return newRequest(method, url);
    }

    public String name() {
        return name;
    }

    public Optional<ProxySelector> proxy() {
        return Optional.ofNullable(proxySelector);
    }

    public boolean useHttp2() {
        return useHttp2;
    }

    public boolean verifyCert() {
        return verifyCert;
    }

    public boolean followRedirect() {
        return followRedirect;
    }

    public Duration timeout() {
        return timeout;
    }

    public Duration connectTimeout() {
        return connectTimeout;
    }

    /**
     * The user-agent of this http client.
     */
    public String userAgent() {
        return userAgent;
    }

    public boolean acceptCompress() {
        return acceptCompress;
    }

    public Optional<Authenticator> authenticator() {
        return Optional.ofNullable(authenticator);
    }

    public CookieHandler cookieHandler() {
        return cookieHandler;
    }

    public Optional<KeyStore> keyStore() {
        return Optional.ofNullable(keyStore);
    }

    /**
     * The thread pool used for http executing.
     */
    public Executor executor() {
        return executor;
    }
}
