package net.dongliu.cute.http;

import net.dongliu.commons.concurrent.Futures;
import net.dongliu.commons.io.Closeables;
import net.dongliu.cute.http.internal.AsyncInflater;
import net.dongliu.cute.http.internal.ByteBuffers;
import org.checkerframework.checker.nullness.qual.Nullable;

import java.io.IOException;
import java.net.http.HttpResponse;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Flow;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.StandardOpenOption.CREATE;
import static java.nio.file.StandardOpenOption.WRITE;
import static java.util.Objects.requireNonNull;

/**
 * Async Http Response.
 */
public class AsyncHTTPResponseContext {
    private final HTTPMethod method;
    private final CompletableFuture<AsyncResponseInfo> infoFuture;
    // user specified charset
    @Nullable
    private Charset charset = null;
    private boolean autoDecompress = true;

    AsyncHTTPResponseContext(HTTPMethod method, CompletableFuture<AsyncResponseInfo> infoFuture) {
        this.method = method;
        this.infoFuture = infoFuture;
    }

    /**
     * Set response read charset.
     * If not set, would get charset from response headers. If not found, would use UTF-8.
     *
     * @param charset the charset used to decode response body. Cannot be null
     */
    public AsyncHTTPResponseContext charset(Charset charset) {
        this.charset = requireNonNull(charset);
        return this;
    }

    public AsyncHTTPResponseContext autoDecompress(boolean autoDecompress) {
        this.autoDecompress = autoDecompress;
        return this;
    }

    /**
     * Handle the response, return a response future. If any exceptions were thrown during call handler, will return a
     * exceptional Future.
     *
     * @param handler the response handler
     * @param <T>     type return value type
     */
    public <T> CompletableFuture<HTTPResponse<T>> handle(AsyncHTTPResponseHandler<T> handler) {
        requireNonNull(handler);
        return infoFuture.thenCompose(info -> {
            AsyncResponseInfo asyncResponseInfo = info;
            if (autoDecompress) {
                var publisher = wrapCompressedPublisher(method, info.statusCode(), info.headers(), info.body());
                asyncResponseInfo = new AsyncResponseInfo(info.url(), info.statusCode(), info.headers(), publisher);
            }

            var finalInfo = asyncResponseInfo;
            try {
                handler.onHeader(info.statusCode(), info.headers(), () -> this.getCharset(finalInfo));
            } catch (Throwable e) {
                return Futures.error(e);
            }
            var bodySubscriber = new BodyHandlerSubscriber<>(handler);
            info.body().subscribe(bodySubscriber);
            var bodyFuture = bodySubscriber.getBody().toCompletableFuture();
            return bodyFuture.thenApply(body -> new HTTPResponse<>(
                    info.url(),
                    finalInfo.statusCode(),
                    finalInfo.headers(),
                    body
            ));
        });
    }

    /**
     * Treat the response body as String.
     *
     * @return the response Future
     */
    public CompletableFuture<HTTPResponse<String>> toStringResponse() {
        return handle(new AsyncHTTPResponseHandler<>() {
            private Charset charset;
            private final List<ByteBuffer> buffers = new ArrayList<>();

            @Override
            public void onHeader(int statusCode, HTTPHeaders headers, Supplier<Charset> charset) {
                this.charset = charset.get();
            }

            @Override
            public void onBodyChunk(ByteBuffer buffer) {
                buffers.add(buffer);
            }

            @Override
            public String onBodyEnd() {
                return ByteBuffers.toString(buffers, charset);
            }
        });
    }

    /**
     * Collect the response body to byte array.
     *
     * @return the response Future
     */
    public CompletableFuture<HTTPResponse<byte[]>> toBinaryResponse() {
        return handle(new AsyncHTTPResponseHandler<>() {
            private final List<ByteBuffer> buffers = new ArrayList<>();

            @Override
            public void onBodyChunk(ByteBuffer buffer) {
                buffers.add(buffer);
            }

            @Override
            public byte[] onBodyEnd() {
                return ByteBuffers.toByteArray(buffers);
            }
        });
    }

    /**
     * Discard all response body.
     *
     * @return the response Future
     */
    public CompletableFuture<HTTPResponse<Void>> discard() {
        return handle(new AsyncHTTPResponseHandler<>() {
            @Override
            public void onBodyChunk(ByteBuffer buffer) {
            }

            @Override
            public Void onBodyEnd() {
                return null;
            }
        });
    }

    /**
     * Write response body to file.
     *
     * @param path the file path
     * @return the response future
     */
    public CompletableFuture<HTTPResponse<Path>> writeTo(Path path) {
        return handle(new AsyncHTTPResponseHandler<>() {
            private FileChannel channel;

            @Override
            public void onHeader(int statusCode, HTTPHeaders headers, Supplier<Charset> charset) throws IOException {
                channel = FileChannel.open(path, CREATE, WRITE);
            }

            @Override
            public void onBodyChunk(ByteBuffer buffer) throws IOException {
                try {
                    channel.write(buffer);
                } catch (Throwable e) {
                    Closeables.closeQuietly(channel);
                    throw e;
                }
            }

            @Override
            public Path onBodyEnd() {
                Closeables.closeQuietly(channel);
                return path;
            }
        });
    }

    private Charset getCharset(AsyncResponseInfo info) {
        if (charset != null) {
            return charset;
        }
        return info.headers().contentType()
                .flatMap(ContentType::charset)
                .orElse(UTF_8);
    }


    private Flow.Publisher<List<ByteBuffer>> wrapCompressedPublisher(HTTPMethod method, int status, HTTPHeaders headers,
                                                                     Flow.Publisher<List<ByteBuffer>> publisher) {
        if (responseHasNoBody(method, status)) {
            return publisher;
        }
        var contentEncoding = headers.getHeader(HeaderNames.CONTENT_ENCODING).orElse("").trim();
        switch (contentEncoding) {
            case "gzip":
                return subscriber -> publisher.subscribe(new DecompressedBodySubscriber(subscriber, AsyncInflater.GZIP));
            case "deflate":
                // Note: deflate implements may or may not wrap in zlib due to rfc confusing.
                // here deal with deflate without zlib header
                return subscriber -> publisher.subscribe(new DecompressedBodySubscriber(subscriber, AsyncInflater.ZLIB));
            case "identity":
            default:
                return publisher;
        }
    }

    private boolean responseHasNoBody(HTTPMethod method, int status) {
        return method.equals(HTTPMethod.HEAD)
                || (status >= 100 && status < 200)
                || status == StatusCodes.NOT_MODIFIED || status == StatusCodes.NO_CONTENT;
    }

    /**
     * Delegated Body Subscriber which decompress response content.
     */
    private static class DecompressedBodySubscriber implements Flow.Subscriber<List<ByteBuffer>> {

        private final Flow.Subscriber<? super List<ByteBuffer>> subscriber;
        private final AsyncInflater asyncInflater;

        public DecompressedBodySubscriber(Flow.Subscriber<? super List<ByteBuffer>> subscriber, int wrapper) {
            this.subscriber = subscriber;
            this.asyncInflater = new AsyncInflater(wrapper);
        }

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            subscriber.onSubscribe(subscription);
        }

        @Override
        public void onNext(List<ByteBuffer> item) {
            var buffers = new ArrayList<ByteBuffer>();
            for (var in : item) {
                asyncInflater.decode(in, buffers::add);
            }
            subscriber.onNext(buffers);
        }

        @Override
        public void onError(Throwable throwable) {
            asyncInflater.onFinish();
            subscriber.onError(throwable);
        }

        @Override
        public void onComplete() {
            asyncInflater.onFinish();
            subscriber.onComplete();
        }
    }

    /**
     * Adapter AsyncHTTPResponseHandler to work with HttpClient Async API
     *
     * @param <T>
     */
    private static class BodyHandlerSubscriber<T> implements HttpResponse.BodySubscriber<T> {
        private final AsyncHTTPResponseHandler<T> handler;
        private Flow.Subscription subscription;
        private final CompletableFuture<T> result = new CompletableFuture<>();
        private final AtomicBoolean subscribed = new AtomicBoolean();

        public BodyHandlerSubscriber(AsyncHTTPResponseHandler<T> handler) {
            this.handler = requireNonNull(handler);
        }

        @Override
        public CompletionStage<T> getBody() {
            return result;
        }

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            if (!subscribed.compareAndSet(false, true)) {
                subscription.cancel();
            } else {
                this.subscription = subscription;
                subscription.request(1);
            }
        }

        @Override
        public void onNext(List<ByteBuffer> items) {
            for (ByteBuffer item : items) {
                try {
                    handler.onBodyChunk(item);
                } catch (Throwable e) {
                    result.completeExceptionally(e);
                    subscription.cancel();
                    return;
                }
            }
            subscription.request(1);
        }

        @Override
        public void onError(Throwable throwable) {
            result.completeExceptionally(throwable);
        }

        @Override
        public void onComplete() {
            try {
                result.complete(handler.onBodyEnd());
            } catch (Throwable e) {
                result.completeExceptionally(e);
            }
        }
    }
}
