package net.azae.xray;

import org.junit.platform.engine.TestExecutionResult;
import org.junit.platform.engine.reporting.ReportEntry;
import org.junit.platform.launcher.TestExecutionListener;
import org.junit.platform.launcher.TestIdentifier;
import org.junit.platform.launcher.TestPlan;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.HttpsURLConnection;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.*;

public class XrayListener implements TestExecutionListener {
    private final static Logger logger = LoggerFactory.getLogger(XrayListener.class);
    // TODO : get base URL from conf
    private static final String baseUrl = "https://xray.cloud.xpand-it.com";
    private XrayTestSuite testSuite;

    public XrayListener() {
        logger.info("XrayListener created");
    }

    public static void publishToXray(String token, String inputJson) {
        logger.debug("Publish input: " + inputJson);
        String response = performPostCall(baseUrl + "/api/v1/import/execution", inputJson, token);
        logger.debug("Publish response: " + response);
    }

    public static Optional<String> getXrayToken() {
        String xray_client_secret = getEnv("XRAY_CLIENT_SECRET");
        String xray_client_id = getEnv("XRAY_CLIENT_ID");
        String json = "{ " +
                "\"client_id\": \"" + xray_client_id + "\", " +
                "\"client_secret\": \"" + xray_client_secret + "\"" +
                " }";
        String response = performPostCall(baseUrl + "/api/v1/authenticate", json, null);
        return Optional.of(response.replace("\"", ""));
    }

    private static String getEnv(String env) {
        String variable = System.getenv(env);
        if (variable == null) throw new RuntimeException("Missing " + env + " environment variable");
        return variable;
    }

    static List<String> extractRunningEnvironment(Map<String, String> env) {
        List<String> runningEnvironment = new ArrayList<>();
        if (env.containsKey("CI"))
            runningEnvironment.add("gitlab");
        else if (env.containsKey("JENKINS_URL")) {
            runningEnvironment.add("jenkins");
        } else {
            runningEnvironment.add("development");
        }
        return runningEnvironment;
    }


    public static String performPostCall(String requestURL, String postDataParams, String token) {
        URL url;
        String response = "";
        try {
            url = new URL(requestURL);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setReadTimeout(15000);
            conn.setConnectTimeout(15000);
            conn.setRequestMethod("POST");
            conn.setRequestProperty("Content-Type", "application/json");
            if (token != null)
                conn.setRequestProperty("Authorization", "Bearer " + token);

            conn.setDoInput(true);
            conn.setDoOutput(true);
            OutputStream os = conn.getOutputStream();
            BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(os, "UTF-8"));
            writer.write(postDataParams);
            writer.flush();
            writer.close();
            os.close();
            int responseCode = conn.getResponseCode();
            if (responseCode == HttpsURLConnection.HTTP_OK) {
                String line;
                BufferedReader br = new BufferedReader(new InputStreamReader(conn.getInputStream()));
                while ((line = br.readLine()) != null) {
                    response += line;
                }
            } else {
                response = "";
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return response;
    }

    private String getPostDataString(HashMap<String, String> params) throws UnsupportedEncodingException {
        StringBuilder result = new StringBuilder();
        boolean first = true;
        for (Map.Entry<String, String> entry : params.entrySet()) {
            if (first)
                first = false;
            else
                result.append("&");
            result.append(URLEncoder.encode(entry.getKey(), "UTF-8"));
            result.append("=");
            result.append(URLEncoder.encode(entry.getValue(), "UTF-8"));
        }
        return result.toString();
    }

    @Override
    public void reportingEntryPublished(final TestIdentifier testIdentifier, final ReportEntry entry) {
        String uniqueId = testIdentifier.getUniqueId();
        XrayTest test = getTestSuite(System.getenv()).getTestFromId(uniqueId);
        updateXrayTestWithAnnotation(entry, test);
        getTestSuite(System.getenv()).putTest(test);
    }

    public void updateXrayTestWithAnnotation(ReportEntry entry, XrayTest test) {
        extractXrayTestKey(entry, XrayExtension.XRAY_TEST_ID).ifPresent(testKey -> {
            test.setTestKey(testKey);
            logger.info("Xray test: " + testKey);
        });
        extractXrayTestKey(entry, XrayExtension.UPLOAD_TO_XRAY).ifPresent(up -> {
            test.setUpload(Boolean.parseBoolean(up));
            logger.info("Xray upload: " + up);
        });
    }

    public Optional<String> extractXrayTestKey(ReportEntry entry, String key) {
        Map<String, String> map = entry.getKeyValuePairs();
        if (map.containsKey(key)) {
            return Optional.of(map.get(key));
        }
        return Optional.empty();
    }

    @Override
    public void executionFinished(TestIdentifier testIdentifier, TestExecutionResult testExecutionResult) {
        if (!testIdentifier.isTest()) {
            return;
        }

        logger.debug("Execution finished: " + testIdentifier.getDisplayName() + " - " +
                testExecutionResult.getStatus().toString());

        XrayStatus status = XrayStatus.fromJunitExecution(testExecutionResult.getStatus());
        String uniqueId = testIdentifier.getUniqueId();
        XrayTest test = getTestSuite(System.getenv()).getTestFromId(uniqueId);
        test.setStatus(status);
        test.setComment(extractComment(testIdentifier, testExecutionResult));
        getTestSuite(System.getenv()).putTest(test);
    }

    public String extractComment(TestIdentifier testIdentifier, TestExecutionResult testExecutionResult) {
        if (testExecutionResult.getThrowable().isPresent()) {
            String message = testExecutionResult.getThrowable().get().getMessage();
            if (testIdentifier.getParentId().isPresent()) {
                return message + testIdentifier.getParentId().get();
            }
            return message;
        }
        return testIdentifier.getDisplayName();
    }

    public void testPlanExecutionFinished(TestPlan testPlan) {
        logger.info("Uploading results data to Xray...");
        getTestSuite(System.getenv()).cloneToUpload().ifPresent(suite -> {
            getXrayToken().ifPresent(token -> {
                Mapper.ToJsonAsOptional(suite).ifPresent(json -> {
                    publishToXray(token, json);
                });
            });
        });
        logger.info("done");
    }

    private XrayTestSuite getTestSuite(Map<String, String> env) {
        if (testSuite == null) {
            List<String> runningEnvironment = extractRunningEnvironment(env);
            testSuite = XrayTestSuite.withInfos("Automated test", runningEnvironment);
        }
        return testSuite;
    }
}
