package net.azae.xray;

import org.junit.platform.engine.TestExecutionResult;
import org.junit.platform.engine.reporting.ReportEntry;
import org.junit.platform.launcher.TestExecutionListener;
import org.junit.platform.launcher.TestIdentifier;
import org.junit.platform.launcher.TestPlan;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class XrayListener implements TestExecutionListener {
    private final static Logger logger = LoggerFactory.getLogger(XrayListener.class);
    // TODO : get base URL from conf
    private static String baseUrl = "https://xray.cloud.xpand-it.com";
    private XrayTestSuite testSuite;

    public static void publishToXray(HttpClient client, String token, String inputJson) {
        logger.debug("Publish input: " + inputJson);
        HttpRequest request = HttpRequest.newBuilder(URI.create(baseUrl + "/api/v1/import/execution"))
                .header("Content-Type", "application/json")
                .headers("Authorization", "Bearer " + token)
                .POST(HttpRequest.BodyPublishers.ofString(inputJson)).build();

        HttpResponse<String> response = null;
        try {
            response = client.send(request, HttpResponse.BodyHandlers.ofString());
            logger.debug("Publish response: " + response.body());
        } catch (IOException | InterruptedException e) {
            e.printStackTrace();
        }
    }

    public static Optional<String> getXrayToken(HttpClient client) {
        String xray_client_secret = getEnv("XRAY_CLIENT_SECRET");
        String xray_client_id = getEnv("XRAY_CLIENT_ID");
        String json = "{ " +
                "\"client_id\": \"" + xray_client_id + "\", " +
                "\"client_secret\": \"" + xray_client_secret + "\"" +
                " }";
        HttpRequest tokenRequest = HttpRequest.newBuilder(URI.create(baseUrl + "/api/v1/authenticate"))
                .header("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers.ofString(json)).build();
        HttpResponse<String> tokenResponse = null;
        try {
            tokenResponse = client.send(tokenRequest, HttpResponse.BodyHandlers.ofString());
        } catch (IOException | InterruptedException e) {
            e.printStackTrace();
            return Optional.empty();
        }
        return Optional.of(tokenResponse.body().replace("\"", ""));

    }

    private static String getEnv(String env) {
        String variable = System.getenv(env);
        if (variable == null) throw new RuntimeException("Missing " + env + " environment variable");
        return variable;
    }

    static List<String> extractRunningEnvironment(Map<String, String> env) {
        List<String> runningEnvironment = new ArrayList<>();
        if (env.keySet().contains("CI"))
            runningEnvironment.add("gitlab");
        else if (env.keySet().contains("JENKINS_URL")) {
            runningEnvironment.add("jenkins");
        } else {
            runningEnvironment.add("development");
        }
        return runningEnvironment;
    }

    @Override
    public void reportingEntryPublished(final TestIdentifier testIdentifier, final ReportEntry entry) {
        String uniqueId = testIdentifier.getUniqueId();
        XrayTest test = getTestSuite(System.getenv()).getTestFromId(uniqueId);
        updateXrayTestWithAnnotation(entry, test);
        getTestSuite(System.getenv()).putTest(test);
    }

    public void updateXrayTestWithAnnotation(ReportEntry entry, XrayTest test) {
        extractXrayTestKey(entry, XrayExtension.XRAY_TEST_ID).ifPresent(testKey -> {
            test.setTestKey(testKey);
            logger.info("Xray test: " + testKey);
        });
        extractXrayTestKey(entry, XrayExtension.UPLOAD_TO_XRAY).ifPresent(up -> {
            test.setUpload(Boolean.parseBoolean(up));
            logger.info("Xray upload: " + up);
        });
    }

    public Optional<String> extractXrayTestKey(ReportEntry entry, String key) {
        Map<String, String> map = entry.getKeyValuePairs();
        if (map.containsKey(key)) {
            return Optional.of(map.get(key));
        }
        return Optional.empty();
    }

    @Override
    public void executionFinished(TestIdentifier testIdentifier, TestExecutionResult testExecutionResult) {
        if (!testIdentifier.isTest()) {
            return;
        }

        logger.debug("Execution finished: " + testIdentifier.getDisplayName() + " - " +
                testExecutionResult.getStatus().toString());

        XrayStatus status = XrayStatus.fromJunitExecution(testExecutionResult.getStatus());
        String uniqueId = testIdentifier.getUniqueId();
        XrayTest test = getTestSuite(System.getenv()).getTestFromId(uniqueId);
        test.setStatus(status);
        test.setComment(extractComment(testIdentifier, testExecutionResult));
        getTestSuite(System.getenv()).putTest(test);
    }

    public String extractComment(TestIdentifier testIdentifier, TestExecutionResult testExecutionResult) {
        if (testExecutionResult.getThrowable().isPresent()) {
            String message = testExecutionResult.getThrowable().get().getMessage();
            if (testIdentifier.getParentId().isPresent()) {
                return message + testIdentifier.getParentId().get();
            }
            return message;
        }
        return testIdentifier.getDisplayName();
    }

    public void testPlanExecutionFinished(TestPlan testPlan) {
        logger.info("Uploading results data to Xray...");
        HttpClient client = HttpClient.newBuilder().build();
        getTestSuite(System.getenv()).cloneToUpload().ifPresent(suite -> {
            getXrayToken(client).ifPresent(token -> {
                Mapper.ToJsonAsOptional(suite).ifPresent(json -> {
                    publishToXray(client, token, json);
                });
            });
        });
        logger.info("done");
    }

    private XrayTestSuite getTestSuite(Map<String, String> env) {
        if (testSuite == null) {
            List<String> runningEnvironment = extractRunningEnvironment(env);
            testSuite = XrayTestSuite.withInfos("Automated test", runningEnvironment);
        }
        return testSuite;
    }
}
