package net.axay.memoire

import kotlinx.atomicfu.update
import kotlinx.datetime.Clock

class MemoryCache<K, V>(
    validationConfig: CacheValidationConfig
) : SyncCache<K, K, V, V>(validationConfig) {

    private val internalMap = HashMap<K, CacheEntry<V>>()

    override fun internalGet(key: K): V? {
        val entry = internalMap[key] ?: return null
        entry.accessTime.update { Clock.System.now() }
        return entry.value
    }

    override fun internalContains(key: K): Boolean {
        return key in internalMap
    }

    override fun internalPut(key: K, value: V) {
        internalMap[key] = CacheEntry(value)
    }

    override fun internalPutAndGet(key: K, value: V): V? {
        internalPut(key, value)
        return value
    }

    override fun internalRemove(key: K): Boolean {
        return (key in internalMap).also { internalMap -= key }
    }

    override fun internalGetAndRemove(key: K): V? {
        return internalMap.remove(key)?.value
    }

    override fun internalEvict() {
        internalMap.keys.forEach(::internalRemoveIfInvalid)
    }

    override fun internalIsInvalid(key: K): Boolean? {
        val entry = internalMap[key] ?: return null

        return !validationConfig.isValid(entry.writeTime, entry.accessTime.value)
    }

    override val K.internal: K
        get() = this
}
