package net.adoptopenjdk.v3.api;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * A version bound. A version bound may be inclusive or exclusive.
 */
@SuppressWarnings({"all"})
public final class AOV3VersionBound implements AOV3VersionBoundType {
  private final String name;
  private final boolean exclusive;

  private AOV3VersionBound(Optional<String> name, boolean exclusive) {
    this.name = name.orElse(null);
    this.exclusive = exclusive;
  }

  private AOV3VersionBound(AOV3VersionBound original, String name, boolean exclusive) {
    this.name = name;
    this.exclusive = exclusive;
  }

  /**
   * @return The name of the bound
   */
  @Override
  public Optional<String> name() {
    return Optional.ofNullable(name);
  }

  /**
   * @return {@code true} iff the bound is exclusive
   */
  @Override
  public boolean isExclusive() {
    return exclusive;
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link AOV3VersionBoundType#name() name} attribute.
   * @param value The value for name
   * @return A modified copy of {@code this} object
   */
  public final AOV3VersionBound withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (Objects.equals(this.name, newValue)) return this;
    return new AOV3VersionBound(this, newValue, this.exclusive);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link AOV3VersionBoundType#name() name} attribute.
   * An equality check is used on inner nullable value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for name
   * @return A modified copy of {@code this} object
   */
  public final AOV3VersionBound withName(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.name, value)) return this;
    return new AOV3VersionBound(this, value, this.exclusive);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3VersionBoundType#isExclusive() exclusive} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for exclusive
   * @return A modified copy of the {@code this} object
   */
  public final AOV3VersionBound withExclusive(boolean value) {
    if (this.exclusive == value) return this;
    return new AOV3VersionBound(this, this.name, value);
  }

  /**
   * This instance is equal to all instances of {@code AOV3VersionBound} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof AOV3VersionBound
        && equalTo((AOV3VersionBound) another);
  }

  private boolean equalTo(AOV3VersionBound another) {
    return Objects.equals(name, another.name)
        && exclusive == another.exclusive;
  }

  /**
   * Computes a hash code from attributes: {@code name}, {@code exclusive}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(name);
    h += (h << 5) + Boolean.hashCode(exclusive);
    return h;
  }

  /**
   * Prints the immutable value {@code AOV3VersionBound} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("AOV3VersionBound{");
    if (name != null) {
      builder.append("name=").append(name);
    }
    if (builder.length() > 17) builder.append(", ");
    builder.append("exclusive=").append(exclusive);
    return builder.append("}").toString();
  }

  /**
   * Construct a new immutable {@code AOV3VersionBound} instance.
   * @param name The value for the {@code name} attribute
   * @param exclusive The value for the {@code exclusive} attribute
   * @return An immutable AOV3VersionBound instance
   */
  public static AOV3VersionBound of(Optional<String> name, boolean exclusive) {
    return new AOV3VersionBound(name, exclusive);
  }

  /**
   * Creates an immutable copy of a {@link AOV3VersionBoundType} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable AOV3VersionBound instance
   */
  public static AOV3VersionBound copyOf(AOV3VersionBoundType instance) {
    if (instance instanceof AOV3VersionBound) {
      return (AOV3VersionBound) instance;
    }
    return AOV3VersionBound.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link AOV3VersionBound AOV3VersionBound}.
   * <pre>
   * AOV3VersionBound.builder()
   *    .setName(String) // optional {@link AOV3VersionBoundType#name() name}
   *    .setExclusive(boolean) // required {@link AOV3VersionBoundType#isExclusive() exclusive}
   *    .build();
   * </pre>
   * @return A new AOV3VersionBound builder
   */
  public static AOV3VersionBound.Builder builder() {
    return new AOV3VersionBound.Builder();
  }

  /**
   * Builds instances of type {@link AOV3VersionBound AOV3VersionBound}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_EXCLUSIVE = 0x1L;
    private long initBits = 0x1L;

    private String name;
    private boolean exclusive;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code AOV3VersionBoundType} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(AOV3VersionBoundType instance) {
      Objects.requireNonNull(instance, "instance");
      Optional<String> nameOptional = instance.name();
      if (nameOptional.isPresent()) {
        setName(nameOptional);
      }
      setExclusive(instance.isExclusive());
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3VersionBoundType#name() name} to name.
     * @param name The value for name
     * @return {@code this} builder for chained invocation
     */
    public final Builder setName(String name) {
      this.name = Objects.requireNonNull(name, "name");
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3VersionBoundType#name() name} to name.
     * @param name The value for name
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setName(Optional<String> name) {
      this.name = name.orElse(null);
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3VersionBoundType#isExclusive() exclusive} attribute.
     * @param exclusive The value for exclusive 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setExclusive(boolean exclusive) {
      this.exclusive = exclusive;
      initBits &= ~INIT_BIT_EXCLUSIVE;
      return this;
    }

    /**
     * Builds a new {@link AOV3VersionBound AOV3VersionBound}.
     * @return An immutable instance of AOV3VersionBound
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public AOV3VersionBound build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new AOV3VersionBound(null, name, exclusive);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_EXCLUSIVE) != 0) attributes.add("exclusive");
      return "Cannot build AOV3VersionBound, some of required attributes are not set " + attributes;
    }
  }
}
