package net.adoptopenjdk.v3.api;

import java.math.BigInteger;
import java.net.URI;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Information about a release.
 */
@SuppressWarnings({"all"})
public final class AOV3Release implements AOV3ReleaseType {
  private final List<AOV3BinaryType> binaries;
  private final BigInteger downloadCount;
  private final String id;
  private final URI releaseLink;
  private final String releaseName;
  private final AOV3ReleaseKind releaseType;
  private final AOV3Source source;
  private final OffsetDateTime timestamp;
  private final OffsetDateTime updatedAt;
  private final AOV3Vendor vendor;
  private final AOV3VersionData versionData;

  private AOV3Release(
      List<AOV3BinaryType> binaries,
      BigInteger downloadCount,
      String id,
      URI releaseLink,
      String releaseName,
      AOV3ReleaseKind releaseType,
      AOV3Source source,
      OffsetDateTime timestamp,
      OffsetDateTime updatedAt,
      AOV3Vendor vendor,
      AOV3VersionData versionData) {
    this.binaries = binaries;
    this.downloadCount = downloadCount;
    this.id = id;
    this.releaseLink = releaseLink;
    this.releaseName = releaseName;
    this.releaseType = releaseType;
    this.source = source;
    this.timestamp = timestamp;
    this.updatedAt = updatedAt;
    this.vendor = vendor;
    this.versionData = versionData;
  }

  /**
   * @return The release binaries
   */
  @Override
  public List<AOV3BinaryType> binaries() {
    return binaries;
  }

  /**
   * @return The number of times the release has been downloaded
   */
  @Override
  public BigInteger downloadCount() {
    return downloadCount;
  }

  /**
   * @return The release identifier
   */
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The release link
   */
  @Override
  public URI releaseLink() {
    return releaseLink;
  }

  /**
   * @return The release name
   */
  @Override
  public String releaseName() {
    return releaseName;
  }

  /**
   * @return The release availability type
   */
  @Override
  public AOV3ReleaseKind releaseType() {
    return releaseType;
  }

  /**
   * @return The source information, if any
   */
  @Override
  public Optional<AOV3Source> source() {
    return Optional.ofNullable(source);
  }

  /**
   * @return The release timestamp
   */
  @Override
  public OffsetDateTime timestamp() {
    return timestamp;
  }

  /**
   * @return The release last modification time
   */
  @Override
  public OffsetDateTime updatedAt() {
    return updatedAt;
  }

  /**
   * @return The release vendor
   */
  @Override
  public AOV3Vendor vendor() {
    return vendor;
  }

  /**
   * @return The release version information
   */
  @Override
  public AOV3VersionData versionData() {
    return versionData;
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link AOV3ReleaseType#binaries() binaries}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final AOV3Release withBinaries(AOV3BinaryType... elements) {
    List<AOV3BinaryType> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new AOV3Release(
        newValue,
        this.downloadCount,
        this.id,
        this.releaseLink,
        this.releaseName,
        this.releaseType,
        this.source,
        this.timestamp,
        this.updatedAt,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link AOV3ReleaseType#binaries() binaries}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of binaries elements to set
   * @return A modified copy of {@code this} object
   */
  public final AOV3Release withBinaries(Iterable<? extends AOV3BinaryType> elements) {
    if (this.binaries == elements) return this;
    List<AOV3BinaryType> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new AOV3Release(
        newValue,
        this.downloadCount,
        this.id,
        this.releaseLink,
        this.releaseName,
        this.releaseType,
        this.source,
        this.timestamp,
        this.updatedAt,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ReleaseType#downloadCount() downloadCount} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for downloadCount
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Release withDownloadCount(BigInteger value) {
    BigInteger newValue = Objects.requireNonNull(value, "downloadCount");
    if (this.downloadCount.equals(newValue)) return this;
    return new AOV3Release(
        this.binaries,
        newValue,
        this.id,
        this.releaseLink,
        this.releaseName,
        this.releaseType,
        this.source,
        this.timestamp,
        this.updatedAt,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ReleaseType#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Release withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new AOV3Release(
        this.binaries,
        this.downloadCount,
        newValue,
        this.releaseLink,
        this.releaseName,
        this.releaseType,
        this.source,
        this.timestamp,
        this.updatedAt,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ReleaseType#releaseLink() releaseLink} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for releaseLink
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Release withReleaseLink(URI value) {
    if (this.releaseLink == value) return this;
    URI newValue = Objects.requireNonNull(value, "releaseLink");
    return new AOV3Release(
        this.binaries,
        this.downloadCount,
        this.id,
        newValue,
        this.releaseName,
        this.releaseType,
        this.source,
        this.timestamp,
        this.updatedAt,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ReleaseType#releaseName() releaseName} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for releaseName
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Release withReleaseName(String value) {
    String newValue = Objects.requireNonNull(value, "releaseName");
    if (this.releaseName.equals(newValue)) return this;
    return new AOV3Release(
        this.binaries,
        this.downloadCount,
        this.id,
        this.releaseLink,
        newValue,
        this.releaseType,
        this.source,
        this.timestamp,
        this.updatedAt,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ReleaseType#releaseType() releaseType} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for releaseType
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Release withReleaseType(AOV3ReleaseKind value) {
    if (this.releaseType == value) return this;
    AOV3ReleaseKind newValue = Objects.requireNonNull(value, "releaseType");
    if (this.releaseType.equals(newValue)) return this;
    return new AOV3Release(
        this.binaries,
        this.downloadCount,
        this.id,
        this.releaseLink,
        this.releaseName,
        newValue,
        this.source,
        this.timestamp,
        this.updatedAt,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link AOV3ReleaseType#source() source} attribute.
   * @param value The value for source
   * @return A modified copy of {@code this} object
   */
  public final AOV3Release withSource(AOV3Source value) {
    AOV3Source newValue = Objects.requireNonNull(value, "source");
    if (this.source == newValue) return this;
    return new AOV3Release(
        this.binaries,
        this.downloadCount,
        this.id,
        this.releaseLink,
        this.releaseName,
        this.releaseType,
        newValue,
        this.timestamp,
        this.updatedAt,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link AOV3ReleaseType#source() source} attribute.
   * A shallow reference equality check is used on unboxed optional value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for source
   * @return A modified copy of {@code this} object
   */
  @SuppressWarnings("unchecked") // safe covariant cast
  public final AOV3Release withSource(Optional<? extends AOV3Source> optional) {
    AOV3Source value = optional.orElse(null);
    if (this.source == value) return this;
    return new AOV3Release(
        this.binaries,
        this.downloadCount,
        this.id,
        this.releaseLink,
        this.releaseName,
        this.releaseType,
        value,
        this.timestamp,
        this.updatedAt,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ReleaseType#timestamp() timestamp} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for timestamp
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Release withTimestamp(OffsetDateTime value) {
    if (this.timestamp == value) return this;
    OffsetDateTime newValue = Objects.requireNonNull(value, "timestamp");
    return new AOV3Release(
        this.binaries,
        this.downloadCount,
        this.id,
        this.releaseLink,
        this.releaseName,
        this.releaseType,
        this.source,
        newValue,
        this.updatedAt,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ReleaseType#updatedAt() updatedAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for updatedAt
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Release withUpdatedAt(OffsetDateTime value) {
    if (this.updatedAt == value) return this;
    OffsetDateTime newValue = Objects.requireNonNull(value, "updatedAt");
    return new AOV3Release(
        this.binaries,
        this.downloadCount,
        this.id,
        this.releaseLink,
        this.releaseName,
        this.releaseType,
        this.source,
        this.timestamp,
        newValue,
        this.vendor,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ReleaseType#vendor() vendor} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for vendor
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Release withVendor(AOV3Vendor value) {
    if (this.vendor == value) return this;
    AOV3Vendor newValue = Objects.requireNonNull(value, "vendor");
    if (this.vendor.equals(newValue)) return this;
    return new AOV3Release(
        this.binaries,
        this.downloadCount,
        this.id,
        this.releaseLink,
        this.releaseName,
        this.releaseType,
        this.source,
        this.timestamp,
        this.updatedAt,
        newValue,
        this.versionData);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ReleaseType#versionData() versionData} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for versionData
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Release withVersionData(AOV3VersionData value) {
    if (this.versionData == value) return this;
    AOV3VersionData newValue = Objects.requireNonNull(value, "versionData");
    return new AOV3Release(
        this.binaries,
        this.downloadCount,
        this.id,
        this.releaseLink,
        this.releaseName,
        this.releaseType,
        this.source,
        this.timestamp,
        this.updatedAt,
        this.vendor,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code AOV3Release} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof AOV3Release
        && equalTo((AOV3Release) another);
  }

  private boolean equalTo(AOV3Release another) {
    return binaries.equals(another.binaries)
        && downloadCount.equals(another.downloadCount)
        && id.equals(another.id)
        && releaseLink.equals(another.releaseLink)
        && releaseName.equals(another.releaseName)
        && releaseType.equals(another.releaseType)
        && Objects.equals(source, another.source)
        && timestamp.equals(another.timestamp)
        && updatedAt.equals(another.updatedAt)
        && vendor.equals(another.vendor)
        && versionData.equals(another.versionData);
  }

  /**
   * Computes a hash code from attributes: {@code binaries}, {@code downloadCount}, {@code id}, {@code releaseLink}, {@code releaseName}, {@code releaseType}, {@code source}, {@code timestamp}, {@code updatedAt}, {@code vendor}, {@code versionData}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + binaries.hashCode();
    h += (h << 5) + downloadCount.hashCode();
    h += (h << 5) + id.hashCode();
    h += (h << 5) + releaseLink.hashCode();
    h += (h << 5) + releaseName.hashCode();
    h += (h << 5) + releaseType.hashCode();
    h += (h << 5) + Objects.hashCode(source);
    h += (h << 5) + timestamp.hashCode();
    h += (h << 5) + updatedAt.hashCode();
    h += (h << 5) + vendor.hashCode();
    h += (h << 5) + versionData.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code AOV3Release} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("AOV3Release{");
    builder.append("binaries=").append(binaries);
    builder.append(", ");
    builder.append("downloadCount=").append(downloadCount);
    builder.append(", ");
    builder.append("id=").append(id);
    builder.append(", ");
    builder.append("releaseLink=").append(releaseLink);
    builder.append(", ");
    builder.append("releaseName=").append(releaseName);
    builder.append(", ");
    builder.append("releaseType=").append(releaseType);
    if (source != null) {
      builder.append(", ");
      builder.append("source=").append(source);
    }
    builder.append(", ");
    builder.append("timestamp=").append(timestamp);
    builder.append(", ");
    builder.append("updatedAt=").append(updatedAt);
    builder.append(", ");
    builder.append("vendor=").append(vendor);
    builder.append(", ");
    builder.append("versionData=").append(versionData);
    return builder.append("}").toString();
  }

  /**
   * Creates an immutable copy of a {@link AOV3ReleaseType} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable AOV3Release instance
   */
  public static AOV3Release copyOf(AOV3ReleaseType instance) {
    if (instance instanceof AOV3Release) {
      return (AOV3Release) instance;
    }
    return AOV3Release.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link AOV3Release AOV3Release}.
   * <pre>
   * AOV3Release.builder()
   *    .addBinaries|addAllBinaries(net.adoptopenjdk.v3.api.AOV3BinaryType) // {@link AOV3ReleaseType#binaries() binaries} elements
   *    .setDownloadCount(java.math.BigInteger) // required {@link AOV3ReleaseType#downloadCount() downloadCount}
   *    .setId(String) // required {@link AOV3ReleaseType#id() id}
   *    .setReleaseLink(java.net.URI) // required {@link AOV3ReleaseType#releaseLink() releaseLink}
   *    .setReleaseName(String) // required {@link AOV3ReleaseType#releaseName() releaseName}
   *    .setReleaseType(net.adoptopenjdk.v3.api.AOV3ReleaseKind) // required {@link AOV3ReleaseType#releaseType() releaseType}
   *    .setSource(AOV3Source) // optional {@link AOV3ReleaseType#source() source}
   *    .setTimestamp(java.time.OffsetDateTime) // required {@link AOV3ReleaseType#timestamp() timestamp}
   *    .setUpdatedAt(java.time.OffsetDateTime) // required {@link AOV3ReleaseType#updatedAt() updatedAt}
   *    .setVendor(net.adoptopenjdk.v3.api.AOV3Vendor) // required {@link AOV3ReleaseType#vendor() vendor}
   *    .setVersionData(net.adoptopenjdk.v3.api.AOV3VersionData) // required {@link AOV3ReleaseType#versionData() versionData}
   *    .build();
   * </pre>
   * @return A new AOV3Release builder
   */
  public static AOV3Release.Builder builder() {
    return new AOV3Release.Builder();
  }

  /**
   * Builds instances of type {@link AOV3Release AOV3Release}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_DOWNLOAD_COUNT = 0x1L;
    private static final long INIT_BIT_ID = 0x2L;
    private static final long INIT_BIT_RELEASE_LINK = 0x4L;
    private static final long INIT_BIT_RELEASE_NAME = 0x8L;
    private static final long INIT_BIT_RELEASE_TYPE = 0x10L;
    private static final long INIT_BIT_TIMESTAMP = 0x20L;
    private static final long INIT_BIT_UPDATED_AT = 0x40L;
    private static final long INIT_BIT_VENDOR = 0x80L;
    private static final long INIT_BIT_VERSION_DATA = 0x100L;
    private long initBits = 0x1ffL;

    private List<AOV3BinaryType> binaries = new ArrayList<AOV3BinaryType>();
    private BigInteger downloadCount;
    private String id;
    private URI releaseLink;
    private String releaseName;
    private AOV3ReleaseKind releaseType;
    private AOV3Source source;
    private OffsetDateTime timestamp;
    private OffsetDateTime updatedAt;
    private AOV3Vendor vendor;
    private AOV3VersionData versionData;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code AOV3ReleaseType} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(AOV3ReleaseType instance) {
      Objects.requireNonNull(instance, "instance");
      addAllBinaries(instance.binaries());
      setDownloadCount(instance.downloadCount());
      setId(instance.id());
      setReleaseLink(instance.releaseLink());
      setReleaseName(instance.releaseName());
      setReleaseType(instance.releaseType());
      Optional<AOV3Source> sourceOptional = instance.source();
      if (sourceOptional.isPresent()) {
        setSource(sourceOptional);
      }
      setTimestamp(instance.timestamp());
      setUpdatedAt(instance.updatedAt());
      setVendor(instance.vendor());
      setVersionData(instance.versionData());
      return this;
    }

    /**
     * Adds one element to {@link AOV3ReleaseType#binaries() binaries} list.
     * @param element A binaries element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addBinaries(AOV3BinaryType element) {
      this.binaries.add(Objects.requireNonNull(element, "binaries element"));
      return this;
    }

    /**
     * Adds elements to {@link AOV3ReleaseType#binaries() binaries} list.
     * @param elements An array of binaries elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addBinaries(AOV3BinaryType... elements) {
      for (AOV3BinaryType element : elements) {
        this.binaries.add(Objects.requireNonNull(element, "binaries element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link AOV3ReleaseType#binaries() binaries} list.
     * @param elements An iterable of binaries elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setBinaries(Iterable<? extends AOV3BinaryType> elements) {
      this.binaries.clear();
      return addAllBinaries(elements);
    }

    /**
     * Adds elements to {@link AOV3ReleaseType#binaries() binaries} list.
     * @param elements An iterable of binaries elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllBinaries(Iterable<? extends AOV3BinaryType> elements) {
      for (AOV3BinaryType element : elements) {
        this.binaries.add(Objects.requireNonNull(element, "binaries element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ReleaseType#downloadCount() downloadCount} attribute.
     * @param downloadCount The value for downloadCount 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setDownloadCount(BigInteger downloadCount) {
      this.downloadCount = Objects.requireNonNull(downloadCount, "downloadCount");
      initBits &= ~INIT_BIT_DOWNLOAD_COUNT;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ReleaseType#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setId(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ReleaseType#releaseLink() releaseLink} attribute.
     * @param releaseLink The value for releaseLink 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setReleaseLink(URI releaseLink) {
      this.releaseLink = Objects.requireNonNull(releaseLink, "releaseLink");
      initBits &= ~INIT_BIT_RELEASE_LINK;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ReleaseType#releaseName() releaseName} attribute.
     * @param releaseName The value for releaseName 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setReleaseName(String releaseName) {
      this.releaseName = Objects.requireNonNull(releaseName, "releaseName");
      initBits &= ~INIT_BIT_RELEASE_NAME;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ReleaseType#releaseType() releaseType} attribute.
     * @param releaseType The value for releaseType 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setReleaseType(AOV3ReleaseKind releaseType) {
      this.releaseType = Objects.requireNonNull(releaseType, "releaseType");
      initBits &= ~INIT_BIT_RELEASE_TYPE;
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3ReleaseType#source() source} to source.
     * @param source The value for source
     * @return {@code this} builder for chained invocation
     */
    public final Builder setSource(AOV3Source source) {
      this.source = Objects.requireNonNull(source, "source");
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3ReleaseType#source() source} to source.
     * @param source The value for source
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setSource(Optional<? extends AOV3Source> source) {
      this.source = source.orElse(null);
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ReleaseType#timestamp() timestamp} attribute.
     * @param timestamp The value for timestamp 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setTimestamp(OffsetDateTime timestamp) {
      this.timestamp = Objects.requireNonNull(timestamp, "timestamp");
      initBits &= ~INIT_BIT_TIMESTAMP;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ReleaseType#updatedAt() updatedAt} attribute.
     * @param updatedAt The value for updatedAt 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setUpdatedAt(OffsetDateTime updatedAt) {
      this.updatedAt = Objects.requireNonNull(updatedAt, "updatedAt");
      initBits &= ~INIT_BIT_UPDATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ReleaseType#vendor() vendor} attribute.
     * @param vendor The value for vendor 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setVendor(AOV3Vendor vendor) {
      this.vendor = Objects.requireNonNull(vendor, "vendor");
      initBits &= ~INIT_BIT_VENDOR;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ReleaseType#versionData() versionData} attribute.
     * @param versionData The value for versionData 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setVersionData(AOV3VersionData versionData) {
      this.versionData = Objects.requireNonNull(versionData, "versionData");
      initBits &= ~INIT_BIT_VERSION_DATA;
      return this;
    }

    /**
     * Builds a new {@link AOV3Release AOV3Release}.
     * @return An immutable instance of AOV3Release
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public AOV3Release build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new AOV3Release(
          createUnmodifiableList(true, binaries),
          downloadCount,
          id,
          releaseLink,
          releaseName,
          releaseType,
          source,
          timestamp,
          updatedAt,
          vendor,
          versionData);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DOWNLOAD_COUNT) != 0) attributes.add("downloadCount");
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      if ((initBits & INIT_BIT_RELEASE_LINK) != 0) attributes.add("releaseLink");
      if ((initBits & INIT_BIT_RELEASE_NAME) != 0) attributes.add("releaseName");
      if ((initBits & INIT_BIT_RELEASE_TYPE) != 0) attributes.add("releaseType");
      if ((initBits & INIT_BIT_TIMESTAMP) != 0) attributes.add("timestamp");
      if ((initBits & INIT_BIT_UPDATED_AT) != 0) attributes.add("updatedAt");
      if ((initBits & INIT_BIT_VENDOR) != 0) attributes.add("vendor");
      if ((initBits & INIT_BIT_VERSION_DATA) != 0) attributes.add("versionData");
      return "Cannot build AOV3Release, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>();
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
