package net.adoptopenjdk.v3.api;

import java.math.BigInteger;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Installer information.
 */
@SuppressWarnings({"all"})
public final class AOV3Installer implements AOV3InstallerType {
  private final String checksum;
  private final URI checksumLink;
  private final BigInteger downloadCount;
  private final URI link;
  private final String name;
  private final URI signatureLink;
  private final BigInteger size;

  private AOV3Installer(
      String checksum,
      URI checksumLink,
      BigInteger downloadCount,
      URI link,
      String name,
      URI signatureLink,
      BigInteger size) {
    this.checksum = checksum;
    this.checksumLink = checksumLink;
    this.downloadCount = downloadCount;
    this.link = link;
    this.name = name;
    this.signatureLink = signatureLink;
    this.size = size;
  }

  /**
   * @return The checksum of the data
   */
  @Override
  public Optional<String> checksum() {
    return Optional.ofNullable(checksum);
  }

  /**
   * @return A link to the checksum of the data
   */
  @Override
  public Optional<URI> checksumLink() {
    return Optional.ofNullable(checksumLink);
  }

  /**
   * @return The number of times the package has been downloaded
   */
  @Override
  public BigInteger downloadCount() {
    return downloadCount;
  }

  /**
   * @return The link to the data
   */
  @Override
  public URI link() {
    return link;
  }

  /**
   * @return The package name
   */
  @Override
  public String name() {
    return name;
  }

  /**
   * @return The link to the package signature
   */
  @Override
  public Optional<URI> signatureLink() {
    return Optional.ofNullable(signatureLink);
  }

  /**
   * @return The package size
   */
  @Override
  public BigInteger size() {
    return size;
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link AOV3InstallerType#checksum() checksum} attribute.
   * @param value The value for checksum
   * @return A modified copy of {@code this} object
   */
  public final AOV3Installer withChecksum(String value) {
    String newValue = Objects.requireNonNull(value, "checksum");
    if (Objects.equals(this.checksum, newValue)) return this;
    return new AOV3Installer(
        newValue,
        this.checksumLink,
        this.downloadCount,
        this.link,
        this.name,
        this.signatureLink,
        this.size);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link AOV3InstallerType#checksum() checksum} attribute.
   * An equality check is used on inner nullable value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for checksum
   * @return A modified copy of {@code this} object
   */
  public final AOV3Installer withChecksum(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.checksum, value)) return this;
    return new AOV3Installer(
        value,
        this.checksumLink,
        this.downloadCount,
        this.link,
        this.name,
        this.signatureLink,
        this.size);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link AOV3InstallerType#checksumLink() checksumLink} attribute.
   * @param value The value for checksumLink
   * @return A modified copy of {@code this} object
   */
  public final AOV3Installer withChecksumLink(URI value) {
    URI newValue = Objects.requireNonNull(value, "checksumLink");
    if (this.checksumLink == newValue) return this;
    return new AOV3Installer(
        this.checksum,
        newValue,
        this.downloadCount,
        this.link,
        this.name,
        this.signatureLink,
        this.size);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link AOV3InstallerType#checksumLink() checksumLink} attribute.
   * A shallow reference equality check is used on unboxed optional value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for checksumLink
   * @return A modified copy of {@code this} object
   */
  @SuppressWarnings("unchecked") // safe covariant cast
  public final AOV3Installer withChecksumLink(Optional<? extends URI> optional) {
    URI value = optional.orElse(null);
    if (this.checksumLink == value) return this;
    return new AOV3Installer(this.checksum, value, this.downloadCount, this.link, this.name, this.signatureLink, this.size);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3InstallerType#downloadCount() downloadCount} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for downloadCount
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Installer withDownloadCount(BigInteger value) {
    BigInteger newValue = Objects.requireNonNull(value, "downloadCount");
    if (this.downloadCount.equals(newValue)) return this;
    return new AOV3Installer(this.checksum, this.checksumLink, newValue, this.link, this.name, this.signatureLink, this.size);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3InstallerType#link() link} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for link
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Installer withLink(URI value) {
    if (this.link == value) return this;
    URI newValue = Objects.requireNonNull(value, "link");
    return new AOV3Installer(
        this.checksum,
        this.checksumLink,
        this.downloadCount,
        newValue,
        this.name,
        this.signatureLink,
        this.size);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3InstallerType#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Installer withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (this.name.equals(newValue)) return this;
    return new AOV3Installer(
        this.checksum,
        this.checksumLink,
        this.downloadCount,
        this.link,
        newValue,
        this.signatureLink,
        this.size);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link AOV3InstallerType#signatureLink() signatureLink} attribute.
   * @param value The value for signatureLink
   * @return A modified copy of {@code this} object
   */
  public final AOV3Installer withSignatureLink(URI value) {
    URI newValue = Objects.requireNonNull(value, "signatureLink");
    if (this.signatureLink == newValue) return this;
    return new AOV3Installer(this.checksum, this.checksumLink, this.downloadCount, this.link, this.name, newValue, this.size);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link AOV3InstallerType#signatureLink() signatureLink} attribute.
   * A shallow reference equality check is used on unboxed optional value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for signatureLink
   * @return A modified copy of {@code this} object
   */
  @SuppressWarnings("unchecked") // safe covariant cast
  public final AOV3Installer withSignatureLink(Optional<? extends URI> optional) {
    URI value = optional.orElse(null);
    if (this.signatureLink == value) return this;
    return new AOV3Installer(this.checksum, this.checksumLink, this.downloadCount, this.link, this.name, value, this.size);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3InstallerType#size() size} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for size
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Installer withSize(BigInteger value) {
    BigInteger newValue = Objects.requireNonNull(value, "size");
    if (this.size.equals(newValue)) return this;
    return new AOV3Installer(
        this.checksum,
        this.checksumLink,
        this.downloadCount,
        this.link,
        this.name,
        this.signatureLink,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code AOV3Installer} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof AOV3Installer
        && equalTo((AOV3Installer) another);
  }

  private boolean equalTo(AOV3Installer another) {
    return Objects.equals(checksum, another.checksum)
        && Objects.equals(checksumLink, another.checksumLink)
        && downloadCount.equals(another.downloadCount)
        && link.equals(another.link)
        && name.equals(another.name)
        && Objects.equals(signatureLink, another.signatureLink)
        && size.equals(another.size);
  }

  /**
   * Computes a hash code from attributes: {@code checksum}, {@code checksumLink}, {@code downloadCount}, {@code link}, {@code name}, {@code signatureLink}, {@code size}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(checksum);
    h += (h << 5) + Objects.hashCode(checksumLink);
    h += (h << 5) + downloadCount.hashCode();
    h += (h << 5) + link.hashCode();
    h += (h << 5) + name.hashCode();
    h += (h << 5) + Objects.hashCode(signatureLink);
    h += (h << 5) + size.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code AOV3Installer} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("AOV3Installer{");
    if (checksum != null) {
      builder.append("checksum=").append(checksum);
    }
    if (checksumLink != null) {
      if (builder.length() > 14) builder.append(", ");
      builder.append("checksumLink=").append(checksumLink);
    }
    if (builder.length() > 14) builder.append(", ");
    builder.append("downloadCount=").append(downloadCount);
    builder.append(", ");
    builder.append("link=").append(link);
    builder.append(", ");
    builder.append("name=").append(name);
    if (signatureLink != null) {
      builder.append(", ");
      builder.append("signatureLink=").append(signatureLink);
    }
    builder.append(", ");
    builder.append("size=").append(size);
    return builder.append("}").toString();
  }

  /**
   * Creates an immutable copy of a {@link AOV3InstallerType} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable AOV3Installer instance
   */
  public static AOV3Installer copyOf(AOV3InstallerType instance) {
    if (instance instanceof AOV3Installer) {
      return (AOV3Installer) instance;
    }
    return AOV3Installer.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link AOV3Installer AOV3Installer}.
   * <pre>
   * AOV3Installer.builder()
   *    .setChecksum(String) // optional {@link AOV3InstallerType#checksum() checksum}
   *    .setChecksumLink(java.net.URI) // optional {@link AOV3InstallerType#checksumLink() checksumLink}
   *    .setDownloadCount(java.math.BigInteger) // required {@link AOV3InstallerType#downloadCount() downloadCount}
   *    .setLink(java.net.URI) // required {@link AOV3InstallerType#link() link}
   *    .setName(String) // required {@link AOV3InstallerType#name() name}
   *    .setSignatureLink(java.net.URI) // optional {@link AOV3InstallerType#signatureLink() signatureLink}
   *    .setSize(java.math.BigInteger) // required {@link AOV3InstallerType#size() size}
   *    .build();
   * </pre>
   * @return A new AOV3Installer builder
   */
  public static AOV3Installer.Builder builder() {
    return new AOV3Installer.Builder();
  }

  /**
   * Builds instances of type {@link AOV3Installer AOV3Installer}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_DOWNLOAD_COUNT = 0x1L;
    private static final long INIT_BIT_LINK = 0x2L;
    private static final long INIT_BIT_NAME = 0x4L;
    private static final long INIT_BIT_SIZE = 0x8L;
    private long initBits = 0xfL;

    private String checksum;
    private URI checksumLink;
    private BigInteger downloadCount;
    private URI link;
    private String name;
    private URI signatureLink;
    private BigInteger size;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code AOV3InstallerType} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(AOV3InstallerType instance) {
      Objects.requireNonNull(instance, "instance");
      Optional<String> checksumOptional = instance.checksum();
      if (checksumOptional.isPresent()) {
        setChecksum(checksumOptional);
      }
      Optional<URI> checksumLinkOptional = instance.checksumLink();
      if (checksumLinkOptional.isPresent()) {
        setChecksumLink(checksumLinkOptional);
      }
      setDownloadCount(instance.downloadCount());
      setLink(instance.link());
      setName(instance.name());
      Optional<URI> signatureLinkOptional = instance.signatureLink();
      if (signatureLinkOptional.isPresent()) {
        setSignatureLink(signatureLinkOptional);
      }
      setSize(instance.size());
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3InstallerType#checksum() checksum} to checksum.
     * @param checksum The value for checksum
     * @return {@code this} builder for chained invocation
     */
    public final Builder setChecksum(String checksum) {
      this.checksum = Objects.requireNonNull(checksum, "checksum");
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3InstallerType#checksum() checksum} to checksum.
     * @param checksum The value for checksum
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setChecksum(Optional<String> checksum) {
      this.checksum = checksum.orElse(null);
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3InstallerType#checksumLink() checksumLink} to checksumLink.
     * @param checksumLink The value for checksumLink
     * @return {@code this} builder for chained invocation
     */
    public final Builder setChecksumLink(URI checksumLink) {
      this.checksumLink = Objects.requireNonNull(checksumLink, "checksumLink");
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3InstallerType#checksumLink() checksumLink} to checksumLink.
     * @param checksumLink The value for checksumLink
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setChecksumLink(Optional<? extends URI> checksumLink) {
      this.checksumLink = checksumLink.orElse(null);
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3InstallerType#downloadCount() downloadCount} attribute.
     * @param downloadCount The value for downloadCount 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setDownloadCount(BigInteger downloadCount) {
      this.downloadCount = Objects.requireNonNull(downloadCount, "downloadCount");
      initBits &= ~INIT_BIT_DOWNLOAD_COUNT;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3InstallerType#link() link} attribute.
     * @param link The value for link 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setLink(URI link) {
      this.link = Objects.requireNonNull(link, "link");
      initBits &= ~INIT_BIT_LINK;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3InstallerType#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setName(String name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3InstallerType#signatureLink() signatureLink} to signatureLink.
     * @param signatureLink The value for signatureLink
     * @return {@code this} builder for chained invocation
     */
    public final Builder setSignatureLink(URI signatureLink) {
      this.signatureLink = Objects.requireNonNull(signatureLink, "signatureLink");
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3InstallerType#signatureLink() signatureLink} to signatureLink.
     * @param signatureLink The value for signatureLink
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setSignatureLink(Optional<? extends URI> signatureLink) {
      this.signatureLink = signatureLink.orElse(null);
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3InstallerType#size() size} attribute.
     * @param size The value for size 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setSize(BigInteger size) {
      this.size = Objects.requireNonNull(size, "size");
      initBits &= ~INIT_BIT_SIZE;
      return this;
    }

    /**
     * Builds a new {@link AOV3Installer AOV3Installer}.
     * @return An immutable instance of AOV3Installer
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public AOV3Installer build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new AOV3Installer(checksum, checksumLink, downloadCount, link, name, signatureLink, size);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DOWNLOAD_COUNT) != 0) attributes.add("downloadCount");
      if ((initBits & INIT_BIT_LINK) != 0) attributes.add("link");
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      if ((initBits & INIT_BIT_SIZE) != 0) attributes.add("size");
      return "Cannot build AOV3Installer, some of required attributes are not set " + attributes;
    }
  }
}
