package net.adoptopenjdk.v3.api;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * A version range consisting of two bounds.
 */
@SuppressWarnings({"all"})
public final class AOV3VersionRange implements AOV3VersionRangeType {
  private final AOV3VersionBound lower;
  private final AOV3VersionBound upper;

  private AOV3VersionRange(AOV3VersionBound lower, AOV3VersionBound upper) {
    this.lower = Objects.requireNonNull(lower, "lower");
    this.upper = Objects.requireNonNull(upper, "upper");
  }

  private AOV3VersionRange(
      AOV3VersionRange original,
      AOV3VersionBound lower,
      AOV3VersionBound upper) {
    this.lower = lower;
    this.upper = upper;
  }

  /**
   * @return The lower bound
   */
  @Override
  public AOV3VersionBound lower() {
    return lower;
  }

  /**
   * @return The upper bound
   */
  @Override
  public AOV3VersionBound upper() {
    return upper;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3VersionRangeType#lower() lower} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for lower
   * @return A modified copy of the {@code this} object
   */
  public final AOV3VersionRange withLower(AOV3VersionBound value) {
    if (this.lower == value) return this;
    AOV3VersionBound newValue = Objects.requireNonNull(value, "lower");
    return new AOV3VersionRange(this, newValue, this.upper);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3VersionRangeType#upper() upper} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for upper
   * @return A modified copy of the {@code this} object
   */
  public final AOV3VersionRange withUpper(AOV3VersionBound value) {
    if (this.upper == value) return this;
    AOV3VersionBound newValue = Objects.requireNonNull(value, "upper");
    return new AOV3VersionRange(this, this.lower, newValue);
  }

  /**
   * This instance is equal to all instances of {@code AOV3VersionRange} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof AOV3VersionRange
        && equalTo((AOV3VersionRange) another);
  }

  private boolean equalTo(AOV3VersionRange another) {
    return lower.equals(another.lower)
        && upper.equals(another.upper);
  }

  /**
   * Computes a hash code from attributes: {@code lower}, {@code upper}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + lower.hashCode();
    h += (h << 5) + upper.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code AOV3VersionRange} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "AOV3VersionRange{"
        + "lower=" + lower
        + ", upper=" + upper
        + "}";
  }

  /**
   * Construct a new immutable {@code AOV3VersionRange} instance.
   * @param lower The value for the {@code lower} attribute
   * @param upper The value for the {@code upper} attribute
   * @return An immutable AOV3VersionRange instance
   */
  public static AOV3VersionRange of(AOV3VersionBound lower, AOV3VersionBound upper) {
    return new AOV3VersionRange(lower, upper);
  }

  /**
   * Creates an immutable copy of a {@link AOV3VersionRangeType} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable AOV3VersionRange instance
   */
  public static AOV3VersionRange copyOf(AOV3VersionRangeType instance) {
    if (instance instanceof AOV3VersionRange) {
      return (AOV3VersionRange) instance;
    }
    return AOV3VersionRange.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link AOV3VersionRange AOV3VersionRange}.
   * <pre>
   * AOV3VersionRange.builder()
   *    .setLower(net.adoptopenjdk.v3.api.AOV3VersionBound) // required {@link AOV3VersionRangeType#lower() lower}
   *    .setUpper(net.adoptopenjdk.v3.api.AOV3VersionBound) // required {@link AOV3VersionRangeType#upper() upper}
   *    .build();
   * </pre>
   * @return A new AOV3VersionRange builder
   */
  public static AOV3VersionRange.Builder builder() {
    return new AOV3VersionRange.Builder();
  }

  /**
   * Builds instances of type {@link AOV3VersionRange AOV3VersionRange}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_LOWER = 0x1L;
    private static final long INIT_BIT_UPPER = 0x2L;
    private long initBits = 0x3L;

    private AOV3VersionBound lower;
    private AOV3VersionBound upper;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code AOV3VersionRangeType} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(AOV3VersionRangeType instance) {
      Objects.requireNonNull(instance, "instance");
      setLower(instance.lower());
      setUpper(instance.upper());
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3VersionRangeType#lower() lower} attribute.
     * @param lower The value for lower 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setLower(AOV3VersionBound lower) {
      this.lower = Objects.requireNonNull(lower, "lower");
      initBits &= ~INIT_BIT_LOWER;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3VersionRangeType#upper() upper} attribute.
     * @param upper The value for upper 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setUpper(AOV3VersionBound upper) {
      this.upper = Objects.requireNonNull(upper, "upper");
      initBits &= ~INIT_BIT_UPPER;
      return this;
    }

    /**
     * Builds a new {@link AOV3VersionRange AOV3VersionRange}.
     * @return An immutable instance of AOV3VersionRange
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public AOV3VersionRange build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new AOV3VersionRange(null, lower, upper);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_LOWER) != 0) attributes.add("lower");
      if ((initBits & INIT_BIT_UPPER) != 0) attributes.add("upper");
      return "Cannot build AOV3VersionRange, some of required attributes are not set " + attributes;
    }
  }
}
