package net.adoptopenjdk.v3.api;

import java.math.BigInteger;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Information about the source that produced a build.
 */
@SuppressWarnings({"all"})
public final class AOV3Source implements AOV3SourceType {
  private final URI link;
  private final String name;
  private final BigInteger size;

  private AOV3Source(URI link, String name, BigInteger size) {
    this.link = link;
    this.name = name;
    this.size = size;
  }

  /**
   * @return The source distribution
   */
  @Override
  public URI link() {
    return link;
  }

  /**
   * @return The source distribution name
   */
  @Override
  public String name() {
    return name;
  }

  /**
   * @return The source distribution size
   */
  @Override
  public BigInteger size() {
    return size;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3SourceType#link() link} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for link
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Source withLink(URI value) {
    if (this.link == value) return this;
    URI newValue = Objects.requireNonNull(value, "link");
    return new AOV3Source(newValue, this.name, this.size);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3SourceType#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Source withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (this.name.equals(newValue)) return this;
    return new AOV3Source(this.link, newValue, this.size);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3SourceType#size() size} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for size
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Source withSize(BigInteger value) {
    BigInteger newValue = Objects.requireNonNull(value, "size");
    if (this.size.equals(newValue)) return this;
    return new AOV3Source(this.link, this.name, newValue);
  }

  /**
   * This instance is equal to all instances of {@code AOV3Source} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof AOV3Source
        && equalTo((AOV3Source) another);
  }

  private boolean equalTo(AOV3Source another) {
    return link.equals(another.link)
        && name.equals(another.name)
        && size.equals(another.size);
  }

  /**
   * Computes a hash code from attributes: {@code link}, {@code name}, {@code size}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + link.hashCode();
    h += (h << 5) + name.hashCode();
    h += (h << 5) + size.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code AOV3Source} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "AOV3Source{"
        + "link=" + link
        + ", name=" + name
        + ", size=" + size
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link AOV3SourceType} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable AOV3Source instance
   */
  public static AOV3Source copyOf(AOV3SourceType instance) {
    if (instance instanceof AOV3Source) {
      return (AOV3Source) instance;
    }
    return AOV3Source.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link AOV3Source AOV3Source}.
   * <pre>
   * AOV3Source.builder()
   *    .setLink(java.net.URI) // required {@link AOV3SourceType#link() link}
   *    .setName(String) // required {@link AOV3SourceType#name() name}
   *    .setSize(java.math.BigInteger) // required {@link AOV3SourceType#size() size}
   *    .build();
   * </pre>
   * @return A new AOV3Source builder
   */
  public static AOV3Source.Builder builder() {
    return new AOV3Source.Builder();
  }

  /**
   * Builds instances of type {@link AOV3Source AOV3Source}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_LINK = 0x1L;
    private static final long INIT_BIT_NAME = 0x2L;
    private static final long INIT_BIT_SIZE = 0x4L;
    private long initBits = 0x7L;

    private URI link;
    private String name;
    private BigInteger size;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code AOV3SourceType} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(AOV3SourceType instance) {
      Objects.requireNonNull(instance, "instance");
      setLink(instance.link());
      setName(instance.name());
      setSize(instance.size());
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3SourceType#link() link} attribute.
     * @param link The value for link 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setLink(URI link) {
      this.link = Objects.requireNonNull(link, "link");
      initBits &= ~INIT_BIT_LINK;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3SourceType#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setName(String name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3SourceType#size() size} attribute.
     * @param size The value for size 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setSize(BigInteger size) {
      this.size = Objects.requireNonNull(size, "size");
      initBits &= ~INIT_BIT_SIZE;
      return this;
    }

    /**
     * Builds a new {@link AOV3Source AOV3Source}.
     * @return An immutable instance of AOV3Source
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public AOV3Source build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new AOV3Source(link, name, size);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_LINK) != 0) attributes.add("link");
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      if ((initBits & INIT_BIT_SIZE) != 0) attributes.add("size");
      return "Cannot build AOV3Source, some of required attributes are not set " + attributes;
    }
  }
}
