package net.adoptopenjdk.v3.api;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * An error that occurred during an API call.
 */
@SuppressWarnings({"all"})
public final class AOV3Error implements AOV3ErrorType {
  private final URI source;
  private final String context;
  private final String message;
  private final Exception exception;

  private AOV3Error(
      URI source,
      String context,
      String message,
      Exception exception) {
    this.source = source;
    this.context = context;
    this.message = message;
    this.exception = exception;
  }

  /**
   * @return The source of the error
   */
  @Override
  public URI source() {
    return source;
  }

  /**
   * @return The context of the error
   */
  @Override
  public String context() {
    return context;
  }

  /**
   * @return The error message
   */
  @Override
  public String message() {
    return message;
  }

  /**
   * @return The exception raised, if any
   */
  @Override
  public Optional<Exception> exception() {
    return Optional.ofNullable(exception);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ErrorType#source() source} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for source
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Error withSource(URI value) {
    if (this.source == value) return this;
    URI newValue = Objects.requireNonNull(value, "source");
    return new AOV3Error(newValue, this.context, this.message, this.exception);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ErrorType#context() context} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for context
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Error withContext(String value) {
    String newValue = Objects.requireNonNull(value, "context");
    if (this.context.equals(newValue)) return this;
    return new AOV3Error(this.source, newValue, this.message, this.exception);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3ErrorType#message() message} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for message
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Error withMessage(String value) {
    String newValue = Objects.requireNonNull(value, "message");
    if (this.message.equals(newValue)) return this;
    return new AOV3Error(this.source, this.context, newValue, this.exception);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link AOV3ErrorType#exception() exception} attribute.
   * @param value The value for exception
   * @return A modified copy of {@code this} object
   */
  public final AOV3Error withException(Exception value) {
    Exception newValue = Objects.requireNonNull(value, "exception");
    if (this.exception == newValue) return this;
    return new AOV3Error(this.source, this.context, this.message, newValue);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link AOV3ErrorType#exception() exception} attribute.
   * A shallow reference equality check is used on unboxed optional value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for exception
   * @return A modified copy of {@code this} object
   */
  @SuppressWarnings("unchecked") // safe covariant cast
  public final AOV3Error withException(Optional<? extends Exception> optional) {
    Exception value = optional.orElse(null);
    if (this.exception == value) return this;
    return new AOV3Error(this.source, this.context, this.message, value);
  }

  /**
   * This instance is equal to all instances of {@code AOV3Error} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof AOV3Error
        && equalTo((AOV3Error) another);
  }

  private boolean equalTo(AOV3Error another) {
    return source.equals(another.source)
        && context.equals(another.context)
        && message.equals(another.message)
        && Objects.equals(exception, another.exception);
  }

  /**
   * Computes a hash code from attributes: {@code source}, {@code context}, {@code message}, {@code exception}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + source.hashCode();
    h += (h << 5) + context.hashCode();
    h += (h << 5) + message.hashCode();
    h += (h << 5) + Objects.hashCode(exception);
    return h;
  }

  /**
   * Prints the immutable value {@code AOV3Error} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("AOV3Error{");
    builder.append("source=").append(source);
    builder.append(", ");
    builder.append("context=").append(context);
    builder.append(", ");
    builder.append("message=").append(message);
    if (exception != null) {
      builder.append(", ");
      builder.append("exception=").append(exception);
    }
    return builder.append("}").toString();
  }

  /**
   * Creates an immutable copy of a {@link AOV3ErrorType} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable AOV3Error instance
   */
  public static AOV3Error copyOf(AOV3ErrorType instance) {
    if (instance instanceof AOV3Error) {
      return (AOV3Error) instance;
    }
    return AOV3Error.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link AOV3Error AOV3Error}.
   * <pre>
   * AOV3Error.builder()
   *    .setSource(java.net.URI) // required {@link AOV3ErrorType#source() source}
   *    .setContext(String) // required {@link AOV3ErrorType#context() context}
   *    .setMessage(String) // required {@link AOV3ErrorType#message() message}
   *    .setException(Exception) // optional {@link AOV3ErrorType#exception() exception}
   *    .build();
   * </pre>
   * @return A new AOV3Error builder
   */
  public static AOV3Error.Builder builder() {
    return new AOV3Error.Builder();
  }

  /**
   * Builds instances of type {@link AOV3Error AOV3Error}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_SOURCE = 0x1L;
    private static final long INIT_BIT_CONTEXT = 0x2L;
    private static final long INIT_BIT_MESSAGE = 0x4L;
    private long initBits = 0x7L;

    private URI source;
    private String context;
    private String message;
    private Exception exception;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code AOV3ErrorType} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(AOV3ErrorType instance) {
      Objects.requireNonNull(instance, "instance");
      setSource(instance.source());
      setContext(instance.context());
      setMessage(instance.message());
      Optional<Exception> exceptionOptional = instance.exception();
      if (exceptionOptional.isPresent()) {
        setException(exceptionOptional);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ErrorType#source() source} attribute.
     * @param source The value for source 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setSource(URI source) {
      this.source = Objects.requireNonNull(source, "source");
      initBits &= ~INIT_BIT_SOURCE;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ErrorType#context() context} attribute.
     * @param context The value for context 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setContext(String context) {
      this.context = Objects.requireNonNull(context, "context");
      initBits &= ~INIT_BIT_CONTEXT;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3ErrorType#message() message} attribute.
     * @param message The value for message 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setMessage(String message) {
      this.message = Objects.requireNonNull(message, "message");
      initBits &= ~INIT_BIT_MESSAGE;
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3ErrorType#exception() exception} to exception.
     * @param exception The value for exception
     * @return {@code this} builder for chained invocation
     */
    public final Builder setException(Exception exception) {
      this.exception = Objects.requireNonNull(exception, "exception");
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3ErrorType#exception() exception} to exception.
     * @param exception The value for exception
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setException(Optional<? extends Exception> exception) {
      this.exception = exception.orElse(null);
      return this;
    }

    /**
     * Builds a new {@link AOV3Error AOV3Error}.
     * @return An immutable instance of AOV3Error
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public AOV3Error build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new AOV3Error(source, context, message, exception);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_SOURCE) != 0) attributes.add("source");
      if ((initBits & INIT_BIT_CONTEXT) != 0) attributes.add("context");
      if ((initBits & INIT_BIT_MESSAGE) != 0) attributes.add("message");
      return "Cannot build AOV3Error, some of required attributes are not set " + attributes;
    }
  }
}
