package net.adoptopenjdk.v3.api;

import java.math.BigInteger;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * A description of a binary distribution.
 */
@SuppressWarnings({"all"})
public final class AOV3Binary implements AOV3BinaryType {
  private final AOV3Architecture architecture;
  private final BigInteger downloadCount;
  private final AOV3HeapSize heapSize;
  private final AOV3ImageKind imageType;
  private final AOV3Installer installer;
  private final AOV3JVMImplementation jvmImplementation;
  private final AOV3OperatingSystem operatingSystem;
  private final AOV3PackageType package_;
  private final String project;
  private final String scmReference;
  private final OffsetDateTime updatedAt;

  private AOV3Binary(
      AOV3Architecture architecture,
      BigInteger downloadCount,
      AOV3HeapSize heapSize,
      AOV3ImageKind imageType,
      AOV3Installer installer,
      AOV3JVMImplementation jvmImplementation,
      AOV3OperatingSystem operatingSystem,
      AOV3PackageType package_,
      String project,
      String scmReference,
      OffsetDateTime updatedAt) {
    this.architecture = architecture;
    this.downloadCount = downloadCount;
    this.heapSize = heapSize;
    this.imageType = imageType;
    this.installer = installer;
    this.jvmImplementation = jvmImplementation;
    this.operatingSystem = operatingSystem;
    this.package_ = package_;
    this.project = project;
    this.scmReference = scmReference;
    this.updatedAt = updatedAt;
  }

  /**
   * @return The target architecture
   */
  @Override
  public AOV3Architecture architecture() {
    return architecture;
  }

  /**
   * @return The number of times the distribution has been downloaded
   */
  @Override
  public BigInteger downloadCount() {
    return downloadCount;
  }

  /**
   * @return The heap size configuration
   */
  @Override
  public AOV3HeapSize heapSize() {
    return heapSize;
  }

  /**
   * @return The type of runtime image
   */
  @Override
  public AOV3ImageKind imageType() {
    return imageType;
  }

  /**
   * @return The installer information
   */
  @Override
  public Optional<AOV3Installer> installer() {
    return Optional.ofNullable(installer);
  }

  /**
   * @return The underlying JVM implementation
   */
  @Override
  public AOV3JVMImplementation jvmImplementation() {
    return jvmImplementation;
  }

  /**
   * @return The target operating system
   */
  @Override
  public AOV3OperatingSystem operatingSystem() {
    return operatingSystem;
  }

  /**
   * @return The package information
   */
  @Override
  public AOV3PackageType package_() {
    return package_;
  }

  /**
   * @return The project associated with the binary (such as "jdk")
   */
  @Override
  public String project() {
    return project;
  }

  /**
   * @return The SCM commit for the build
   */
  @Override
  public Optional<String> scmReference() {
    return Optional.ofNullable(scmReference);
  }

  /**
   * @return The time the distribution was last updated
   */
  @Override
  public OffsetDateTime updatedAt() {
    return updatedAt;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3BinaryType#architecture() architecture} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for architecture
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Binary withArchitecture(AOV3Architecture value) {
    if (this.architecture == value) return this;
    AOV3Architecture newValue = Objects.requireNonNull(value, "architecture");
    if (this.architecture.equals(newValue)) return this;
    return new AOV3Binary(
        newValue,
        this.downloadCount,
        this.heapSize,
        this.imageType,
        this.installer,
        this.jvmImplementation,
        this.operatingSystem,
        this.package_,
        this.project,
        this.scmReference,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3BinaryType#downloadCount() downloadCount} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for downloadCount
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Binary withDownloadCount(BigInteger value) {
    BigInteger newValue = Objects.requireNonNull(value, "downloadCount");
    if (this.downloadCount.equals(newValue)) return this;
    return new AOV3Binary(
        this.architecture,
        newValue,
        this.heapSize,
        this.imageType,
        this.installer,
        this.jvmImplementation,
        this.operatingSystem,
        this.package_,
        this.project,
        this.scmReference,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3BinaryType#heapSize() heapSize} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for heapSize
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Binary withHeapSize(AOV3HeapSize value) {
    if (this.heapSize == value) return this;
    AOV3HeapSize newValue = Objects.requireNonNull(value, "heapSize");
    if (this.heapSize.equals(newValue)) return this;
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        newValue,
        this.imageType,
        this.installer,
        this.jvmImplementation,
        this.operatingSystem,
        this.package_,
        this.project,
        this.scmReference,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3BinaryType#imageType() imageType} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for imageType
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Binary withImageType(AOV3ImageKind value) {
    if (this.imageType == value) return this;
    AOV3ImageKind newValue = Objects.requireNonNull(value, "imageType");
    if (this.imageType.equals(newValue)) return this;
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        this.heapSize,
        newValue,
        this.installer,
        this.jvmImplementation,
        this.operatingSystem,
        this.package_,
        this.project,
        this.scmReference,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link AOV3BinaryType#installer() installer} attribute.
   * @param value The value for installer
   * @return A modified copy of {@code this} object
   */
  public final AOV3Binary withInstaller(AOV3Installer value) {
    AOV3Installer newValue = Objects.requireNonNull(value, "installer");
    if (this.installer == newValue) return this;
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        this.heapSize,
        this.imageType,
        newValue,
        this.jvmImplementation,
        this.operatingSystem,
        this.package_,
        this.project,
        this.scmReference,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link AOV3BinaryType#installer() installer} attribute.
   * A shallow reference equality check is used on unboxed optional value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for installer
   * @return A modified copy of {@code this} object
   */
  @SuppressWarnings("unchecked") // safe covariant cast
  public final AOV3Binary withInstaller(Optional<? extends AOV3Installer> optional) {
    AOV3Installer value = optional.orElse(null);
    if (this.installer == value) return this;
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        this.heapSize,
        this.imageType,
        value,
        this.jvmImplementation,
        this.operatingSystem,
        this.package_,
        this.project,
        this.scmReference,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3BinaryType#jvmImplementation() jvmImplementation} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for jvmImplementation
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Binary withJvmImplementation(AOV3JVMImplementation value) {
    if (this.jvmImplementation == value) return this;
    AOV3JVMImplementation newValue = Objects.requireNonNull(value, "jvmImplementation");
    if (this.jvmImplementation.equals(newValue)) return this;
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        this.heapSize,
        this.imageType,
        this.installer,
        newValue,
        this.operatingSystem,
        this.package_,
        this.project,
        this.scmReference,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3BinaryType#operatingSystem() operatingSystem} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for operatingSystem
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Binary withOperatingSystem(AOV3OperatingSystem value) {
    if (this.operatingSystem == value) return this;
    AOV3OperatingSystem newValue = Objects.requireNonNull(value, "operatingSystem");
    if (this.operatingSystem.equals(newValue)) return this;
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        this.heapSize,
        this.imageType,
        this.installer,
        this.jvmImplementation,
        newValue,
        this.package_,
        this.project,
        this.scmReference,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3BinaryType#package_() package_} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for package_
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Binary withPackage_(AOV3PackageType value) {
    if (this.package_ == value) return this;
    AOV3PackageType newValue = Objects.requireNonNull(value, "package_");
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        this.heapSize,
        this.imageType,
        this.installer,
        this.jvmImplementation,
        this.operatingSystem,
        newValue,
        this.project,
        this.scmReference,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3BinaryType#project() project} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for project
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Binary withProject(String value) {
    String newValue = Objects.requireNonNull(value, "project");
    if (this.project.equals(newValue)) return this;
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        this.heapSize,
        this.imageType,
        this.installer,
        this.jvmImplementation,
        this.operatingSystem,
        this.package_,
        newValue,
        this.scmReference,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link AOV3BinaryType#scmReference() scmReference} attribute.
   * @param value The value for scmReference
   * @return A modified copy of {@code this} object
   */
  public final AOV3Binary withScmReference(String value) {
    String newValue = Objects.requireNonNull(value, "scmReference");
    if (Objects.equals(this.scmReference, newValue)) return this;
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        this.heapSize,
        this.imageType,
        this.installer,
        this.jvmImplementation,
        this.operatingSystem,
        this.package_,
        this.project,
        newValue,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link AOV3BinaryType#scmReference() scmReference} attribute.
   * An equality check is used on inner nullable value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for scmReference
   * @return A modified copy of {@code this} object
   */
  public final AOV3Binary withScmReference(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.scmReference, value)) return this;
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        this.heapSize,
        this.imageType,
        this.installer,
        this.jvmImplementation,
        this.operatingSystem,
        this.package_,
        this.project,
        value,
        this.updatedAt);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AOV3BinaryType#updatedAt() updatedAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for updatedAt
   * @return A modified copy of the {@code this} object
   */
  public final AOV3Binary withUpdatedAt(OffsetDateTime value) {
    if (this.updatedAt == value) return this;
    OffsetDateTime newValue = Objects.requireNonNull(value, "updatedAt");
    return new AOV3Binary(
        this.architecture,
        this.downloadCount,
        this.heapSize,
        this.imageType,
        this.installer,
        this.jvmImplementation,
        this.operatingSystem,
        this.package_,
        this.project,
        this.scmReference,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code AOV3Binary} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof AOV3Binary
        && equalTo((AOV3Binary) another);
  }

  private boolean equalTo(AOV3Binary another) {
    return architecture.equals(another.architecture)
        && downloadCount.equals(another.downloadCount)
        && heapSize.equals(another.heapSize)
        && imageType.equals(another.imageType)
        && Objects.equals(installer, another.installer)
        && jvmImplementation.equals(another.jvmImplementation)
        && operatingSystem.equals(another.operatingSystem)
        && package_.equals(another.package_)
        && project.equals(another.project)
        && Objects.equals(scmReference, another.scmReference)
        && updatedAt.equals(another.updatedAt);
  }

  /**
   * Computes a hash code from attributes: {@code architecture}, {@code downloadCount}, {@code heapSize}, {@code imageType}, {@code installer}, {@code jvmImplementation}, {@code operatingSystem}, {@code package_}, {@code project}, {@code scmReference}, {@code updatedAt}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + architecture.hashCode();
    h += (h << 5) + downloadCount.hashCode();
    h += (h << 5) + heapSize.hashCode();
    h += (h << 5) + imageType.hashCode();
    h += (h << 5) + Objects.hashCode(installer);
    h += (h << 5) + jvmImplementation.hashCode();
    h += (h << 5) + operatingSystem.hashCode();
    h += (h << 5) + package_.hashCode();
    h += (h << 5) + project.hashCode();
    h += (h << 5) + Objects.hashCode(scmReference);
    h += (h << 5) + updatedAt.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code AOV3Binary} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("AOV3Binary{");
    builder.append("architecture=").append(architecture);
    builder.append(", ");
    builder.append("downloadCount=").append(downloadCount);
    builder.append(", ");
    builder.append("heapSize=").append(heapSize);
    builder.append(", ");
    builder.append("imageType=").append(imageType);
    if (installer != null) {
      builder.append(", ");
      builder.append("installer=").append(installer);
    }
    builder.append(", ");
    builder.append("jvmImplementation=").append(jvmImplementation);
    builder.append(", ");
    builder.append("operatingSystem=").append(operatingSystem);
    builder.append(", ");
    builder.append("package_=").append(package_);
    builder.append(", ");
    builder.append("project=").append(project);
    if (scmReference != null) {
      builder.append(", ");
      builder.append("scmReference=").append(scmReference);
    }
    builder.append(", ");
    builder.append("updatedAt=").append(updatedAt);
    return builder.append("}").toString();
  }

  /**
   * Creates an immutable copy of a {@link AOV3BinaryType} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable AOV3Binary instance
   */
  public static AOV3Binary copyOf(AOV3BinaryType instance) {
    if (instance instanceof AOV3Binary) {
      return (AOV3Binary) instance;
    }
    return AOV3Binary.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link AOV3Binary AOV3Binary}.
   * <pre>
   * AOV3Binary.builder()
   *    .setArchitecture(net.adoptopenjdk.v3.api.AOV3Architecture) // required {@link AOV3BinaryType#architecture() architecture}
   *    .setDownloadCount(java.math.BigInteger) // required {@link AOV3BinaryType#downloadCount() downloadCount}
   *    .setHeapSize(net.adoptopenjdk.v3.api.AOV3HeapSize) // required {@link AOV3BinaryType#heapSize() heapSize}
   *    .setImageType(net.adoptopenjdk.v3.api.AOV3ImageKind) // required {@link AOV3BinaryType#imageType() imageType}
   *    .setInstaller(AOV3Installer) // optional {@link AOV3BinaryType#installer() installer}
   *    .setJvmImplementation(net.adoptopenjdk.v3.api.AOV3JVMImplementation) // required {@link AOV3BinaryType#jvmImplementation() jvmImplementation}
   *    .setOperatingSystem(net.adoptopenjdk.v3.api.AOV3OperatingSystem) // required {@link AOV3BinaryType#operatingSystem() operatingSystem}
   *    .setPackage_(net.adoptopenjdk.v3.api.AOV3PackageType) // required {@link AOV3BinaryType#package_() package_}
   *    .setProject(String) // required {@link AOV3BinaryType#project() project}
   *    .setScmReference(String) // optional {@link AOV3BinaryType#scmReference() scmReference}
   *    .setUpdatedAt(java.time.OffsetDateTime) // required {@link AOV3BinaryType#updatedAt() updatedAt}
   *    .build();
   * </pre>
   * @return A new AOV3Binary builder
   */
  public static AOV3Binary.Builder builder() {
    return new AOV3Binary.Builder();
  }

  /**
   * Builds instances of type {@link AOV3Binary AOV3Binary}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_ARCHITECTURE = 0x1L;
    private static final long INIT_BIT_DOWNLOAD_COUNT = 0x2L;
    private static final long INIT_BIT_HEAP_SIZE = 0x4L;
    private static final long INIT_BIT_IMAGE_TYPE = 0x8L;
    private static final long INIT_BIT_JVM_IMPLEMENTATION = 0x10L;
    private static final long INIT_BIT_OPERATING_SYSTEM = 0x20L;
    private static final long INIT_BIT_PACKAGE_ = 0x40L;
    private static final long INIT_BIT_PROJECT = 0x80L;
    private static final long INIT_BIT_UPDATED_AT = 0x100L;
    private long initBits = 0x1ffL;

    private AOV3Architecture architecture;
    private BigInteger downloadCount;
    private AOV3HeapSize heapSize;
    private AOV3ImageKind imageType;
    private AOV3Installer installer;
    private AOV3JVMImplementation jvmImplementation;
    private AOV3OperatingSystem operatingSystem;
    private AOV3PackageType package_;
    private String project;
    private String scmReference;
    private OffsetDateTime updatedAt;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code AOV3BinaryType} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(AOV3BinaryType instance) {
      Objects.requireNonNull(instance, "instance");
      setArchitecture(instance.architecture());
      setDownloadCount(instance.downloadCount());
      setHeapSize(instance.heapSize());
      setImageType(instance.imageType());
      Optional<AOV3Installer> installerOptional = instance.installer();
      if (installerOptional.isPresent()) {
        setInstaller(installerOptional);
      }
      setJvmImplementation(instance.jvmImplementation());
      setOperatingSystem(instance.operatingSystem());
      setPackage_(instance.package_());
      setProject(instance.project());
      Optional<String> scmReferenceOptional = instance.scmReference();
      if (scmReferenceOptional.isPresent()) {
        setScmReference(scmReferenceOptional);
      }
      setUpdatedAt(instance.updatedAt());
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3BinaryType#architecture() architecture} attribute.
     * @param architecture The value for architecture 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setArchitecture(AOV3Architecture architecture) {
      this.architecture = Objects.requireNonNull(architecture, "architecture");
      initBits &= ~INIT_BIT_ARCHITECTURE;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3BinaryType#downloadCount() downloadCount} attribute.
     * @param downloadCount The value for downloadCount 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setDownloadCount(BigInteger downloadCount) {
      this.downloadCount = Objects.requireNonNull(downloadCount, "downloadCount");
      initBits &= ~INIT_BIT_DOWNLOAD_COUNT;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3BinaryType#heapSize() heapSize} attribute.
     * @param heapSize The value for heapSize 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setHeapSize(AOV3HeapSize heapSize) {
      this.heapSize = Objects.requireNonNull(heapSize, "heapSize");
      initBits &= ~INIT_BIT_HEAP_SIZE;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3BinaryType#imageType() imageType} attribute.
     * @param imageType The value for imageType 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setImageType(AOV3ImageKind imageType) {
      this.imageType = Objects.requireNonNull(imageType, "imageType");
      initBits &= ~INIT_BIT_IMAGE_TYPE;
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3BinaryType#installer() installer} to installer.
     * @param installer The value for installer
     * @return {@code this} builder for chained invocation
     */
    public final Builder setInstaller(AOV3Installer installer) {
      this.installer = Objects.requireNonNull(installer, "installer");
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3BinaryType#installer() installer} to installer.
     * @param installer The value for installer
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setInstaller(Optional<? extends AOV3Installer> installer) {
      this.installer = installer.orElse(null);
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3BinaryType#jvmImplementation() jvmImplementation} attribute.
     * @param jvmImplementation The value for jvmImplementation 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setJvmImplementation(AOV3JVMImplementation jvmImplementation) {
      this.jvmImplementation = Objects.requireNonNull(jvmImplementation, "jvmImplementation");
      initBits &= ~INIT_BIT_JVM_IMPLEMENTATION;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3BinaryType#operatingSystem() operatingSystem} attribute.
     * @param operatingSystem The value for operatingSystem 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setOperatingSystem(AOV3OperatingSystem operatingSystem) {
      this.operatingSystem = Objects.requireNonNull(operatingSystem, "operatingSystem");
      initBits &= ~INIT_BIT_OPERATING_SYSTEM;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3BinaryType#package_() package_} attribute.
     * @param package_ The value for package_ 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setPackage_(AOV3PackageType package_) {
      this.package_ = Objects.requireNonNull(package_, "package_");
      initBits &= ~INIT_BIT_PACKAGE_;
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3BinaryType#project() project} attribute.
     * @param project The value for project 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setProject(String project) {
      this.project = Objects.requireNonNull(project, "project");
      initBits &= ~INIT_BIT_PROJECT;
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3BinaryType#scmReference() scmReference} to scmReference.
     * @param scmReference The value for scmReference
     * @return {@code this} builder for chained invocation
     */
    public final Builder setScmReference(String scmReference) {
      this.scmReference = Objects.requireNonNull(scmReference, "scmReference");
      return this;
    }

    /**
     * Initializes the optional value {@link AOV3BinaryType#scmReference() scmReference} to scmReference.
     * @param scmReference The value for scmReference
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setScmReference(Optional<String> scmReference) {
      this.scmReference = scmReference.orElse(null);
      return this;
    }

    /**
     * Initializes the value for the {@link AOV3BinaryType#updatedAt() updatedAt} attribute.
     * @param updatedAt The value for updatedAt 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setUpdatedAt(OffsetDateTime updatedAt) {
      this.updatedAt = Objects.requireNonNull(updatedAt, "updatedAt");
      initBits &= ~INIT_BIT_UPDATED_AT;
      return this;
    }

    /**
     * Builds a new {@link AOV3Binary AOV3Binary}.
     * @return An immutable instance of AOV3Binary
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public AOV3Binary build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new AOV3Binary(
          architecture,
          downloadCount,
          heapSize,
          imageType,
          installer,
          jvmImplementation,
          operatingSystem,
          package_,
          project,
          scmReference,
          updatedAt);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ARCHITECTURE) != 0) attributes.add("architecture");
      if ((initBits & INIT_BIT_DOWNLOAD_COUNT) != 0) attributes.add("downloadCount");
      if ((initBits & INIT_BIT_HEAP_SIZE) != 0) attributes.add("heapSize");
      if ((initBits & INIT_BIT_IMAGE_TYPE) != 0) attributes.add("imageType");
      if ((initBits & INIT_BIT_JVM_IMPLEMENTATION) != 0) attributes.add("jvmImplementation");
      if ((initBits & INIT_BIT_OPERATING_SYSTEM) != 0) attributes.add("operatingSystem");
      if ((initBits & INIT_BIT_PACKAGE_) != 0) attributes.add("package_");
      if ((initBits & INIT_BIT_PROJECT) != 0) attributes.add("project");
      if ((initBits & INIT_BIT_UPDATED_AT) != 0) attributes.add("updatedAt");
      return "Cannot build AOV3Binary, some of required attributes are not set " + attributes;
    }
  }
}
